<?php
// Initialize session if not already started
session_start();

// Define JSON file paths
define('MEDIA_PARTNERS_FILE', 'data/media_partners.json');
define('ORGANIZERS_FILE', 'data/organizers.json');
define('EVENTS_FILE', 'data/events.json');
define('EVENT_PLANS_FILE', 'data/event_plans.json');
define('PROMOTIONS_FILE', 'data/promotions.json');
define('PINS_FILE', 'data/pins.json');

// Authentication settings
define('ACCESS_PIN', '051205');
define('COOKIE_NAME', 'event_sinergi_auth');
define('COOKIE_DURATION', 30 * 24 * 60 * 60); // 30 days

// Check if user is logged in
function isLoggedIn() {
    // Check session
    if (isset($_SESSION['authenticated']) && $_SESSION['authenticated'] === true) {
        return true;
    }
    
    // Check remember me cookie
    if (isset($_COOKIE[COOKIE_NAME])) {
        // Validate the cookie value
        if ($_COOKIE[COOKIE_NAME] === hash('sha256', ACCESS_PIN)) {
            $_SESSION['authenticated'] = true;
            return true;
        }
    }
    
    return false;
}

// Handle login
if (isset($_POST['action']) && $_POST['action'] === 'login') {
    if ($_POST['pin'] === ACCESS_PIN) {
        $_SESSION['authenticated'] = true;
        
        // Set remember me cookie if requested
        if (isset($_POST['remember']) && $_POST['remember'] === 'on') {
            setcookie(COOKIE_NAME, hash('sha256', ACCESS_PIN), time() + COOKIE_DURATION, '/');
        }
        
        // Redirect to the dashboard to avoid form resubmission
        header('Location: index.php');
        exit;
    } else {
        $loginError = 'PIN yang Anda masukkan salah.';
    }
}

// Handle logout
if (isset($_GET['logout'])) {
    // Clear session
    $_SESSION = array();
    session_destroy();
    
    // Clear cookie
    setcookie(COOKIE_NAME, '', time() - 3600, '/');
    
    // Redirect to login page
    header('Location: index.php');
    exit;
}

// Create data directory if it doesn't exist
if (!file_exists('data')) {
    mkdir('data', 0777, true);
}

// Initialize JSON files if they don't exist
function initializeJsonFile($file, $defaultData = []) {
    if (!file_exists($file)) {
        file_put_contents($file, json_encode($defaultData, JSON_PRETTY_PRINT));
    }
}

initializeJsonFile(MEDIA_PARTNERS_FILE);
initializeJsonFile(ORGANIZERS_FILE);
initializeJsonFile(EVENTS_FILE);
initializeJsonFile(EVENT_PLANS_FILE);
initializeJsonFile(PROMOTIONS_FILE);
initializeJsonFile(PINS_FILE);

// Helper functions
function loadData($file) {
    return json_decode(file_get_contents($file), true) ?: [];
}

function saveData($file, $data) {
    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT));
}

function generateId() {
    return uniqid();
}

function generateShareableLink() {
    return bin2hex(random_bytes(8)); // 16 character random string
}

// Get current page
$page = isset($_GET['page']) ? $_GET['page'] : 'dashboard';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isLoggedIn()) {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'add_media_partner':
                $mediaPartners = loadData(MEDIA_PARTNERS_FILE);
                $newPartner = [
                    'id' => generateId(),
                    'name' => $_POST['name'],
                    'instagram' => $_POST['instagram'],
                    'whatsapp' => $_POST['whatsapp']
                ];
                $mediaPartners[] = $newPartner;
                saveData(MEDIA_PARTNERS_FILE, $mediaPartners);
                header('Location: index.php?page=media_partners');
                exit;
                
            case 'add_organizer':
                $organizers = loadData(ORGANIZERS_FILE);
                $newOrganizer = [
                    'id' => generateId(),
                    'name' => $_POST['name'],
                    'contact' => $_POST['contact']
                ];
                $organizers[] = $newOrganizer;
                saveData(ORGANIZERS_FILE, $organizers);
                header('Location: index.php?page=organizers');
                exit;
            
            case 'add_event_plan':
                $eventPlans = loadData(EVENT_PLANS_FILE);
                $newEventPlan = [
                    'id' => generateId(),
                    'name' => $_POST['name'],
                    'organizer_id' => $_POST['organizer_id'],
                    'registration_start' => $_POST['registration_start'],
                    'registration_end' => $_POST['registration_end'],
                    'event_start' => $_POST['event_start'],
                    'event_end' => $_POST['event_end'],
                    'award_access' => $_POST['award_access'],
                    'cleanup_date' => $_POST['cleanup_date'],
                    'created_at' => date('Y-m-d H:i:s'),
                    'converted_to_event' => false
                ];
                $eventPlans[] = $newEventPlan;
                saveData(EVENT_PLANS_FILE, $eventPlans);
                header('Location: index.php?page=event_plans');
                exit;
                
            case 'convert_to_event':
                // Load data
                $eventPlans = loadData(EVENT_PLANS_FILE);
                $events = loadData(EVENTS_FILE);
                
                // Find the event plan
                $planId = $_POST['plan_id'];
                $planIndex = -1;
                $plan = null;
                
                foreach ($eventPlans as $index => $eventPlan) {
                    if ($eventPlan['id'] === $planId) {
                        $plan = $eventPlan;
                        $planIndex = $index;
                        break;
                    }
                }
                
                if ($plan) {
                    // Mark plan as converted
                    $eventPlans[$planIndex]['converted_to_event'] = true;
                    saveData(EVENT_PLANS_FILE, $eventPlans);
                    
                    // Create new event
                    $images = [];
                    // Handle multiple image links
                    if (isset($_POST['image_links']) && is_array($_POST['image_links'])) {
                        foreach ($_POST['image_links'] as $link) {
                            if (!empty($link)) {
                                $images[] = $link;
                            }
                        }
                    }
                    
                    $shareableLink = generateShareableLink();
                    
                    $newEvent = [
                        'id' => generateId(),
                        'plan_id' => $planId,
                        'name' => $plan['name'],
                        'organizer_id' => $plan['organizer_id'],
                        'description' => $_POST['description'],
                        'images' => $images,
                        'registration_start' => $plan['registration_start'],
                        'registration_end' => $plan['registration_end'],
                        'event_start' => $plan['event_start'],
                        'event_end' => $plan['event_end'],
                        'award_access' => $plan['award_access'],
                        'cleanup_date' => $plan['cleanup_date'],
                        'shareable_link' => $shareableLink,
                        'created_at' => date('Y-m-d H:i:s')
                    ];
                    
                    $events[] = $newEvent;
                    saveData(EVENTS_FILE, $events);
                    
                    header('Location: index.php?page=events');
                    exit;
                }
                break;
                
            case 'add_promotion':
                $promotions = loadData(PROMOTIONS_FILE);
                $postDates = $_POST['post_dates'];
                
                foreach ($postDates as $postDate) {
                    if (!empty($postDate)) {
                        $newPromotion = [
                            'id' => generateId(),
                            'event_id' => $_POST['event_id'],
                            'media_partner_id' => $_POST['media_partner_id'],
                            'post_date' => $postDate,
                            'cost' => $_POST['cost'],
                            'completed' => false
                        ];
                        $promotions[] = $newPromotion;
                    }
                }
                
                saveData(PROMOTIONS_FILE, $promotions);
                header('Location: index.php');
                exit;
                
            case 'add_pin':
                $pins = loadData(PINS_FILE);
                $newPin = [
                    'id' => generateId(),
                    'event_id' => $_POST['event_id'],
                    'media_partner_id' => $_POST['media_partner_id'],
                    'start_date' => $_POST['start_date'],
                    'end_date' => $_POST['end_date'],
                    'completed' => false
                ];
                $pins[] = $newPin;
                saveData(PINS_FILE, $pins);
                header('Location: index.php?page=pins');
                exit;
                
            case 'mark_completed':
                $promotions = loadData(PROMOTIONS_FILE);
                foreach ($promotions as &$promotion) {
                    if ($promotion['id'] === $_POST['promotion_id']) {
                        $promotion['completed'] = true;
                        break;
                    }
                }
                saveData(PROMOTIONS_FILE, $promotions);
                header('Location: index.php');
                exit;
                
            case 'mark_pin_completed':
                $pins = loadData(PINS_FILE);
                foreach ($pins as &$pin) {
                    if ($pin['id'] === $_POST['pin_id']) {
                        $pin['completed'] = true;
                        break;
                    }
                }
                saveData(PINS_FILE, $pins);
                header('Location: index.php?page=pins');
                exit;
                
            case 'regenerate_link':
                $events = loadData(EVENTS_FILE);
                $eventId = $_POST['event_id'];
                
                foreach ($events as &$event) {
                    if ($event['id'] === $eventId) {
                        $event['shareable_link'] = generateShareableLink();
                        break;
                    }
                }
                
                saveData(EVENTS_FILE, $events);
                header('Location: index.php?page=events');
                exit;
        }
    }
}

// Helper function to get entity by ID
function getEntityById($file, $id) {
    $data = loadData($file);
    foreach ($data as $entity) {
        if ($entity['id'] === $id) {
            return $entity;
        }
    }
    return null;
}

// Get current and upcoming promotions
function getCurrentPromotions() {
    $promotions = loadData(PROMOTIONS_FILE);
    $result = [];
    
    foreach ($promotions as $promotion) {
        $postDate = strtotime($promotion['post_date']);
        $now = time();
        
        // Show promotions that are within the next 30 days or past but not completed
        if ($postDate <= $now + (30 * 24 * 60 * 60)) {
            $result[] = $promotion;
        }
    }
    
    // Sort by post date
    usort($result, function($a, $b) {
        return strtotime($a['post_date']) - strtotime($b['post_date']);
    });
    
    return $result;
}

// Get promotions for a specific event
function getEventPromotions($eventId) {
    $promotions = loadData(PROMOTIONS_FILE);
    $result = [];
    
    foreach ($promotions as $promotion) {
        if ($promotion['event_id'] === $eventId) {
            $result[] = $promotion;
        }
    }
    
    // Sort by post date
    usort($result, function($a, $b) {
        return strtotime($a['post_date']) - strtotime($b['post_date']);
    });
    
    return $result;
}

// Get pins for a specific event
function getEventPins($eventId) {
    $pins = loadData(PINS_FILE);
    $result = [];
    
    foreach ($pins as $pin) {
        if ($pin['event_id'] === $eventId) {
            $result[] = $pin;
        }
    }
    
    // Sort by start date
    usort($result, function($a, $b) {
        return strtotime($a['start_date']) - strtotime($b['start_date']);
    });
    
    return $result;
}

// Get current and upcoming pins
function getCurrentPins() {
    $pins = loadData(PINS_FILE);
    $result = [];
    
    foreach ($pins as $pin) {
        $endDate = strtotime($pin['end_date']);
        $now = time();
        
        // Show pins that are not yet ended or not completed
        if ($endDate >= $now || !$pin['completed']) {
            $result[] = $pin;
        }
    }
    
    // Sort by end date
    usort($result, function($a, $b) {
        return strtotime($a['end_date']) - strtotime($b['end_date']);
    });
    
    return $result;
}

// Get event by shareable link
function getEventByShareableLink($link) {
    $events = loadData(EVENTS_FILE);
    
    foreach ($events as $event) {
        if (isset($event['shareable_link']) && $event['shareable_link'] === $link) {
            return $event;
        }
    }
    
    return null;
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="robots" content="noindex, nofollow">
    <link rel="icon" type="image/png" href="https://app.dezki.biz.id/linkpic/assets/uploads/680379d79bff3.png" />
    <title>Event-Sinergi - Sistem Pengelolaan Event Media Partner</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css">
    <style>
        :root {
            --primary: #e60000;
            --secondary: #000000;
            --light: #ffffff;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #f8f9fa;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
        }
        
        .navbar {
            background-color: var(--primary);
        }
        
        .navbar-brand {
            color: var(--light) !important;
            font-weight: bold;
        }
        
        .nav-link {
            color: var(--light) !important;
        }
        
        .nav-link:hover {
            color: rgba(255, 255, 255, 0.8) !important;
        }
        
        .btn-primary {
            background-color: var(--primary);
            border-color: var(--primary);
        }
        
        .btn-primary:hover {
            background-color: #b80000;
            border-color: #b80000;
        }
        
        .btn-secondary {
            background-color: var(--secondary);
            border-color: var(--secondary);
        }
        
        .btn-secondary:hover {
            background-color: #333333;
            border-color: #333333;
        }
        
        .card {
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            margin-bottom: 20px;
        }
        
        .card-header {
            background-color: var(--primary);
            color: var(--light);
            font-weight: bold;
        }
        
        .promotion-card {
            border-left: 5px solid var(--primary);
        }
        
        .completed {
            background-color: #e8f5e9;
            border-left: 5px solid #4caf50;
        }
        
        .overdue {
            background-color: #ffebee;
            border-left: 5px solid #f44336;
        }
        
        .upcoming {
            background-color: #e3f2fd;
            border-left: 5px solid #2196f3;
        }
        
        .today {
            background-color: #fff8e1;
            border-left: 5px solid #ffc107;
        }
        
        .login-form {
            max-width: 400px;
            margin: 100px auto;
        }
        
        .main-content {
            flex: 1;
        }
        
        footer {
            margin-top: auto;
            background-color: var(--secondary);
            color: var(--light);
            text-align: center;
            padding: 1rem 0;
            width: 100%;
        }
        
        .event-poster {
            max-width: 100%;
            height: auto;
            margin-bottom: 1rem;
            border-radius: 8px;
        }
        
        .timeline {
            position: relative;
            padding-left: 2rem;
            margin-bottom: 2rem;
        }
        
        .timeline::before {
            content: '';
            position: absolute;
            left: 0;
            top: 0;
            height: 100%;
            width: 2px;
            background-color: var(--primary);
        }
        
        .timeline-item {
            position: relative;
            padding-bottom: 1.5rem;
        }
        
        .timeline-item::before {
            content: '';
            position: absolute;
            left: -2rem;
            top: 0;
            width: 1rem;
            height: 1rem;
            border-radius: 50%;
            background-color: var(--primary);
        }
        
        .badge-outline {
            border: 1px solid currentColor;
            background-color: transparent;
        }
        
        .share-link-input {
            background-color: #f0f0f0;
            cursor: text;
        }
        
        /* Responsive adjustments */
        @media (max-width: 768px) {
            .container {
                padding: 10px;
            }
            
            .card {
                margin-bottom: 15px;
            }
            
            .btn-sm {
                padding: 0.25rem 0.5rem;
                font-size: 0.75rem;
            }
            
            .login-form {
                margin: 50px auto;
                padding: 0 15px;
            }
        }
    </style>
</head>
<body>
    <?php 
    // Check if viewing a shared event
    if (isset($_GET['share']) && !empty($_GET['share'])) {
        $sharedEvent = getEventByShareableLink($_GET['share']);
        if ($sharedEvent) {
            include 'shared_event.php';
            exit;
        }
    }
    
    // Regular application flow
    if (!isLoggedIn()): 
    ?>
        <!-- Login Form -->
        <div class="container">
            <div class="login-form">
                <div class="card">
                    <div class="card-header">
                        <h4 class="mb-0">Masuk ke Sistem</h4>
                    </div>
                    <div class="card-body">
                        <?php if (isset($loginError)): ?>
                            <div class="alert alert-danger"><?= $loginError ?></div>
                        <?php endif; ?>
                        
                        <form method="post">
                            <input type="hidden" name="action" value="login">
                            <div class="mb-3">
                                <label for="pin" class="form-label">PIN Akses</label>
                                <input type="password" class="form-control" id="pin" name="pin" required>
                            </div>
                            <div class="mb-3 form-check">
                                <input type="checkbox" class="form-check-input" id="remember" name="remember">
                                <label class="form-check-label" for="remember">Ingat Saya</label>
                            </div>
                            <button type="submit" class="btn btn-primary w-100">Masuk</button>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    <?php else: ?>
        <!-- Navigation -->
        <nav class="navbar navbar-expand-lg navbar-dark">
            <div class="container">
                <a class="navbar-brand" href="index.php">Event-Sinergi</a>
                <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                    <span class="navbar-toggler-icon"></span>
                </button>
                <div class="collapse navbar-collapse" id="navbarNav">
                    <ul class="navbar-nav ms-auto">
                        <li class="nav-item">
                            <a class="nav-link <?= $page === 'dashboard' ? 'active' : '' ?>" href="index.php">Dashboard</a>
                        </li>
                        <li class="nav-item dropdown">
                            <a class="nav-link dropdown-toggle <?= in_array($page, ['event_plans', 'events']) ? 'active' : '' ?>" href="#" role="button" data-bs-toggle="dropdown">
                                Event
                            </a>
                            <ul class="dropdown-menu">
                                <li><a class="dropdown-item <?= $page === 'event_plans' ? 'active' : '' ?>" href="index.php?page=event_plans">Rencana Event</a></li>
                                <li><a class="dropdown-item <?= $page === 'events' ? 'active' : '' ?>" href="index.php?page=events">Event</a></li>
                            </ul>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link <?= $page === 'media_partners' ? 'active' : '' ?>" href="index.php?page=media_partners">Media Partner</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link <?= $page === 'organizers' ? 'active' : '' ?>" href="index.php?page=organizers">Penyelenggara</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link <?= $page === 'add_promotion' ? 'active' : '' ?>" href="index.php?page=add_promotion">Tambah Promosi</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link <?= $page === 'pins' ? 'active' : '' ?>" href="index.php?page=pins">Pin Event</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="index.php?logout=1">Keluar</a>
                        </li>
                    </ul>
                </div>
            </div>
        </nav>

        <!-- Main Content -->
        <div class="container my-4 main-content">
            <?php 
            // Load different pages based on the current page value
            switch ($page) {
                case 'dashboard':
                    include 'pages/dashboard.php';
                    break;
                case 'media_partners':
                    include 'pages/media_partners.php';
                    break;
                case 'organizers':
                    include 'pages/organizers.php';
                    break;
                case 'event_plans':
                    include 'pages/event_plans.php';
                    break;
                case 'events':
                    include 'pages/events.php';
                    break;
                case 'add_promotion':
                    include 'pages/add_promotion.php';
                    break;
                case 'pins':
                    include 'pages/pins.php';
                    break;
                case 'convert_event_plan':
                    include 'pages/convert_event_plan.php';
                    break;
                default:
                    include 'pages/dashboard.php';
                    break;
            }
            ?>
        </div>
    <?php endif; ?>

    <!-- Footer -->
    <footer>
        <p class="mb-0">Event-Sinergi &copy; <?= date('Y') ?> - Sistem Pengelolaan Event Media Partner</p>
    </footer>

    <!-- JavaScript -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Handle adding multiple post dates
            const addDateButton = document.querySelector('.add-date');
            const postDatesContainer = document.getElementById('post-dates-container');
            
            if (addDateButton && postDatesContainer) {
                addDateButton.addEventListener('click', function() {
                    const dateRow = document.createElement('div');
                    dateRow.className = 'mb-2 row';
                    dateRow.innerHTML = `
                        <div class="col-md-6">
                            <input type="date" class="form-control" name="post_dates[]">
                        </div>
                        <div class="col-md-6">
                            <button type="button" class="btn btn-sm btn-outline-danger remove-date">Hapus</button>
                        </div>
                    `;
                    postDatesContainer.appendChild(dateRow);
                    
                    // Add event listener to the remove button
                    const removeButton = dateRow.querySelector('.remove-date');
                    removeButton.addEventListener('click', function() {
                        postDatesContainer.removeChild(dateRow);
                    });
                });
            }
            
            // Handle adding multiple image links
            const addImageButton = document.querySelector('.add-image');
            const imageLinksContainer = document.getElementById('image-links-container');
            
            if (addImageButton && imageLinksContainer) {
                addImageButton.addEventListener('click', function() {
                    const imageRow = document.createElement('div');
                    imageRow.className = 'mb-2 row';
                    imageRow.innerHTML = `
                        <div class="col-md-9">
                            <input type="url" class="form-control" name="image_links[]" placeholder="https://example.com/image.jpg">
                        </div>
                        <div class="col-md-3">
                            <button type="button" class="btn btn-sm btn-outline-danger remove-image">Hapus</button>
                        </div>
                    `;
                    imageLinksContainer.appendChild(imageRow);
                    
                    // Add event listener to the remove button
                    const removeButton = imageRow.querySelector('.remove-image');
                    removeButton.addEventListener('click', function() {
                        imageLinksContainer.removeChild(imageRow);
                    });
                });
            }
            
            // Copy to clipboard functionality
            const copyButtons = document.querySelectorAll('.copy-btn');
            
            if (copyButtons) {
                copyButtons.forEach(button => {
                    button.addEventListener('click', function() {
                        const textToCopy = this.getAttribute('data-copy');
                        const textArea = document.createElement('textarea');
                        textArea.value = textToCopy;
                        document.body.appendChild(textArea);
                        textArea.select();
                        document.execCommand('copy');
                        document.body.removeChild(textArea);
                        
                        // Change button text temporarily
                        const originalText = this.innerHTML;
                        this.innerHTML = '<i class="bi bi-check"></i> Disalin!';
                        setTimeout(() => {
                            this.innerHTML = originalText;
                        }, 2000);
                    });
                });
            }
        });
    </script>
</body>
</html>