/**
 * File JavaScript utama aplikasi Tugasin
 */

// DOM Functions
document.addEventListener('DOMContentLoaded', function() {
    // Inisialisasi tab aktif
    initTabs();
    
    // Inisialisasi modal events
    initModals();
    
    // Inisialisasi fitur tema
    initThemeSelection();
    
    // Inisialisasi FAB visibility
    initFAB();
    
    // Inisialisasi tombol back
    initBackButton();
    
    // Inisialisasi container notifikasi toast
    createToastContainer();
});

/**
 * Inisialisasi tab navigation
 */
function initTabs() {
    // Desktop Tabs
    document.querySelectorAll('.desktop-tab').forEach(tab => {
        tab.addEventListener('click', function() {
            const tabName = this.getAttribute('data-tab');
            switchTab(tabName, true);
        });
    });
    
    // Mobile Tabs
    document.querySelectorAll('.mobile-tab').forEach(tab => {
        tab.addEventListener('click', function() {
            const tabName = this.getAttribute('data-tab');
            switchTab(tabName, true);
        });
    });
}

/**
 * Berganti tab
 * 
 * @param {string} tabName - Nama tab tujuan
 * @param {boolean} updateUrl - Apakah URL perlu diupdate
 */
function switchTab(tabName, updateUrl = true) {
    // Update URL jika diperlukan
    if (updateUrl) {
        const baseUrl = window.location.href.split('?')[0];
        const newUrl = baseUrl + '?tab=' + tabName;
        
        // History API untuk update URL tanpa refresh
        history.pushState({tab: tabName}, '', newUrl);
    }
    
    // Show loading
    showPageLoading();
    
    // Fetch content via AJAX
    fetchTabContent(tabName)
        .then(function() {
            // Setelah content dimuat, update tab aktif
            // Desktop
            document.querySelectorAll('.desktop-tab').forEach(tab => {
                if (tab.getAttribute('data-tab') === tabName) {
                    tab.classList.add('active');
                } else {
                    tab.classList.remove('active');
                }
            });
            
            // Mobile
            document.querySelectorAll('.mobile-tab').forEach(tab => {
                if (tab.getAttribute('data-tab') === tabName) {
                    tab.classList.add('active');
                } else {
                    tab.classList.remove('active');
                }
            });
            
            // Update visibilitas FAB button
            updateFabVisibility(tabName);
            
            // Update stats display di mobile
            updateStatsVisibility(tabName);
            
            // Hide loading
            hidePageLoading();
        })
        .catch(function(error) {
            // Jika terjadi error
            console.error('Error switching tab:', error);
            showToast('Terjadi kesalahan saat memuat halaman', 'error');
            hidePageLoading();
        });
}

/**
 * Memuat konten tab via AJAX
 * 
 * @param {string} tabName - Nama tab yang akan dimuat
 * @returns {Promise} - Promise object
 */
function fetchTabContent(tabName) {
    return new Promise((resolve, reject) => {
        const xhr = new XMLHttpRequest();
        xhr.open('GET', tabName + '.php', true);
        
        xhr.onload = function() {
            if (this.status >= 200 && this.status < 300) {
                document.getElementById('page-content').innerHTML = xhr.responseText;
                // Reinisialisasi event handler untuk konten baru
                initTabEvents();
                resolve();
            } else {
                reject({
                    status: this.status,
                    statusText: xhr.statusText
                });
            }
        };
        
        xhr.onerror = function() {
            reject({
                status: this.status,
                statusText: xhr.statusText
            });
        };
        
        xhr.send();
    });
}

/**
 * Inisialisasi event handler untuk konten tab
 */
function initTabEvents() {
    // Inisialisasi tombol hapus dengan konfirmasi
    document.querySelectorAll('[data-delete]').forEach(btn => {
        btn.addEventListener('click', function(e) {
            e.preventDefault();
            const type = this.getAttribute('data-delete');
            const id = this.getAttribute('data-id');
            confirmDelete(type, id);
        });
    });
    
    // Inisialisasi chapter items di halaman baca
    document.querySelectorAll('.chapter-item').forEach(chapter => {
        chapter.addEventListener('click', function() {
            const bacaId = this.getAttribute('data-baca');
            const babNomor = this.getAttribute('data-bab');
            markChapterAsRead(bacaId, babNomor);
        });
    });
    
    // Inisialisasi input pencarian
    document.querySelectorAll('.search-box input').forEach(input => {
        input.addEventListener('keyup', function() {
            const target = this.getAttribute('data-target');
            searchItems(this.value, target);
        });
    });
}

/**
 * Inisialisasi modal events
 */
function initModals() {
    // Close modal when clicking outside
    window.onclick = function(event) {
        document.querySelectorAll('.modal').forEach(modal => {
            if (event.target === modal) {
                closeModal(modal.id);
            }
        });
    };
    
    // Close modals with escape key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            closeAllModals();
        }
    });
}

/**
 * Buka modal
 * 
 * @param {string} modalId - ID modal yang akan dibuka
 */
function openModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.style.display = 'block';
        
        // Focus first input in modal
        const firstInput = modal.querySelector('input, select, textarea');
        if (firstInput) {
            setTimeout(() => firstInput.focus(), 100);
        }
    }
}

/**
 * Tutup modal
 * 
 * @param {string} modalId - ID modal yang akan ditutup
 */
function closeModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.style.display = 'none';
    }
}

/**
 * Tutup semua modal
 */
function closeAllModals() {
    document.querySelectorAll('.modal').forEach(modal => {
        modal.style.display = 'none';
    });
}

/**
 * Inisialisasi pemilihan tema
 */
function initThemeSelection() {
    document.querySelectorAll('.theme-option').forEach(option => {
        option.addEventListener('click', function() {
            const theme = this.getAttribute('data-theme');
            document.querySelectorAll('.theme-option').forEach(opt => opt.classList.remove('active'));
            this.classList.add('active');
            document.getElementById('selected-theme').value = theme;
        });
    });
    
    // Custom theme color pickers
    document.querySelectorAll('.color-input-wrapper input[type="color"]').forEach(input => {
        input.addEventListener('input', function() {
            const targetInput = this.parentElement.querySelector('input[type="text"]');
            if (targetInput) {
                targetInput.value = this.value;
                updateCustomThemePreview();
            }
        });
    });
    
    document.querySelectorAll('.color-input-wrapper input[type="text"]').forEach(input => {
        input.addEventListener('input', function() {
            const colorInput = this.parentElement.querySelector('input[type="color"]');
            if (colorInput) {
                // Validate hex color
                if (/^#[0-9A-F]{6}$/i.test(this.value)) {
                    colorInput.value = this.value;
                    updateCustomThemePreview();
                }
            }
        });
    });
}

/**
 * Update preview tema kustom
 */
function updateCustomThemePreview() {
    const primaryColor = document.getElementById('custom-primary').value;
    const previewEl = document.getElementById('custom-theme-preview');
    
    if (previewEl && primaryColor) {
        previewEl.style.backgroundColor = primaryColor;
    }
}

/**
 * Inisialisasi Floating Action Button
 */
function initFAB() {
    const fab = document.getElementById('fab-button');
    if (fab) {
        fab.addEventListener('click', function() {
            const currentTab = getCurrentTab();
            switch (currentTab) {
                case 'diskusi':
                    openModal('add-diskusi-modal');
                    break;
                case 'tugas':
                    openModal('add-tugas-modal');
                    break;
                case 'baca':
                    openModal('add-baca-modal');
                    break;
            }
        });
        
        // Set initial visibility
        updateFabVisibility(getCurrentTab());
    }
}

/**
 * Update visibility FAB button berdasarkan tab aktif
 * 
 * @param {string} tabName - Nama tab aktif
 */
function updateFabVisibility(tabName) {
    const fab = document.getElementById('fab-button');
    if (fab) {
        // Tampilkan hanya di diskusi, tugas, dan baca
        if (tabName === 'diskusi' || tabName === 'tugas' || tabName === 'baca') {
            fab.style.display = 'flex';
        } else {
            fab.style.display = 'none';
        }
    }
}

/**
 * Update visibility statistik di mobile berdasarkan tab aktif
 * 
 * @param {string} tabName - Nama tab aktif
 */
function updateStatsVisibility(tabName) {
    const statsCard = document.querySelector('.stats-card');
    if (statsCard) {
        if (window.innerWidth <= 768) {
            // Di mobile, tampilkan stats hanya di tab statistik
            if (tabName === 'statistik') {
                statsCard.style.display = 'flex';
            } else {
                statsCard.style.display = 'none';
            }
        } else {
            // Di desktop, selalu tampilkan
            statsCard.style.display = 'flex';
        }
    }
}

/**
 * Inisialisasi tombol back
 */
function initBackButton() {
    // Handle browser back button
    window.addEventListener('popstate', function(e) {
        if (e.state && e.state.tab) {
            switchTab(e.state.tab, false);
        } else {
            // Default tab jika state tidak ada
            switchTab('diskusi', false);
        }
    });
}

/**
 * Cari items berdasarkan keyword
 * 
 * @param {string} query - Keyword pencarian
 * @param {string} containerId - ID container yang berisi item
 */
function searchItems(query, containerId) {
    const container = document.getElementById(containerId);
    if (!container) return;
    
    query = query.toLowerCase();
    const items = container.getElementsByClassName('card');
    
    for (let i = 0; i < items.length; i++) {
        const item = items[i];
        const title = item.querySelector('h3');
        
        if (title) {
            const text = title.textContent || title.innerText;
            
            if (text.toLowerCase().indexOf(query) > -1) {
                item.style.display = '';
            } else {
                item.style.display = 'none';
            }
        }
    }
}

/**
 * Konfirmasi hapus
 * 
 * @param {string} type - Tipe item (diskusi, tugas, baca)
 * @param {string} id - ID item
 */
function confirmDelete(type, id) {
    if (confirm('Apakah Anda yakin ingin menghapus item ini?')) {
        // Show loading
        showPageLoading();
        
        // Redirect ke action delete
        window.location.href = '?action=delete_' + type + '&id=' + id;
    }
}

/**
 * Tandai bab sebagai sudah dibaca
 * 
 * @param {string} bacaId - ID baca
 * @param {string} babNomor - Nomor bab
 */
function markChapterAsRead(bacaId, babNomor) {
    // Show loading
    showPageLoading();
    
    // Kirim form via AJAX
    const formData = new FormData();
    formData.append('action', 'complete_bab');
    formData.append('baca_id', bacaId);
    formData.append('bab_nomor', babNomor);
    
    fetch('ajax.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Update UI
            const chapterItem = document.querySelector(`.chapter-item[data-baca="${bacaId}"][data-bab="${babNomor}"]`);
            if (chapterItem) {
                chapterItem.classList.add('completed');
            }
            
            // Update progress bar jika ada
            const progressBar = document.querySelector(`.baca-card[data-id="${bacaId}"] .progress`);
            if (progressBar && data.progress) {
                progressBar.style.width = data.progress + '%';
            }
            
            // Update progress count
            const progressCount = document.querySelector(`.baca-card[data-id="${bacaId}"] .progress-count`);
            if (progressCount && data.selesai_count !== undefined && data.total_count !== undefined) {
                progressCount.textContent = `${data.selesai_count} dari ${data.total_count} bab`;
            }
            
            showToast('Bab ' + babNomor + ' ditandai sebagai selesai', 'success');
            
            // Jika sudah 100% selesai, refresh halaman
            if (data.is_complete) {
                showToast('Semua bab sudah selesai dibaca!', 'success');
                setTimeout(() => {
                    window.location.reload();
                }, 1500);
            }
        } else {
            showToast(data.message || 'Gagal menandai bab', 'error');
        }
        
        // Hide loading
        hidePageLoading();
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('Terjadi kesalahan saat memproses permintaan', 'error');
        hidePageLoading();
    });
}

/**
 * Dapatkan tab aktif saat ini
 * 
 * @returns {string} - Nama tab aktif
 */
function getCurrentTab() {
    const urlParams = new URLSearchParams(window.location.search);
    return urlParams.get('tab') || 'diskusi';
}

/**
 * Buat container untuk toast notifications
 */
function createToastContainer() {
    if (!document.querySelector('.toast-container')) {
        const toastContainer = document.createElement('div');
        toastContainer.className = 'toast-container';
        document.body.appendChild(toastContainer);
    }
}

/**
 * Tampilkan toast notification
 * 
 * @param {string} message - Pesan notifikasi
 * @param {string} type - Tipe notifikasi (success, error, info, warning)
 */
function showToast(message, type = 'success') {
    createToastContainer();
    
    const toastContainer = document.querySelector('.toast-container');
    const toastId = 'toast-' + Date.now();
    
    let iconClass = 'fa-check-circle';
    let bgColor = 'var(--success-color)';
    
    switch (type) {
        case 'error':
            iconClass = 'fa-exclamation-circle';
            bgColor = 'var(--danger-color)';
            break;
        case 'warning':
            iconClass = 'fa-exclamation-triangle';
            bgColor = 'var(--warning-color)';
            break;
        case 'info':
            iconClass = 'fa-info-circle';
            bgColor = 'var(--info-color)';
            break;
    }
    
    const toast = document.createElement('div');
    toast.id = toastId;
    toast.className = 'toast';
    toast.style.backgroundColor = bgColor;
    
    toast.innerHTML = `
        <div class="toast-icon">
            <i class="fas ${iconClass}"></i>
        </div>
        <div class="toast-body">${message}</div>
        <button class="toast-close" onclick="closeToast('${toastId}')">
            <i class="fas fa-times"></i>
        </button>
    `;
    
    toastContainer.appendChild(toast);
    
    // Auto close after 3 seconds
    setTimeout(() => {
        closeToast(toastId);
    }, 3000);
}

/**
 * Tutup toast notification
 * 
 * @param {string} toastId - ID toast
 */
function closeToast(toastId) {
    const toast = document.getElementById(toastId);
    if (toast) {
        toast.style.opacity = '0';
        setTimeout(() => {
            toast.remove();
        }, 300);
    }
}

/**
 * Tampilkan loading fullscreen
 */
function showPageLoading() {
    // Jika belum ada, buat elemen loading
    if (!document.querySelector('.page-loading')) {
        const loadingEl = document.createElement('div');
        loadingEl.className = 'page-loading';
        
        // Ambil warna tema primary
        const primaryColor = getComputedStyle(document.documentElement).getPropertyValue('--primary-color').trim();
        
        loadingEl.innerHTML = `
            <div class="spinner">
                <div class="bounce1" style="background-color: ${primaryColor}"></div>
                <div class="bounce2" style="background-color: ${primaryColor}"></div>
                <div class="bounce3" style="background-color: ${primaryColor}"></div>
            </div>
        `;
        
        document.body.appendChild(loadingEl);
    }
    
    document.querySelector('.page-loading').style.display = 'flex';
}

/**
 * Sembunyikan loading fullscreen
 */
function hidePageLoading() {
    const loadingEl = document.querySelector('.page-loading');
    if (loadingEl) {
        loadingEl.style.display = 'none';
    }
}