<?php
/**
 * File fungsi database
 * 
 * Berisi fungsi-fungsi untuk mengakses dan memanipulasi database
 */

// Pastikan file config.php sudah di-include sebelum file ini

/**
 * Mendapatkan data pengguna berdasarkan username
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @return array|null Data pengguna atau null jika tidak ditemukan
 */
function getUserByUsername($db, $username) {
    $stmt = $db->prepare("SELECT * FROM users WHERE username = ?");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        return $result->fetch_assoc();
    }
    
    return null;
}

/**
 * Menambahkan pengguna baru
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @param string $password Password pengguna
 * @param string $nama Nama pengguna
 * @param string $nim NIM pengguna
 * @param string $role Role pengguna
 * @param string|null $expires Tanggal kadaluarsa (nullable)
 * @return bool Berhasil atau tidak
 */
function addUser($db, $username, $password, $nama, $nim, $role = 'user', $expires = null) {
    $stmt = $db->prepare("INSERT INTO users (username, password, nama, nim, role, expires) VALUES (?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("ssssss", $username, $password, $nama, $nim, $role, $expires);
    return $stmt->execute();
}

/**
 * Menghapus pengguna
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @return bool Berhasil atau tidak
 */
function deleteUser($db, $username) {
    // Hindari menghapus super_admin
    $stmt = $db->prepare("SELECT role FROM users WHERE username = ?");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $user = $result->fetch_assoc();
        if ($user['role'] === 'super_admin' && $username === 'rizaldi') {
            return false;
        }
    }
    
    $stmt = $db->prepare("DELETE FROM users WHERE username = ?");
    $stmt->bind_param("s", $username);
    return $stmt->execute();
}

/**
 * Mengubah tema pengguna
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @param string $theme ID tema
 * @return bool Berhasil atau tidak
 */
function updateUserTheme($db, $username, $theme) {
    $stmt = $db->prepare("UPDATE users SET theme = ? WHERE username = ?");
    $stmt->bind_param("ss", $theme, $username);
    return $stmt->execute();
}

/**
 * Mendapatkan semua pengguna
 * 
 * @param mysqli $db Koneksi database
 * @return array Data semua pengguna
 */
function getAllUsers($db) {
    $result = $db->query("SELECT * FROM users ORDER BY username");
    $users = [];
    
    while ($row = $result->fetch_assoc()) {
        $users[] = $row;
    }
    
    return $users;
}

/**
 * Mendapatkan data tema berdasarkan ID
 * 
 * @param mysqli $db Koneksi database
 * @param string $themeId ID tema
 * @return array|null Data tema atau null jika tidak ditemukan
 */
function getThemeById($db, $themeId) {
    $stmt = $db->prepare("SELECT * FROM themes WHERE id = ?");
    $stmt->bind_param("s", $themeId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        return $result->fetch_assoc();
    }
    
    return null;
}

/**
 * Mendapatkan semua tema
 * 
 * @param mysqli $db Koneksi database
 * @return array Data semua tema
 */
function getAllThemes($db) {
    $result = $db->query("SELECT * FROM themes ORDER BY nama");
    $themes = [];
    
    while ($row = $result->fetch_assoc()) {
        $themes[] = $row;
    }
    
    return $themes;
}

/**
 * Menambahkan tema kustom
 * 
 * @param mysqli $db Koneksi database
 * @param string $id ID tema
 * @param string $nama Nama tema
 * @param array $colors Array warna tema
 * @return bool Berhasil atau tidak
 */
function addCustomTheme($db, $id, $nama, $colors) {
    $stmt = $db->prepare("INSERT INTO themes (id, nama, primary_color, secondary_color, danger_color, warning_color, info_color, dark_color, light_color, success_color) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("ssssssssss", $id, $nama, $colors['primary'], $colors['secondary'], $colors['danger'], $colors['warning'], $colors['info'], $colors['dark'], $colors['light'], $colors['success']);
    return $stmt->execute();
}

/**
 * Mendapatkan mata pelajaran pengguna
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @return array Data mata pelajaran
 */
function getUserMapel($db, $username) {
    $stmt = $db->prepare("SELECT * FROM mapel WHERE username = ? ORDER BY nama_mapel");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    $mapel = [];
    
    while ($row = $result->fetch_assoc()) {
        $mapel[] = $row;
    }
    
    return $mapel;
}

/**
 * Mendapatkan mata pelajaran pengguna (hanya nama)
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @return array Data nama mata pelajaran
 */
function getUserMapelNames($db, $username) {
    $stmt = $db->prepare("SELECT id, nama_mapel FROM mapel WHERE username = ? ORDER BY nama_mapel");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    $mapel = [];
    
    while ($row = $result->fetch_assoc()) {
        $mapel[$row['id']] = $row['nama_mapel'];
    }
    
    return $mapel;
}

/**
 * Menambahkan mata pelajaran
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @param string $namaMapel Nama mata pelajaran
 * @return bool Berhasil atau tidak
 */
function addMapel($db, $username, $namaMapel) {
    $stmt = $db->prepare("INSERT INTO mapel (username, nama_mapel) VALUES (?, ?)");
    $stmt->bind_param("ss", $username, $namaMapel);
    return $stmt->execute();
}

/**
 * Mengupdate mata pelajaran
 * 
 * @param mysqli $db Koneksi database
 * @param int $mapelId ID mata pelajaran
 * @param string $namaMapel Nama mata pelajaran baru
 * @return bool Berhasil atau tidak
 */
function updateMapel($db, $mapelId, $namaMapel) {
    $stmt = $db->prepare("UPDATE mapel SET nama_mapel = ? WHERE id = ?");
    $stmt->bind_param("si", $namaMapel, $mapelId);
    return $stmt->execute();
}

/**
 * Menghapus mata pelajaran
 * 
 * @param mysqli $db Koneksi database
 * @param int $mapelId ID mata pelajaran
 * @return bool Berhasil atau tidak
 */
function deleteMapel($db, $mapelId) {
    $stmt = $db->prepare("DELETE FROM mapel WHERE id = ?");
    $stmt->bind_param("i", $mapelId);
    return $stmt->execute();
}

/**
 * Mendapatkan diskusi pengguna
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @return array Data diskusi
 */
function getUserDiskusi($db, $username) {
    $stmt = $db->prepare("
        SELECT d.*, 
        (SELECT COUNT(*) FROM diskusi_selesai ds WHERE ds.diskusi_id = d.id) as selesai_count,
        (SELECT COUNT(*) FROM mapel m WHERE m.username = ?) as total_mapel
        FROM diskusi d 
        WHERE d.username = ?
        ORDER BY d.deadline IS NULL, d.deadline, d.tanggal_dibuat DESC
    ");
    $stmt->bind_param("ss", $username, $username);
    $stmt->execute();
    $result = $stmt->get_result();
    $diskusi = [];
    
    while ($row = $result->fetch_assoc()) {
        $row['selesai'] = getDiskusiSelesai($db, $row['id']);
        $diskusi[] = $row;
    }
    
    return $diskusi;
}

/**
 * Mendapatkan mata pelajaran yang sudah selesai diskusi
 * 
 * @param mysqli $db Koneksi database
 * @param int $diskusiId ID diskusi
 * @return array Data mata pelajaran selesai
 */
function getDiskusiSelesai($db, $diskusiId) {
    $stmt = $db->prepare("
        SELECT m.id, m.nama_mapel 
        FROM diskusi_selesai ds
        JOIN mapel m ON ds.mapel_id = m.id
        WHERE ds.diskusi_id = ?
    ");
    $stmt->bind_param("i", $diskusiId);
    $stmt->execute();
    $result = $stmt->get_result();
    $mapel = [];
    
    while ($row = $result->fetch_assoc()) {
        $mapel[] = $row;
    }
    
    return $mapel;
}

/**
 * Menambahkan diskusi baru
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @param string $nomor Nomor diskusi
 * @param string $tanggalDibuat Tanggal dibuat
 * @param string|null $deadline Tanggal deadline (nullable)
 * @return int|bool ID diskusi baru atau false jika gagal
 */
function addDiskusi($db, $username, $nomor, $tanggalDibuat, $deadline = null) {
    $stmt = $db->prepare("INSERT INTO diskusi (username, nomor, tanggal_dibuat, deadline) VALUES (?, ?, ?, ?)");
    $stmt->bind_param("ssss", $username, $nomor, $tanggalDibuat, $deadline);
    
    if ($stmt->execute()) {
        return $db->insert_id;
    }
    
    return false;
}

/**
 * Mengupdate diskusi
 * 
 * @param mysqli $db Koneksi database
 * @param int $diskusiId ID diskusi
 * @param string $nomor Nomor diskusi
 * @param string $tanggalDibuat Tanggal dibuat
 * @param string|null $deadline Tanggal deadline (nullable)
 * @return bool Berhasil atau tidak
 */
function updateDiskusi($db, $diskusiId, $nomor, $tanggalDibuat, $deadline = null) {
    $stmt = $db->prepare("UPDATE diskusi SET nomor = ?, tanggal_dibuat = ?, deadline = ? WHERE id = ?");
    $stmt->bind_param("sssi", $nomor, $tanggalDibuat, $deadline, $diskusiId);
    return $stmt->execute();
}

/**
 * Menghapus diskusi
 * 
 * @param mysqli $db Koneksi database
 * @param int $diskusiId ID diskusi
 * @return bool Berhasil atau tidak
 */
function deleteDiskusi($db, $diskusiId) {
    $stmt = $db->prepare("DELETE FROM diskusi WHERE id = ?");
    $stmt->bind_param("i", $diskusiId);
    return $stmt->execute();
}

/**
 * Menandai mata pelajaran sebagai selesai diskusi
 * 
 * @param mysqli $db Koneksi database
 * @param int $diskusiId ID diskusi
 * @param int $mapelId ID mata pelajaran
 * @return bool Berhasil atau tidak
 */
function completeDiskusiMapel($db, $diskusiId, $mapelId) {
    $stmt = $db->prepare("INSERT INTO diskusi_selesai (diskusi_id, mapel_id) VALUES (?, ?)");
    $stmt->bind_param("ii", $diskusiId, $mapelId);
    return $stmt->execute();
}

/**
 * Memindahkan diskusi ke arsip (selesai)
 * 
 * @param mysqli $db Koneksi database
 * @param int $diskusiId ID diskusi
 * @return bool Berhasil atau tidak
 */
function archiveDiskusi($db, $diskusiId) {
    // Ambil data diskusi
    $stmt = $db->prepare("SELECT username, nomor, tanggal_dibuat, deadline FROM diskusi WHERE id = ?");
    $stmt->bind_param("i", $diskusiId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $diskusi = $result->fetch_assoc();
        $tanggalSelesai = date('Y-m-d');
        
        // Tambahkan ke arsip
        $stmt = $db->prepare("INSERT INTO diskusi_archive (username, nomor, tanggal_dibuat, deadline, tanggal_selesai) VALUES (?, ?, ?, ?, ?)");
        $stmt->bind_param("sssss", $diskusi['username'], $diskusi['nomor'], $diskusi['tanggal_dibuat'], $diskusi['deadline'], $tanggalSelesai);
        
        if ($stmt->execute()) {
            // Hapus diskusi lama
            return deleteDiskusi($db, $diskusiId);
        }
    }
    
    return false;
}

/**
 * Mendapatkan diskusi arsip (selesai) pengguna
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @return array Data diskusi arsip
 */
function getUserDiskusiArchive($db, $username) {
    $stmt = $db->prepare("SELECT * FROM diskusi_archive WHERE username = ? ORDER BY tanggal_selesai DESC");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    $diskusi = [];
    
    while ($row = $result->fetch_assoc()) {
        $diskusi[] = $row;
    }
    
    return $diskusi;
}

/**
 * Mendapatkan tugas pengguna
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @return array Data tugas
 */
function getUserTugas($db, $username) {
    $stmt = $db->prepare("
        SELECT t.*, 
        (SELECT COUNT(*) FROM tugas_selesai ts WHERE ts.tugas_id = t.id) as selesai_count,
        (SELECT COUNT(*) FROM mapel m WHERE m.username = ?) as total_mapel
        FROM tugas t 
        WHERE t.username = ?
        ORDER BY t.deadline IS NULL, t.deadline, t.tanggal_dibuat DESC
    ");
    $stmt->bind_param("ss", $username, $username);
    $stmt->execute();
    $result = $stmt->get_result();
    $tugas = [];
    
    while ($row = $result->fetch_assoc()) {
        $row['selesai'] = getTugasSelesai($db, $row['id']);
        $tugas[] = $row;
    }
    
    return $tugas;
}

/**
 * Mendapatkan mata pelajaran yang sudah selesai tugas
 * 
 * @param mysqli $db Koneksi database
 * @param int $tugasId ID tugas
 * @return array Data mata pelajaran selesai
 */
function getTugasSelesai($db, $tugasId) {
    $stmt = $db->prepare("
        SELECT m.id, m.nama_mapel 
        FROM tugas_selesai ts
        JOIN mapel m ON ts.mapel_id = m.id
        WHERE ts.tugas_id = ?
    ");
    $stmt->bind_param("i", $tugasId);
    $stmt->execute();
    $result = $stmt->get_result();
    $mapel = [];
    
    while ($row = $result->fetch_assoc()) {
        $mapel[] = $row;
    }
    
    return $mapel;
}

/**
 * Menambahkan tugas baru
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @param string $nomor Nomor tugas
 * @param string $tanggalDibuat Tanggal dibuat
 * @param string|null $deadline Tanggal deadline (nullable)
 * @return int|bool ID tugas baru atau false jika gagal
 */
function addTugas($db, $username, $nomor, $tanggalDibuat, $deadline = null) {
    $stmt = $db->prepare("INSERT INTO tugas (username, nomor, tanggal_dibuat, deadline) VALUES (?, ?, ?, ?)");
    $stmt->bind_param("ssss", $username, $nomor, $tanggalDibuat, $deadline);
    
    if ($stmt->execute()) {
        return $db->insert_id;
    }
    
    return false;
}

/**
 * Mengupdate tugas
 * 
 * @param mysqli $db Koneksi database
 * @param int $tugasId ID tugas
 * @param string $nomor Nomor tugas
 * @param string $tanggalDibuat Tanggal dibuat
 * @param string|null $deadline Tanggal deadline (nullable)
 * @return bool Berhasil atau tidak
 */
function updateTugas($db, $tugasId, $nomor, $tanggalDibuat, $deadline = null) {
    $stmt = $db->prepare("UPDATE tugas SET nomor = ?, tanggal_dibuat = ?, deadline = ? WHERE id = ?");
    $stmt->bind_param("sssi", $nomor, $tanggalDibuat, $deadline, $tugasId);
    return $stmt->execute();
}

/**
 * Menghapus tugas
 * 
 * @param mysqli $db Koneksi database
 * @param int $tugasId ID tugas
 * @return bool Berhasil atau tidak
 */
function deleteTugas($db, $tugasId) {
    $stmt = $db->prepare("DELETE FROM tugas WHERE id = ?");
    $stmt->bind_param("i", $tugasId);
    return $stmt->execute();
}

/**
 * Menandai mata pelajaran sebagai selesai tugas
 * 
 * @param mysqli $db Koneksi database
 * @param int $tugasId ID tugas
 * @param int $mapelId ID mata pelajaran
 * @return bool Berhasil atau tidak
 */
function completeTugasMapel($db, $tugasId, $mapelId) {
    $stmt = $db->prepare("INSERT INTO tugas_selesai (tugas_id, mapel_id) VALUES (?, ?)");
    $stmt->bind_param("ii", $tugasId, $mapelId);
    return $stmt->execute();
}

/**
 * Memindahkan tugas ke arsip (selesai)
 * 
 * @param mysqli $db Koneksi database
 * @param int $tugasId ID tugas
 * @return bool Berhasil atau tidak
 */
function archiveTugas($db, $tugasId) {
    // Ambil data tugas
    $stmt = $db->prepare("SELECT username, nomor, tanggal_dibuat, deadline FROM tugas WHERE id = ?");
    $stmt->bind_param("i", $tugasId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $tugas = $result->fetch_assoc();
        $tanggalSelesai = date('Y-m-d');
        
        // Tambahkan ke arsip
        $stmt = $db->prepare("INSERT INTO tugas_archive (username, nomor, tanggal_dibuat, deadline, tanggal_selesai) VALUES (?, ?, ?, ?, ?)");
        $stmt->bind_param("sssss", $tugas['username'], $tugas['nomor'], $tugas['tanggal_dibuat'], $tugas['deadline'], $tanggalSelesai);
        
        if ($stmt->execute()) {
            // Hapus tugas lama
            return deleteTugas($db, $tugasId);
        }
    }
    
    return false;
}

/**
 * Mendapatkan tugas arsip (selesai) pengguna
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @return array Data tugas arsip
 */
function getUserTugasArchive($db, $username) {
    $stmt = $db->prepare("SELECT * FROM tugas_archive WHERE username = ? ORDER BY tanggal_selesai DESC");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    $tugas = [];
    
    while ($row = $result->fetch_assoc()) {
        $tugas[] = $row;
    }
    
    return $tugas;
}

/**
 * Mendapatkan data baca pengguna
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @return array Data baca
 */
function getUserBaca($db, $username) {
    $stmt = $db->prepare("
        SELECT b.*, m.nama_mapel,
        (SELECT COUNT(*) FROM baca_selesai bs WHERE bs.baca_id = b.id) as selesai_count
        FROM baca b
        JOIN mapel m ON b.mapel_id = m.id
        WHERE b.username = ?
        ORDER BY m.nama_mapel
    ");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    $baca = [];
    
    while ($row = $result->fetch_assoc()) {
        $row['bab_selesai'] = getBacaSelesai($db, $row['id']);
        $baca[] = $row;
    }
    
    return $baca;
}

/**
 * Mendapatkan bab yang sudah selesai dibaca
 * 
 * @param mysqli $db Koneksi database
 * @param int $bacaId ID baca
 * @return array Data bab selesai
 */
function getBacaSelesai($db, $bacaId) {
    $stmt = $db->prepare("SELECT bab_nomor FROM baca_selesai WHERE baca_id = ? ORDER BY bab_nomor");
    $stmt->bind_param("i", $bacaId);
    $stmt->execute();
    $result = $stmt->get_result();
    $bab = [];
    
    while ($row = $result->fetch_assoc()) {
        $bab[] = $row['bab_nomor'];
    }
    
    return $bab;
}

/**
 * Menambahkan data baca baru
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @param int $mapelId ID mata pelajaran
 * @param int $jumlahBab Jumlah bab
 * @return int|bool ID baca baru atau false jika gagal
 */
function addBaca($db, $username, $mapelId, $jumlahBab) {
    // Cek apakah sudah ada
    $stmt = $db->prepare("SELECT id FROM baca WHERE username = ? AND mapel_id = ?");
    $stmt->bind_param("si", $username, $mapelId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $row = $result->fetch_assoc();
        $bacaId = $row['id'];
        
        // Update jumlah bab saja
        $stmt = $db->prepare("UPDATE baca SET jumlah_bab = ? WHERE id = ?");
        $stmt->bind_param("ii", $jumlahBab, $bacaId);
        
        if ($stmt->execute()) {
            return $bacaId;
        }
        
        return false;
    }
    
    // Tambah baru
    $stmt = $db->prepare("INSERT INTO baca (username, mapel_id, jumlah_bab) VALUES (?, ?, ?)");
    $stmt->bind_param("sii", $username, $mapelId, $jumlahBab);
    
    if ($stmt->execute()) {
        return $db->insert_id;
    }
    
    return false;
}

/**
 * Mengupdate data baca
 * 
 * @param mysqli $db Koneksi database
 * @param int $bacaId ID baca
 * @param int $jumlahBab Jumlah bab baru
 * @return bool Berhasil atau tidak
 */
function updateBaca($db, $bacaId, $jumlahBab) {
    $stmt = $db->prepare("UPDATE baca SET jumlah_bab = ? WHERE id = ?");
    $stmt->bind_param("ii", $jumlahBab, $bacaId);
    return $stmt->execute();
}

/**
 * Menghapus data baca
 * 
 * @param mysqli $db Koneksi database
 * @param int $bacaId ID baca
 * @return bool Berhasil atau tidak
 */
function deleteBaca($db, $bacaId) {
    $stmt = $db->prepare("DELETE FROM baca WHERE id = ?");
    $stmt->bind_param("i", $bacaId);
    return $stmt->execute();
}

/**
 * Menandai bab sebagai selesai dibaca
 * 
 * @param mysqli $db Koneksi database
 * @param int $bacaId ID baca
 * @param int $babNomor Nomor bab
 * @return bool Berhasil atau tidak
 */
function completeBacaBab($db, $bacaId, $babNomor) {
    $stmt = $db->prepare("INSERT INTO baca_selesai (baca_id, bab_nomor) VALUES (?, ?)");
    $stmt->bind_param("ii", $bacaId, $babNomor);
    return $stmt->execute();
}

/**
 * Memindahkan baca ke arsip (selesai)
 * 
 * @param mysqli $db Koneksi database
 * @param int $bacaId ID baca
 * @return bool Berhasil atau tidak
 */
function archiveBaca($db, $bacaId) {
    // Ambil data baca
    $stmt = $db->prepare("
        SELECT b.username, b.jumlah_bab, m.nama_mapel
        FROM baca b
        JOIN mapel m ON b.mapel_id = m.id
        WHERE b.id = ?
    ");
    $stmt->bind_param("i", $bacaId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $baca = $result->fetch_assoc();
        $tanggalSelesai = date('Y-m-d');
        
        // Tambahkan ke arsip
        $stmt = $db->prepare("INSERT INTO baca_archive (username, nama_mapel, jumlah_bab, tanggal_selesai) VALUES (?, ?, ?, ?)");
        $stmt->bind_param("ssis", $baca['username'], $baca['nama_mapel'], $baca['jumlah_bab'], $tanggalSelesai);
        
        if ($stmt->execute()) {
            // Hapus baca lama
            return deleteBaca($db, $bacaId);
        }
    }
    
    return false;
}

/**
 * Mendapatkan baca arsip (selesai) pengguna
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @return array Data baca arsip
 */
function getUserBacaArchive($db, $username) {
    $stmt = $db->prepare("SELECT * FROM baca_archive WHERE username = ? ORDER BY tanggal_selesai DESC");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    $baca = [];
    
    while ($row = $result->fetch_assoc()) {
        $baca[] = $row;
    }
    
    return $baca;
}

/**
 * Mendapatkan statistik pengguna
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @return array Data statistik
 */
function getUserStats($db, $username) {
    $stats = [];
    
    // Jumlah diskusi aktif
    $stmt = $db->prepare("SELECT COUNT(*) as count FROM diskusi WHERE username = ?");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    $stats['diskusi_aktif'] = $result->fetch_assoc()['count'];
    
    // Jumlah tugas aktif
    $stmt = $db->prepare("SELECT COUNT(*) as count FROM tugas WHERE username = ?");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    $stats['tugas_aktif'] = $result->fetch_assoc()['count'];
    
    // Jumlah baca aktif
    $stmt = $db->prepare("SELECT COUNT(*) as count FROM baca WHERE username = ?");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    $stats['baca_aktif'] = $result->fetch_assoc()['count'];
    
    // Total selesai (arsip)
    $stmt = $db->prepare("
        SELECT 
        (SELECT COUNT(*) FROM diskusi_archive WHERE username = ?) +
        (SELECT COUNT(*) FROM tugas_archive WHERE username = ?) +
        (SELECT COUNT(*) FROM baca_archive WHERE username = ?) as count
    ");
    $stmt->bind_param("sss", $username, $username, $username);
    $stmt->execute();
    $result = $stmt->get_result();
    $stats['total_selesai'] = $result->fetch_assoc()['count'];
    
    return $stats;
}

/**
 * Memeriksa apakah diskusi sudah selesai
 * 
 * @param mysqli $db Koneksi database
 * @param int $diskusiId ID diskusi
 * @return bool Sudah selesai atau belum
 */
function isDiskusiComplete($db, $diskusiId) {
    $stmt = $db->prepare("
        SELECT 
        (SELECT COUNT(*) FROM diskusi_selesai WHERE diskusi_id = ?) as selesai_count,
        (SELECT COUNT(*) FROM mapel m JOIN diskusi d ON m.username = d.username WHERE d.id = ?) as total_mapel
    ");
    $stmt->bind_param("ii", $diskusiId, $diskusiId);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    
    return $row['selesai_count'] >= $row['total_mapel'];
}

/**
 * Memeriksa apakah tugas sudah selesai
 * 
 * @param mysqli $db Koneksi database
 * @param int $tugasId ID tugas
 * @return bool Sudah selesai atau belum
 */
function isTugasComplete($db, $tugasId) {
    $stmt = $db->prepare("
        SELECT 
        (SELECT COUNT(*) FROM tugas_selesai WHERE tugas_id = ?) as selesai_count,
        (SELECT COUNT(*) FROM mapel m JOIN tugas t ON m.username = t.username WHERE t.id = ?) as total_mapel
    ");
    $stmt->bind_param("ii", $tugasId, $tugasId);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    
    return $row['selesai_count'] >= $row['total_mapel'];
}

/**
 * Memeriksa apakah baca sudah selesai
 * 
 * @param mysqli $db Koneksi database
 * @param int $bacaId ID baca
 * @return bool Sudah selesai atau belum
 */
function isBacaComplete($db, $bacaId) {
    $stmt = $db->prepare("
        SELECT 
        (SELECT COUNT(*) FROM baca_selesai WHERE baca_id = ?) as selesai_count,
        (SELECT jumlah_bab FROM baca WHERE id = ?) as total_bab
    ");
    $stmt->bind_param("ii", $bacaId, $bacaId);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    
    return $row['selesai_count'] >= $row['total_bab'];
}

/**
 * Format tanggal ke format Indonesia
 * 
 * @param string|null $date Tanggal dalam format Y-m-d
 * @param bool $withDay Tampilkan nama hari
 * @return string Tanggal yang diformat atau string kosong jika null
 */
function formatDate($date, $withDay = false) {
    if (empty($date)) {
        return '';
    }
    
    $timestamp = strtotime($date);
    $month_names = [
        'Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni', 
        'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'
    ];
    $day_names = [
        'Minggu', 'Senin', 'Selasa', 'Rabu', 'Kamis', 'Jumat', 'Sabtu'
    ];
    
    $day = date('j', $timestamp);
    $month = $month_names[date('n', $timestamp) - 1];
    $year = date('Y', $timestamp);
    
    if ($withDay) {
        $dayName = $day_names[date('w', $timestamp)];
        return "$dayName, $day $month $year";
    }
    
    return "$day $month $year";
}

/**
 * Hitung sisa hari dari tanggal sekarang
 * 
 * @param string $date Tanggal deadline
 * @return int Jumlah hari tersisa
 */
function hitungSisaHari($date) {
    if (empty($date)) {
        return 0;
    }
    
    $today = time();
    $deadline = strtotime($date);
    return ceil(($deadline - $today) / (60 * 60 * 24));
}