// Initialize datepickers
const startDatePicker = flatpickr("#start_date", {
    dateFormat: "Y-m-d",
    locale: {
        firstDayOfWeek: 1,
        weekdays: {
            shorthand: ["Min", "Sen", "Sel", "Rab", "Kam", "Jum", "Sab"],
            longhand: ["Minggu", "Senin", "Selasa", "Rabu", "Kamis", "Jumat", "Sabtu"]
        },
        months: {
            shorthand: ["Jan", "Feb", "Mar", "Apr", "Mei", "Jun", "Jul", "Agt", "Sep", "Okt", "Nov", "Des"],
            longhand: ["Januari", "Februari", "Maret", "April", "Mei", "Juni", "Juli", "Agustus", "September", "Oktober", "November", "Desember"]
        }
    },
    onChange: function(selectedDates, dateStr) {
        // Update the min date of deadline picker to be the selected start date
        deadlinePicker.set('minDate', dateStr);
    }
});

const deadlinePicker = flatpickr("#deadline", {
    dateFormat: "Y-m-d",
    locale: {
        firstDayOfWeek: 1,
        weekdays: {
            shorthand: ["Min", "Sen", "Sel", "Rab", "Kam", "Jum", "Sab"],
            longhand: ["Minggu", "Senin", "Selasa", "Rabu", "Kamis", "Jumat", "Sabtu"]
        },
        months: {
            shorthand: ["Jan", "Feb", "Mar", "Apr", "Mei", "Jun", "Jul", "Agt", "Sep", "Okt", "Nov", "Des"],
            longhand: ["Januari", "Februari", "Maret", "April", "Mei", "Juni", "Juli", "Agustus", "September", "Oktober", "November", "Desember"]
        }
    }
});

// Initialize Select2 for subjects
$(document).ready(function() {
    $('#subjects').select2({
        placeholder: 'Pilih mata pelajaran',
        maximumSelectionLength: 7,
        language: {
            maximumSelected: function() {
                return 'Anda hanya dapat memilih 7 mata pelajaran';
            }
        }
    });
    
    // Show/hide subjects based on category
    $('#category').change(function() {
        if ($(this).val() === 'college') {
            $('#subjectsGroup').show();
        } else {
            $('#subjectsGroup').hide();
        }
    });
});

// Modal functionality
const modal = document.getElementById("taskModal");
const addTaskBtn = document.getElementById("addTaskBtn");
const mobileFabBtn = document.getElementById("mobileFabBtn");
const emptyAddTaskBtn = document.getElementById("emptyAddTaskBtn");
const closeModal = document.getElementById("closeModal");
const cancelTask = document.getElementById("cancelTask");
const modalTitle = document.getElementById("modalTitle").querySelector("span");
const taskForm = document.getElementById("taskForm");
const saveTaskBtn = document.getElementById("saveTaskBtn");

// Open modal in add mode
function openAddTaskModal() {
    resetForm();
    modalTitle.textContent = "Tambah Tugas Baru";
    saveTaskBtn.name = "add_task";
    modal.style.display = "flex";
    
    // Show/hide subjects based on initial category
    if (document.getElementById("category").value === 'college') {
        document.getElementById("subjectsGroup").style.display = 'block';
    } else {
        document.getElementById("subjectsGroup").style.display = 'none';
    }
}

// Add event listeners for all add task buttons
if (addTaskBtn) addTaskBtn.addEventListener("click", openAddTaskModal);
if (mobileFabBtn) mobileFabBtn.addEventListener("click", openAddTaskModal);
if (emptyAddTaskBtn) emptyAddTaskBtn.addEventListener("click", openAddTaskModal);

// Close modal
closeModal.addEventListener("click", function() {
    modal.style.display = "none";
});

cancelTask.addEventListener("click", function() {
    modal.style.display = "none";
});

window.addEventListener("click", function(event) {
    if (event.target == modal) {
        modal.style.display = "none";
    }
});

// Reset form fields
function resetForm() {
    taskForm.reset();
    document.getElementById("task_id").value = "";
    document.getElementById("original_category").value = "";
    
    // Reset select2
    $('#subjects').val(null).trigger('change');
    
    // Reset flatpickr
    const today = new Date();
    startDatePicker.setDate(today);
    deadlinePicker.setDate(today);
}

// Calendar cell click to add task
document.querySelectorAll('.calendar td[data-date]').forEach(cell => {
    cell.addEventListener('click', function() {
        const dateStr = this.getAttribute('data-date');
        resetForm();
        startDatePicker.setDate(dateStr);
        deadlinePicker.setDate(dateStr);
        
        modalTitle.textContent = "Tambah Tugas Baru";
        saveTaskBtn.name = "add_task";
        modal.style.display = "flex";
        
        // Show/hide subjects based on initial category
        if (document.getElementById("category").value === 'college') {
            document.getElementById("subjectsGroup").style.display = 'block';
        } else {
            document.getElementById("subjectsGroup").style.display = 'none';
        }
    });
});

// Edit task button functionality
const editButtons = document.querySelectorAll('.edit-task-btn');
editButtons.forEach(button => {
    button.addEventListener('click', function(e) {
        e.preventDefault();
        const taskId = this.getAttribute('data-id');
        editTask(taskId);
    });
});

// Function to load task data for editing - this uses the tasks data from PHP
// The tasks data will be populated in index.php with PHP json_encode
function editTask(taskId) {
    // Find the task by ID across all categories
    let foundTask = null;
    let category = '';
    
    for (const cat in tasks) {
        for (const task of tasks[cat]) {
            if (task.id === taskId) {
                foundTask = task;
                category = cat;
                break;
            }
        }
        if (foundTask) break;
    }
    
    if (foundTask) {
        // Set form values
        document.getElementById("task_id").value = foundTask.id;
        document.getElementById("original_category").value = category;
        document.getElementById("title").value = foundTask.title;
        document.getElementById("category").value = category;
        document.getElementById("description").value = foundTask.description || '';
        document.getElementById("link").value = foundTask.link || '';
        
        // Show subjects if this is a college task
        if (category === 'college') {
            document.getElementById("subjectsGroup").style.display = 'block';
            
            // Set selected subjects
            if (foundTask.subjects && Array.isArray(foundTask.subjects)) {
                const subjectNames = foundTask.subjects.map(s => s.name);
                $('#subjects').val(subjectNames).trigger('change');
            } else {
                $('#subjects').val(null).trigger('change');
            }
        } else {
            document.getElementById("subjectsGroup").style.display = 'none';
        }
        
        // Set dates in flatpickr
        startDatePicker.setDate(foundTask.start_date);
        deadlinePicker.setDate(foundTask.deadline);
        
        // Update modal title and submit button name
        modalTitle.textContent = "Edit Tugas";
        saveTaskBtn.name = "edit_task";
        
        // Show the modal
        modal.style.display = "flex";
    } else {
        alert("Tugas tidak ditemukan!");
    }
}

// Submit task form
taskForm.addEventListener('submit', function(e) {
    // Client-side validation
    const startDate = new Date(document.getElementById("start_date").value);
    const deadline = new Date(document.getElementById("deadline").value);
    
    if (deadline < startDate) {
        e.preventDefault();
        alert("Deadline tidak boleh sebelum tanggal mulai!");
        return false;
    }
    
    // Validate if college task has at least one subject
    if (document.getElementById("category").value === 'college') {
        const selectedSubjects = $('#subjects').val();
        if (!selectedSubjects || selectedSubjects.length === 0) {
            e.preventDefault();
            alert("Tugas kuliah harus memiliki setidaknya satu mata pelajaran!");
            return false;
        }
    }
    
    return true;
});

// Mobile Dropdown Functionality
document.getElementById('mobileCategoryDropdown').addEventListener('click', function(e) {
    e.preventDefault();
    const menu = document.getElementById('categoryDropdownMenu');
    menu.style.display = menu.style.display === 'block' ? 'none' : 'block';
});

document.getElementById('mobileStatusDropdown').addEventListener('click', function(e) {
    e.preventDefault();
    const menu = document.getElementById('statusDropdownMenu');
    menu.style.display = menu.style.display === 'block' ? 'none' : 'block';
});

// Close dropdowns when clicking outside
document.addEventListener('click', function(e) {
    if (!e.target.matches('.dropdown-toggle') && !e.target.closest('.dropdown-toggle')) {
        document.querySelectorAll('.dropdown-menu').forEach(menu => {
            menu.style.display = 'none';
        });
    }
});

// Toggle sidebar for mobile
const menuToggle = document.getElementById('menuToggle');
const sidebar = document.getElementById('sidebar');

menuToggle.addEventListener('click', function() {
    sidebar.classList.toggle('active');
});

// Close sidebar when clicking outside on mobile
document.addEventListener('click', function(e) {
    if (window.innerWidth <= 992) {
        if (!e.target.closest('.sidebar') && !e.target.closest('.menu-toggle') && sidebar.classList.contains('active')) {
            sidebar.classList.remove('active');
        }
    }
});

// Calendar popup functionality
// This requires the taskDates data to be set from PHP
// Add this in index.php:
// echo "const taskDates = " . json_encode($taskDates) . ";\n";
// echo "const tasks = " . json_encode($tasks) . ";\n";

const calendarCells = document.querySelectorAll('.calendar td[data-date]');
const calendarPopup = document.getElementById('calendar-task-popup');
const popupDate = document.getElementById('popup-date');
const popupTasks = document.getElementById('popup-tasks');
const closePopupBtn = document.getElementById('close-popup');

// Format date to more readable format
function formatDate(dateStr) {
    const date = new Date(dateStr);
    const options = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
    return date.toLocaleDateString('id-ID', options);
}

// Show calendar popup with tasks
function showCalendarPopup(dateStr, x, y) {
    if (typeof taskDates === 'undefined' || !taskDates) return;
    
    const tasks = taskDates[dateStr] || [];
    
    if (tasks.length === 0) return;
    
    popupDate.textContent = formatDate(dateStr);
    popupTasks.innerHTML = '';
    
    // Group tasks by ID to avoid duplicates (start and deadline on same day)
    const uniqueTasks = {};
    tasks.forEach(task => {
        if (!uniqueTasks[task.id]) {
            uniqueTasks[task.id] = task;
        }
    });
    
    for (const taskId in uniqueTasks) {
        const task = uniqueTasks[taskId];
        const categoryClass = `category-${task.category}`;
        const completedClass = task.completed ? 'task-completed' : '';
        
        const taskItem = document.createElement('div');
        taskItem.className = `popup-task-item ${categoryClass} ${completedClass}`;
        
        // Get icon based on category
        let categoryIcon = '';
        switch (task.category) {
            case 'work':
                categoryIcon = 'briefcase';
                break;
            case 'college':
                categoryIcon = 'graduation-cap';
                break;
            case 'personal':
                categoryIcon = 'user';
                break;
            default:
                categoryIcon = 'tasks';
        }
        
        taskItem.innerHTML = `
            <div class="popup-task-header">
                <span class="popup-task-title">
                    <i class="fas fa-${categoryIcon}"></i>
                    ${task.title}
                </span>
                ${task.completed ? '<span class="popup-task-status completed">Selesai</span>' : ''}
            </div>
            <div class="popup-task-actions">
                <a href="#" class="btn btn-xs btn-warning edit-popup-task" data-id="${task.id}">
                    <i class="fas fa-edit"></i> Edit
                </a>
                <a href="index.php?toggle=${task.id}&category=${task.category}&status=<?php echo $currentStatus; ?>" class="btn btn-xs ${task.completed ? 'btn-primary' : 'btn-success'}">
                    <i class="fas ${task.completed ? 'fa-undo' : 'fa-check'}"></i>
                    ${task.completed ? 'Batalkan' : 'Selesai'}
                </a>
            </div>
        `;
        
        popupTasks.appendChild(taskItem);
    }
    
    // Position and show popup
    calendarPopup.style.display = 'block';
    
    // Add event listeners to edit buttons in popup
    document.querySelectorAll('.edit-popup-task').forEach(btn => {
        btn.addEventListener('click', function(e) {
            e.preventDefault();
            const taskId = this.getAttribute('data-id');
            calendarPopup.style.display = 'none';
            editTask(taskId);
        });
    });
}

// Close calendar popup
if (closePopupBtn) {
    closePopupBtn.addEventListener('click', function() {
        calendarPopup.style.display = 'none';
    });
}

// Show popup when hovering over calendar cells with tasks
calendarCells.forEach(cell => {
    if (cell.classList.contains('has-tasks')) {
        cell.addEventListener('click', function(e) {
            const dateStr = this.getAttribute('data-date');
            showCalendarPopup(dateStr, e.pageX, e.pageY);
            e.stopPropagation(); // Prevent adding new task
        });
    }
});

// Close popup when clicking outside
document.addEventListener('click', function(e) {
    if (calendarPopup && !e.target.closest('.calendar-task-popup') && !e.target.closest('.calendar td.has-tasks')) {
        calendarPopup.style.display = 'none';
    }
});