<?php
// update.php - Sleep Tracker Update System

require_once 'config.php';

class SleepTrackerUpdater {
    
    private $currentVersion;
    private $latestVersion;
    private $updateUrl = 'https://api.github.com/repos/sleep-tracker/releases/latest';
    private $backupDir = 'backup/updates';
    private $logFile;
    
    public function __construct() {
        $this->currentVersion = SleepTrackerConfig::APP_VERSION;
        $this->logFile = 'logs/update_' . date('Y-m-d') . '.log';
        
        if (!file_exists($this->backupDir)) {
            mkdir($this->backupDir, 0755, true);
        }
        
        $this->log("Update system initialized - Current version: {$this->currentVersion}");
    }
    
    /**
     * Check for available updates
     */
    public function checkForUpdates() {
        $this->log("Checking for updates...");
        
        try {
            // In a real implementation, this would check a remote server
            // For now, we'll simulate version checking
            $this->latestVersion = $this->getLatestVersion();
            
            if (version_compare($this->latestVersion, $this->currentVersion, '>')) {
                $this->log("Update available: {$this->currentVersion} -> {$this->latestVersion}");
                return [
                    'update_available' => true,
                    'current_version' => $this->currentVersion,
                    'latest_version' => $this->latestVersion,
                    'changelog' => $this->getChangelog($this->latestVersion)
                ];
            } else {
                $this->log("No updates available - running latest version");
                return [
                    'update_available' => false,
                    'current_version' => $this->currentVersion,
                    'latest_version' => $this->currentVersion,
                    'message' => 'You are running the latest version'
                ];
            }
            
        } catch (Exception $e) {
            $this->log("Error checking for updates: " . $e->getMessage());
            return [
                'error' => true,
                'message' => 'Failed to check for updates: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Perform system update
     */
    public function performUpdate() {
        $this->log("Starting system update process");
        
        try {
            // Step 1: Pre-update checks
            $preCheckResult = $this->preUpdateChecks();
            if (!$preCheckResult['success']) {
                throw new Exception('Pre-update checks failed: ' . implode(', ', $preCheckResult['errors']));
            }
            
            // Step 2: Create backup
            $backupResult = $this->createPreUpdateBackup();
            if (!$backupResult['success']) {
                throw new Exception('Backup creation failed: ' . $backupResult['error']);
            }
            
            // Step 3: Download update
            $downloadResult = $this->downloadUpdate();
            if (!$downloadResult['success']) {
                throw new Exception('Update download failed: ' . $downloadResult['error']);
            }
            
            // Step 4: Apply update
            $applyResult = $this->applyUpdate($downloadResult['file']);
            if (!$applyResult['success']) {
                throw new Exception('Update application failed: ' . $applyResult['error']);
            }
            
            // Step 5: Post-update tasks
            $postUpdateResult = $this->postUpdateTasks();
            if (!$postUpdateResult['success']) {
                $this->log("Post-update tasks had issues: " . implode(', ', $postUpdateResult['warnings']));
            }
            
            $this->log("Update completed successfully");
            
            return [
                'success' => true,
                'message' => 'Update completed successfully',
                'old_version' => $this->currentVersion,
                'new_version' => $this->latestVersion,
                'backup_file' => $backupResult['file']
            ];
            
        } catch (Exception $e) {
            $this->log("Update failed: " . $e->getMessage());
            
            // Attempt rollback
            $this->rollbackUpdate();
            
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'rollback_attempted' => true
            ];
        }
    }
    
    /**
     * Pre-update system checks
     */
    private function preUpdateChecks() {
        $this->log("Running pre-update checks");
        
        $checks = [];
        $errors = [];
        
        // Check disk space
        $freeSpace = disk_free_space('.');
        $requiredSpace = 50 * 1024 * 1024; // 50MB
        
        if ($freeSpace < $requiredSpace) {
            $errors[] = "Insufficient disk space. Required: 50MB, Available: " . round($freeSpace / 1024 / 1024) . "MB";
        } else {
            $checks['disk_space'] = true;
        }
        
        // Check file permissions
        $criticalFiles = ['index.php', 'config.php', '.htaccess'];
        foreach ($criticalFiles as $file) {
            if (!is_writable($file)) {
                $errors[] = "File not writable: $file";
            }
        }
        $checks['file_permissions'] = empty($errors);
        
        // Check if maintenance mode can be enabled
        if (!is_writable('.')) {
            $errors[] = "Cannot create maintenance mode file";
        } else {
            $checks['maintenance_mode'] = true;
        }
        
        // Check database/data integrity
        $dataFiles = glob('data/*.json');
        foreach ($dataFiles as $file) {
            $data = json_decode(file_get_contents($file), true);
            if ($data === null) {
                $errors[] = "Invalid JSON in data file: " . basename($file);
            }
        }
        $checks['data_integrity'] = empty($errors);
        
        // Check backup directory
        if (!is_writable($this->backupDir)) {
            $errors[] = "Backup directory not writable: {$this->backupDir}";
        } else {
            $checks['backup_directory'] = true;
        }
        
        return [
            'success' => empty($errors),
            'checks' => $checks,
            'errors' => $errors
        ];
    }
    
    /**
     * Create pre-update backup
     */
    private function createPreUpdateBackup() {
        $this->log("Creating pre-update backup");
        
        try {
            $backupFile = $this->backupDir . '/pre_update_' . $this->currentVersion . '_' . date('Y-m-d_H-i-s') . '.zip';
            
            $zip = new ZipArchive();
            if ($zip->open($backupFile, ZipArchive::CREATE) !== TRUE) {
                throw new Exception("Cannot create backup zip file");
            }
            
            // Backup all PHP files
            $phpFiles = glob('*.php');
            foreach ($phpFiles as $file) {
                $zip->addFile($file);
            }
            
            // Backup configuration files
            $configFiles = ['.htaccess', 'manifest.json', 'robots.txt'];
            foreach ($configFiles as $file) {
                if (file_exists($file)) {
                    $zip->addFile($file);
                }
            }
            
            // Backup data directory
            $dataFiles = glob('data/*');
            foreach ($dataFiles as $file) {
                $zip->addFile($file, 'data/' . basename($file));
            }
            
            // Add system info
            $systemInfo = [
                'backup_created' => date('Y-m-d H:i:s'),
                'version' => $this->currentVersion,
                'php_version' => phpversion(),
                'system_info' => SleepTrackerConfig::getSystemInfo()
            ];
            
            $zip->addFromString('backup_info.json', json_encode($systemInfo, JSON_PRETTY_PRINT));
            
            $zip->close();
            
            $this->log("Backup created: " . basename($backupFile));
            
            return [
                'success' => true,
                'file' => $backupFile,
                'size' => filesize($backupFile)
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Download update files
     */
    private function downloadUpdate() {
        $this->log("Downloading update files");
        
        try {
            // In a real implementation, this would download from a remote server
            // For demonstration, we'll simulate this process
            
            $updateFile = 'temp/update_' . $this->latestVersion . '.zip';
            
            // Simulate download process
            $this->log("Simulating download to: $updateFile");
            
            // Create a dummy update file for demonstration
            $zip = new ZipArchive();
            if ($zip->open($updateFile, ZipArchive::CREATE) !== TRUE) {
                throw new Exception("Cannot create update file");
            }
            
            // Add dummy update files
            $zip->addFromString('version.txt', $this->latestVersion);
            $zip->addFromString('changelog.txt', $this->getChangelog($this->latestVersion));
            $zip->close();
            
            return [
                'success' => true,
                'file' => $updateFile,
                'size' => filesize($updateFile)
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Apply the update
     */
    private function applyUpdate($updateFile) {
        $this->log("Applying update from: $updateFile");
        
        try {
            // Enable maintenance mode
            $this->enableMaintenanceMode();
            
            // Extract update files
            $zip = new ZipArchive();
            if ($zip->open($updateFile) !== TRUE) {
                throw new Exception("Cannot open update file");
            }
            
            // Extract to temp directory first
            $extractPath = 'temp/update_extract';
            if (!file_exists($extractPath)) {
                mkdir($extractPath, 0755, true);
            }
            
            $zip->extractTo($extractPath);
            $zip->close();
            
            // Validate update contents
            $this->validateUpdateContents($extractPath);
            
            // Apply files (with backup of originals)
            $this->applyUpdateFiles($extractPath);
            
            // Update version information
            $this->updateVersionInfo();
            
            // Disable maintenance mode
            $this->disableMaintenanceMode();
            
            $this->log("Update applied successfully");
            
            return [
                'success' => true,
                'message' => 'Update applied successfully'
            ];
            
        } catch (Exception $e) {
            $this->disableMaintenanceMode();
            
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Post-update tasks
     */
    private function postUpdateTasks() {
        $this->log("Running post-update tasks");
        
        $warnings = [];
        
        try {
            // Clear caches
            $cacheFiles = glob('temp/*_cache.json');
            foreach ($cacheFiles as $file) {
                unlink($file);
            }
            
            // Update database schema if needed
            $this->updateDataSchema();
            
            // Regenerate configuration cache
            $this->regenerateConfigCache();
            
            // Run system health check
            $healthCheck = SleepTrackerConfig::healthCheck();
            if ($healthCheck['status'] !== 'healthy') {
                $warnings[] = "System health check shows issues after update";
            }
            
            // Clean up update files
            $this->cleanupUpdateFiles();
            
            $this->log("Post-update tasks completed");
            
            return [
                'success' => true,
                'warnings' => $warnings
            ];
            
        } catch (Exception $e) {
            $warnings[] = $e->getMessage();
            
            return [
                'success' => false,
                'warnings' => $warnings
            ];
        }
    }
    
    /**
     * Enable maintenance mode
     */
    private function enableMaintenanceMode() {
        $maintenanceFile = 'maintenance.flag';
        file_put_contents($maintenanceFile, json_encode([
            'enabled' => true,
            'reason' => 'System update in progress',
            'started' => date('Y-m-d H:i:s'),
            'estimated_duration' => '5-10 minutes'
        ]));
        
        $this->log("Maintenance mode enabled");
    }
    
    /**
     * Disable maintenance mode
     */
    private function disableMaintenanceMode() {
        $maintenanceFile = 'maintenance.flag';
        if (file_exists($maintenanceFile)) {
            unlink($maintenanceFile);
        }
        
        $this->log("Maintenance mode disabled");
    }
    
    /**
     * Validate update contents
     */
    private function validateUpdateContents($extractPath) {
        // Check for required files
        $requiredFiles = ['version.txt'];
        
        foreach ($requiredFiles as $file) {
            if (!file_exists($extractPath . '/' . $file)) {
                throw new Exception("Update package missing required file: $file");
            }
        }
        
        // Validate version
        $updateVersion = trim(file_get_contents($extractPath . '/version.txt'));
        if ($updateVersion !== $this->latestVersion) {
            throw new Exception("Version mismatch in update package");
        }
    }
    
    /**
     * Apply update files
     */
    private function applyUpdateFiles($extractPath) {
        $updateFiles = glob($extractPath . '/*');
        
        foreach ($updateFiles as $file) {
            $fileName = basename($file);
            
            // Skip version.txt and changelog.txt
            if (in_array($fileName, ['version.txt', 'changelog.txt'])) {
                continue;
            }
            
            // Backup original if exists
            if (file_exists($fileName)) {
                $backupName = $fileName . '.backup.' . time();
                copy($fileName, $backupName);
            }
            
            // Copy new file
            copy($file, $fileName);
            
            $this->log("Updated file: $fileName");
        }
    }
    
    /**
     * Update version information
     */
    private function updateVersionInfo() {
        // Update config.php with new version
        $configContent = file_get_contents('config.php');
        $configContent = preg_replace(
            '/const APP_VERSION = [\'"][^\'\"]*[\'"];/',
            "const APP_VERSION = '{$this->latestVersion}';",
            $configContent
        );
        file_put_contents('config.php', $configContent);
        
        $this->log("Version updated to: {$this->latestVersion}");
    }
    
    /**
     * Update data schema if needed
     */
    private function updateDataSchema() {
        // Check if data migration is needed
        $this->log("Checking for data schema updates");
        
        // This would contain version-specific migration code
        // For example:
        // if (version_compare($this->currentVersion, '2.1', '<')) {
        //     $this->migrateToV21();
        // }
    }
    
    /**
     * Regenerate configuration cache
     */
    private function regenerateConfigCache() {
        $cacheFile = 'temp/config_cache.json';
        
        $cache = [
            'app_version' => $this->latestVersion,
            'updated_at' => date('Y-m-d H:i:s'),
            'system_info' => SleepTrackerConfig::getSystemInfo(),
            'health_status' => SleepTrackerConfig::healthCheck()
        ];
        
        file_put_contents($cacheFile, json_encode($cache, JSON_PRETTY_PRINT));
    }
    
    /**
     * Clean up update files
     */
    private function cleanupUpdateFiles() {
        $updateFiles = glob('temp/update_*');
        foreach ($updateFiles as $file) {
            if (is_file($file)) {
                unlink($file);
            } elseif (is_dir($file)) {
                $this->removeDirectory($file);
            }
        }
        
        // Remove backup files older than 7 days
        $backupFiles = glob('*.backup.*');
        $cutoffTime = time() - (7 * 24 * 3600);
        
        foreach ($backupFiles as $file) {
            if (filemtime($file) < $cutoffTime) {
                unlink($file);
            }
        }
    }
    
    /**
     * Rollback update
     */
    private function rollbackUpdate() {
        $this->log("Attempting rollback...");
        
        try {
            // Restore backup files
            $backupFiles = glob('*.backup.*');
            foreach ($backupFiles as $backupFile) {
                $originalFile = preg_replace('/\.backup\.\d+$/', '', $backupFile);
                if (copy($backupFile, $originalFile)) {
                    $this->log("Restored: $originalFile");
                }
            }
            
            $this->disableMaintenanceMode();
            $this->log("Rollback completed");
            
        } catch (Exception $e) {
            $this->log("Rollback failed: " . $e->getMessage());
        }
    }
    
    /**
     * Get latest version (simulated)
     */
    private function getLatestVersion() {
        // In a real implementation, this would fetch from a remote server
        // For demonstration, we'll simulate version checking
        
        $versions = ['2.0', '2.1', '2.2', '2.3'];
        $currentIndex = array_search($this->currentVersion, $versions);
        
        if ($currentIndex !== false && $currentIndex < count($versions) - 1) {
            return $versions[$currentIndex + 1];
        }
        
        return $this->currentVersion; // No update available
    }
    
    /**
     * Get changelog for version
     */
    private function getChangelog($version) {
        $changelogs = [
            '2.1' => [
                'Added advanced sleep pattern analysis',
                'Improved mobile responsive design',
                'Enhanced data export features',
                'Bug fixes and performance improvements'
            ],
            '2.2' => [
                'New AI prediction algorithms',
                'Integration with wearable devices',
                'Advanced notification system',
                'Updated security features'
            ],
            '2.3' => [
                'Machine learning sleep optimization',
                'Social sharing features',
                'Enhanced analytics dashboard',
                'Performance optimizations'
            ]
        ];
        
        return $changelogs[$version] ?? ['General improvements and bug fixes'];
    }
    
    /**
     * Remove directory recursively
     */
    private function removeDirectory($dir) {
        if (!is_dir($dir)) {
            return;
        }
        
        $files = array_diff(scandir($dir), ['.', '..']);
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            if (is_dir($path)) {
                $this->removeDirectory($path);
            } else {
                unlink($path);
            }
        }
        rmdir($dir);
    }
    
    /**
     * Log message
     */
    private function log($message) {
        $timestamp = date('Y-m-d H:i:s');
        $logMessage = "[$timestamp] $message" . PHP_EOL;
        file_put_contents($this->logFile, $logMessage, FILE_APPEND | LOCK_EX);
    }
}

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    $updater = new SleepTrackerUpdater();
    
    try {
        switch ($_POST['action']) {
            case 'check_updates':
                $result = $updater->checkForUpdates();
                echo json_encode($result);
                break;
                
            case 'perform_update':
                $result = $updater->performUpdate();
                echo json_encode($result);
                break;
                
            default:
                echo json_encode(['error' => true, 'message' => 'Unknown action']);
        }
    } catch (Exception $e) {
        echo json_encode(['error' => true, 'message' => $e->getMessage()]);
    }
    exit;
}

// Web interface
if (basename($_SERVER['PHP_SELF']) === 'update.php') {
    ?>
    <!DOCTYPE html>
    <html lang="id">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>System Update - Sleep Tracker</title>
        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
        <style>
            body {
                font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                min-height: 100vh;
                padding: 20px;
                margin: 0;
            }

            .container {
                max-width: 800px;
                margin: 0 auto;
                background: rgba(255, 255, 255, 0.95);
                border-radius: 20px;
                padding: 30px;
                box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            }

            .header {
                text-align: center;
                margin-bottom: 30px;
            }

            .header h1 {
                color: #667eea;
                margin-bottom: 10px;
            }

            .update-card {
                background: white;
                border-radius: 15px;
                padding: 25px;
                margin: 20px 0;
                box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            }

            .version-info {
                display: flex;
                justify-content: space-between;
                align-items: center;
                margin-bottom: 20px;
                padding: 15px;
                background: #f8f9fa;
                border-radius: 10px;
            }

            .btn {
                background: linear-gradient(45deg, #667eea, #764ba2);
                color: white;
                padding: 12px 25px;
                border: none;
                border-radius: 10px;
                font-size: 16px;
                cursor: pointer;
                transition: all 0.3s ease;
                margin: 5px;
            }

            .btn:hover {
                transform: translateY(-2px);
                box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
            }

            .btn:disabled {
                background: #6c757d;
                cursor: not-allowed;
                transform: none;
            }

            .btn-success {
                background: linear-gradient(45deg, #28a745, #20c997);
            }

            .btn-warning {
                background: linear-gradient(45deg, #ffc107, #fd7e14);
            }

            .btn-danger {
                background: linear-gradient(45deg, #dc3545, #c82333);
            }

            .changelog {
                background: #e8f4fd;
                border: 1px solid #b8daff;
                border-radius: 10px;
                padding: 15px;
                margin-top: 15px;
            }

            .changelog ul {
                margin: 10px 0;
                padding-left: 20px;
            }

            .progress {
                width: 100%;
                height: 20px;
                background: #e9ecef;
                border-radius: 10px;
                overflow: hidden;
                margin: 15px 0;
            }

            .progress-bar {
                height: 100%;
                background: linear-gradient(45deg, #667eea, #764ba2);
                width: 0%;
                transition: width 0.3s ease;
            }

            .status {
                padding: 15px;
                border-radius: 10px;
                margin: 15px 0;
            }

            .status.success {
                background: #d4edda;
                border: 1px solid #c3e6cb;
                color: #155724;
            }

            .status.error {
                background: #f8d7da;
                border: 1px solid #f5c6cb;
                color: #721c24;
            }

            .status.warning {
                background: #fff3cd;
                border: 1px solid #ffeaa7;
                color: #856404;
            }

            .loading {
                display: none;
                text-align: center;
                padding: 20px;
            }

            .loading i {
                font-size: 2em;
                color: #667eea;
                animation: spin 1s linear infinite;
            }

            @keyframes spin {
                from { transform: rotate(0deg); }
                to { transform: rotate(360deg); }
            }

            .nav-links {
                text-align: center;
                margin-bottom: 20px;
            }

            .nav-links a {
                display: inline-block;
                padding: 8px 15px;
                margin: 0 5px;
                background: #667eea;
                color: white;
                text-decoration: none;
                border-radius: 20px;
                font-size: 0.9em;
            }
        </style>
    </head>
    <body>
        <div class="container">
            <div class="header">
                <h1><i class="fas fa-sync-alt"></i> System Update</h1>
                <p>Sleep Tracker Update Management</p>
            </div>

            <div class="nav-links">
                <a href="index.php"><i class="fas fa-home"></i> Dashboard</a>
                <a href="config.php"><i class="fas fa-cog"></i> Config</a>
                <a href="maintenance.php"><i class="fas fa-tools"></i> Maintenance</a>
            </div>

            <div class="update-card">
                <h3><i class="fas fa-info-circle"></i> Version Information</h3>
                <div class="version-info">
                    <div>
                        <strong>Current Version:</strong>
                        <span id="currentVersion"><?php echo SleepTrackerConfig::APP_VERSION; ?></span>
                    </div>
                    <div>
                        <strong>Latest Version:</strong>
                        <span id="latestVersion">Checking...</span>
                    </div>
                </div>
                
                <div style="text-align: center;">
                    <button id="checkUpdatesBtn" class="btn" onclick="checkForUpdates()">
                        <i class="fas fa-search"></i> Check for Updates
                    </button>
                    <button id="performUpdateBtn" class="btn btn-success" onclick="performUpdate()" style="display: none;">
                        <i class="fas fa-download"></i> Install Update
                    </button>
                </div>
            </div>

            <div id="updateInfo" class="update-card" style="display: none;">
                <h3><i class="fas fa-list"></i> What's New</h3>
                <div id="changelog" class="changelog"></div>
            </div>

            <div id="updateProgress" class="update-card" style="display: none;">
                <h3><i class="fas fa-progress"></i> Update Progress</h3>
                <div class="progress">
                    <div id="progressBar" class="progress-bar"></div>
                </div>
                <div id="progressText" style="text-align: center; margin-top: 10px;">
                    Preparing update...
                </div>
            </div>

            <div class="loading" id="loading">
                <i class="fas fa-spinner"></i>
                <p>Processing...</p>
            </div>

            <div id="statusMessage"></div>

            <div class="update-card">
                <h3><i class="fas fa-shield-alt"></i> Update Safety</h3>
                <p><strong>Automatic Backup:</strong> Your data will be automatically backed up before any update.</p>
                <p><strong>Rollback Available:</strong> If anything goes wrong, the system can be rolled back to the previous version.</p>
                <p><strong>Zero Downtime:</strong> The update process is designed to minimize service interruption.</p>
                
                <div style="margin-top: 15px; padding: 10px; background: #fff3cd; border-radius: 5px;">
                    <strong>⚠️ Important:</strong> Please ensure you have a stable internet connection and avoid closing this page during the update process.
                </div>
            </div>
        </div>

        <script>
            let updateAvailable = false;

            async function checkForUpdates() {
                const btn = document.getElementById('checkUpdatesBtn');
                const loading = document.getElementById('loading');
                
                btn.disabled = true;
                loading.style.display = 'block';
                
                try {
                    const response = await fetch('update.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: 'action=check_updates'
                    });
                    
                    const result = await response.json();
                    
                    if (result.error) {
                        showStatus(result.message, 'error');
                    } else if (result.update_available) {
                        document.getElementById('latestVersion').textContent = result.latest_version;
                        document.getElementById('performUpdateBtn').style.display = 'inline-block';
                        
                        // Show changelog
                        const changelog = document.getElementById('changelog');
                        changelog.innerHTML = '<h4>New in version ' + result.latest_version + ':</h4><ul>';
                        result.changelog.forEach(item => {
                            changelog.innerHTML += '<li>' + item + '</li>';
                        });
                        changelog.innerHTML += '</ul>';
                        
                        document.getElementById('updateInfo').style.display = 'block';
                        updateAvailable = true;
                        
                        showStatus('Update available! Version ' + result.latest_version + ' is ready to install.', 'warning');
                    } else {
                        document.getElementById('latestVersion').textContent = result.current_version;
                        showStatus('You are running the latest version!', 'success');
                    }
                    
                } catch (error) {
                    showStatus('Error checking for updates: ' + error.message, 'error');
                } finally {
                    btn.disabled = false;
                    loading.style.display = 'none';
                }
            }

            async function performUpdate() {
                if (!updateAvailable) {
                    showStatus('No update available to install.', 'warning');
                    return;
                }

                if (!confirm('Are you sure you want to perform the update? The system will be temporarily unavailable during the process.')) {
                    return;
                }

                const btn = document.getElementById('performUpdateBtn');
                const progressCard = document.getElementById('updateProgress');
                const progressBar = document.getElementById('progressBar');
                const progressText = document.getElementById('progressText');
                
                btn.disabled = true;
                progressCard.style.display = 'block';
                
                // Simulate progress
                let progress = 0;
                const progressInterval = setInterval(() => {
                    progress += Math.random() * 10;
                    if (progress > 90) progress = 90;
                    
                    progressBar.style.width = progress + '%';
                    
                    if (progress < 30) {
                        progressText.textContent = 'Downloading update files...';
                    } else if (progress < 60) {
                        progressText.textContent = 'Creating backup...';
                    } else if (progress < 90) {
                        progressText.textContent = 'Installing update...';
                    }
                }, 500);
                
                try {
                    const response = await fetch('update.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: 'action=perform_update'
                    });
                    
                    const result = await response.json();
                    
                    clearInterval(progressInterval);
                    progressBar.style.width = '100%';
                    progressText.textContent = 'Update completed!';
                    
                    if (result.success) {
                        showStatus('Update completed successfully! The system has been updated from version ' + 
                                 result.old_version + ' to ' + result.new_version + '.', 'success');
                        
                        setTimeout(() => {
                            if (confirm('Update completed! Would you like to refresh the page to see the new version?')) {
                                location.reload();
                            }
                        }, 2000);
                    } else {
                        showStatus('Update failed: ' + result.error, 'error');
                        if (result.rollback_attempted) {
                            showStatus('System has been rolled back to the previous version.', 'warning');
                        }
                    }
                    
                } catch (error) {
                    clearInterval(progressInterval);
                    showStatus('Update failed: ' + error.message, 'error');
                } finally {
                    btn.disabled = false;
                }
            }

            function showStatus(message, type) {
                const statusDiv = document.getElementById('statusMessage');
                statusDiv.innerHTML = '<div class="status ' + type + '">' + message + '</div>';
                
                // Auto-hide after 10 seconds for success messages
                if (type === 'success') {
                    setTimeout(() => {
                        statusDiv.innerHTML = '';
                    }, 10000);
                }
            }

            // Check for updates on page load
            window.addEventListener('load', () => {
                checkForUpdates();
            });
        </script>
    </body>
    </html>
    <?php
}
?>