<?php
session_start();

// Helper function to calculate sleep duration
function calculateSleepDuration($bedTime, $wakeTime) {
    $bedDateTime = new DateTime("2000-01-01 $bedTime");
    $wakeDateTime = new DateTime("2000-01-01 $wakeTime");
    
    // Handle case where wake time is next day
    if ($wakeDateTime <= $bedDateTime) {
        $wakeDateTime->add(new DateInterval('P1D'));
    }
    
    $interval = $bedDateTime->diff($wakeDateTime);
    return round($interval->h + ($interval->i / 60), 1);
}

// Helper function to get sleep quality based on duration and type
function getSleepQuality($duration, $type) {
    switch ($type) {
        case 'malam':
            if ($duration >= 6 && $duration <= 9) return 'Excellent';
            if ($duration >= 5 && $duration <= 10) return 'Good';
            if ($duration >= 4 && $duration <= 11) return 'Fair';
            return 'Poor';
        
        case 'siang':
            if ($duration >= 1 && $duration <= 3) return 'Excellent';
            if ($duration >= 0.5 && $duration <= 4) return 'Good';
            return 'Fair';
        
        case 'pagi':
        case 'sore':
            if ($duration >= 1 && $duration <= 2) return 'Excellent';
            if ($duration >= 0.5 && $duration <= 3) return 'Good';
            return 'Fair';
        
        default:
            return 'Unknown';
    }
}

// Helper function to load monthly data
function loadMonthData($month) {
    $dataFile = "data/{$month}.json";
    if (!file_exists($dataFile)) {
        return [];
    }
    
    $data = file_get_contents($dataFile);
    return json_decode($data, true) ?: [];
}

// Helper function to save monthly data
function saveMonthData($month, $data) {
    $dataFile = "data/{$month}.json";
    
    // Create backup before saving
    if (file_exists($dataFile)) {
        $backupFile = "backup/{$month}_" . date('Y-m-d_H-i-s') . ".json";
        copy($dataFile, $backupFile);
        
        // Keep only last 10 backups per month
        $backupFiles = glob("backup/{$month}_*.json");
        if (count($backupFiles) > 10) {
            usort($backupFiles, function($a, $b) {
                return filemtime($a) - filemtime($b);
            });
            unlink($backupFiles[0]);
        }
    }
    
    return file_put_contents($dataFile, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

// Helper function to find or create daily entry
function findOrCreateDailyEntry(&$monthData, $date) {
    foreach ($monthData as &$entry) {
        if ($entry['date'] === $date) {
            return $entry;
        }
    }
    
    // Create new entry if not found
    $newEntry = [
        'date' => $date,
        'sleepSessions' => [],
        'workActivity' => '',
        'location' => '',
        'notes' => '',
        'createdAt' => date('Y-m-d H:i:s'),
        'updatedAt' => date('Y-m-d H:i:s')
    ];
    
    $monthData[] = $newEntry;
    return $monthData[count($monthData) - 1];
}

// Handle GET requests for AJAX
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action'])) {
    if ($_GET['action'] === 'get_predictions') {
        include 'generate_predictions.php';
        exit;
    }
}

// Handle POST requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $date = $_POST['date'] ?? date('Y-m-d');
    $month = substr($date, 0, 7); // YYYY-MM
    
    // Load current month data
    $monthData = loadMonthData($month);
    
    switch ($action) {
        case 'add_session':
            try {
                $sleepType = $_POST['sleepType'] ?? '';
                $bedTime = $_POST['bedTime'] ?? '';
                $wakeTime = $_POST['wakeTime'] ?? '';
                
                if (empty($sleepType) || empty($bedTime) || empty($wakeTime)) {
                    throw new Exception('Data tidak lengkap!');
                }
                
                // Validate sleep type time ranges
                $hour = (int)substr($bedTime, 0, 2);
                $validTypes = [
                    'pagi' => [4, 5, 6, 7, 8, 9, 10],
                    'siang' => [11, 12, 13, 14, 15],
                    'sore' => [16, 17, 18, 19],
                    'malam' => [20, 21, 22, 23, 0, 1, 2, 3]
                ];
                
                if (!in_array($hour, $validTypes[$sleepType])) {
                    throw new Exception("Jam tidur tidak sesuai dengan jenis tidur {$sleepType}!");
                }
                
                $duration = calculateSleepDuration($bedTime, $wakeTime);
                
                if ($duration > 12) {
                    throw new Exception('Durasi tidur terlalu lama! Maksimal 12 jam.');
                }
                
                if ($duration < 0.5) {
                    throw new Exception('Durasi tidur terlalu pendek! Minimal 30 menit.');
                }
                
                $quality = getSleepQuality($duration, $sleepType);
                $cycles = round($duration / 1.5);
                
                // Find or create daily entry
                $dailyEntry = &findOrCreateDailyEntry($monthData, $date);
                
                // Check for overlapping sessions
                foreach ($dailyEntry['sleepSessions'] as $session) {
                    $existingBed = new DateTime("2000-01-01 {$session['bedTime']}");
                    $existingWake = new DateTime("2000-01-01 {$session['wakeTime']}");
                    $newBed = new DateTime("2000-01-01 $bedTime");
                    $newWake = new DateTime("2000-01-01 $wakeTime");
                    
                    if ($existingWake <= $existingBed) {
                        $existingWake->add(new DateInterval('P1D'));
                    }
                    if ($newWake <= $newBed) {
                        $newWake->add(new DateInterval('P1D'));
                    }
                    
                    // Check overlap
                    if (($newBed >= $existingBed && $newBed < $existingWake) ||
                        ($newWake > $existingBed && $newWake <= $existingWake) ||
                        ($newBed <= $existingBed && $newWake >= $existingWake)) {
                        throw new Exception('Waktu tidur bertabrakan dengan sesi yang sudah ada!');
                    }
                }
                
                // Add new session
                $newSession = [
                    'type' => $sleepType,
                    'bedTime' => $bedTime,
                    'wakeTime' => $wakeTime,
                    'duration' => $duration,
                    'cycles' => $cycles,
                    'quality' => $quality,
                    'createdAt' => date('Y-m-d H:i:s')
                ];
                
                $dailyEntry['sleepSessions'][] = $newSession;
                $dailyEntry['updatedAt'] = date('Y-m-d H:i:s');
                
                // Save data
                if (saveMonthData($month, $monthData)) {
                    $_SESSION['message'] = "Sesi tidur {$sleepType} berhasil ditambahkan! Durasi: {$duration}h ({$cycles} siklus)";
                    $_SESSION['message_type'] = 'success';
                } else {
                    throw new Exception('Gagal menyimpan data!');
                }
                
            } catch (Exception $e) {
                $_SESSION['message'] = $e->getMessage();
                $_SESSION['message_type'] = 'error';
            }
            break;
            
        case 'delete_session':
            try {
                $sessionIndex = (int)($_POST['sessionIndex'] ?? -1);
                
                // Find daily entry
                $dailyEntry = &findOrCreateDailyEntry($monthData, $date);
                
                if (!isset($dailyEntry['sleepSessions'][$sessionIndex])) {
                    throw new Exception('Sesi tidur tidak ditemukan!');
                }
                
                $deletedSession = $dailyEntry['sleepSessions'][$sessionIndex];
                array_splice($dailyEntry['sleepSessions'], $sessionIndex, 1);
                $dailyEntry['updatedAt'] = date('Y-m-d H:i:s');
                
                // Save data
                if (saveMonthData($month, $monthData)) {
                    $_SESSION['message'] = "Sesi tidur {$deletedSession['type']} berhasil dihapus!";
                    $_SESSION['message_type'] = 'success';
                } else {
                    throw new Exception('Gagal menghapus data!');
                }
                
            } catch (Exception $e) {
                $_SESSION['message'] = $e->getMessage();
                $_SESSION['message_type'] = 'error';
            }
            break;
            
        case 'update_activity':
            try {
                $workActivity = $_POST['workActivity'] ?? '';
                $location = $_POST['location'] ?? '';
                $notes = $_POST['notes'] ?? '';
                
                if (empty($workActivity) || empty($location)) {
                    throw new Exception('Aktivitas dan lokasi harus diisi!');
                }
                
                // Find or create daily entry
                $dailyEntry = &findOrCreateDailyEntry($monthData, $date);
                
                $dailyEntry['workActivity'] = $workActivity;
                $dailyEntry['location'] = $location;
                $dailyEntry['notes'] = $notes;
                $dailyEntry['updatedAt'] = date('Y-m-d H:i:s');
                
                // Save data
                if (saveMonthData($month, $monthData)) {
                    $_SESSION['message'] = 'Aktivitas harian berhasil diupdate!';
                    $_SESSION['message_type'] = 'success';
                } else {
                    throw new Exception('Gagal menyimpan data aktivitas!');
                }
                
            } catch (Exception $e) {
                $_SESSION['message'] = $e->getMessage();
                $_SESSION['message_type'] = 'error';
            }
            break;
            
        default:
            $_SESSION['message'] = 'Aksi tidak valid!';
            $_SESSION['message_type'] = 'error';
            break;
    }
    
    // Redirect back to main page
    header('Location: index.php');
    exit;
}

// Generate statistics for API calls
function generateStatistics() {
    $stats = [
        'totalDays' => 0,
        'totalSessions' => 0,
        'avgDuration' => 0,
        'totalDuration' => 0,
        'qualityDistribution' => [
            'Excellent' => 0,
            'Good' => 0,
            'Fair' => 0,
            'Poor' => 0
        ],
        'typeDistribution' => [
            'malam' => 0,
            'siang' => 0,
            'pagi' => 0,
            'sore' => 0
        ]
    ];
    
    // Get all months data
    $dataFiles = glob('data/*.json');
    
    foreach ($dataFiles as $file) {
        $monthData = json_decode(file_get_contents($file), true) ?: [];
        
        foreach ($monthData as $dayEntry) {
            if (!empty($dayEntry['sleepSessions'])) {
                $stats['totalDays']++;
                
                foreach ($dayEntry['sleepSessions'] as $session) {
                    $stats['totalSessions']++;
                    $stats['totalDuration'] += $session['duration'];
                    $stats['qualityDistribution'][$session['quality']]++;
                    $stats['typeDistribution'][$session['type']]++;
                }
            }
        }
    }
    
    if ($stats['totalSessions'] > 0) {
        $stats['avgDuration'] = round($stats['totalDuration'] / $stats['totalSessions'], 1);
    }
    
    return $stats;
}

// API endpoint for statistics
if (isset($_GET['api']) && $_GET['api'] === 'stats') {
    header('Content-Type: application/json');
    echo json_encode(generateStatistics());
    exit;
}

// If accessed directly, redirect to main page
header('Location: index.php');
exit;
?>