#!/bin/bash

# Sleep Tracker - Automated Installation Script
# =============================================
# Automated setup for Sleep Tracker application
# Created for Rizaldi's personal sleep management system

set -e  # Exit on any error

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
PURPLE='\033[0;35m'
CYAN='\033[0;36m'
WHITE='\033[1;37m'
NC='\033[0m' # No Color

# Configuration
APP_NAME="Sleep Tracker"
APP_VERSION="2.0"
APP_OWNER="Rizaldi"
REQUIRED_PHP_VERSION="7.0"
WEB_ROOT="/var/www/html"
DOMAIN_NAME=""
SSL_ENABLED=false

# Log file
LOG_FILE="install_$(date +%Y%m%d_%H%M%S).log"

# Function to print colored output
print_status() {
    echo -e "${GREEN}[INFO]${NC} $1" | tee -a "$LOG_FILE"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1" | tee -a "$LOG_FILE"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1" | tee -a "$LOG_FILE"
}

print_header() {
    echo -e "${PURPLE}$1${NC}" | tee -a "$LOG_FILE"
}

print_step() {
    echo -e "${CYAN}[STEP]${NC} $1" | tee -a "$LOG_FILE"
}

# Function to check if command exists
command_exists() {
    command -v "$1" >/dev/null 2>&1
}

# Function to check PHP version
check_php_version() {
    if command_exists php; then
        PHP_VERSION=$(php -r "echo PHP_VERSION;" 2>/dev/null)
        if [ $? -eq 0 ]; then
            print_status "PHP version: $PHP_VERSION"
            
            # Compare versions
            if php -r "exit(version_compare(PHP_VERSION, '$REQUIRED_PHP_VERSION', '<') ? 1 : 0);"; then
                print_error "PHP version $REQUIRED_PHP_VERSION or higher required. Found: $PHP_VERSION"
                return 1
            else
                print_status "PHP version check passed"
                return 0
            fi
        else
            print_error "Unable to determine PHP version"
            return 1
        fi
    else
        print_error "PHP is not installed"
        return 1
    fi
}

# Function to check required PHP extensions
check_php_extensions() {
    print_step "Checking required PHP extensions..."
    
    REQUIRED_EXTENSIONS=("json" "curl" "mbstring" "zip" "fileinfo")
    MISSING_EXTENSIONS=()
    
    for ext in "${REQUIRED_EXTENSIONS[@]}"; do
        if php -m | grep -q "^$ext$"; then
            print_status "PHP extension '$ext' is available"
        else
            MISSING_EXTENSIONS+=("$ext")
            print_error "PHP extension '$ext' is missing"
        fi
    done
    
    if [ ${#MISSING_EXTENSIONS[@]} -gt 0 ]; then
        print_error "Missing PHP extensions: ${MISSING_EXTENSIONS[*]}"
        print_status "Install missing extensions with: sudo apt-get install php-${MISSING_EXTENSIONS[*]// / php-}"
        return 1
    else
        print_status "All required PHP extensions are available"
        return 0
    fi
}

# Function to detect web server
detect_web_server() {
    print_step "Detecting web server..."
    
    if command_exists apache2; then
        WEB_SERVER="apache2"
        print_status "Detected Apache web server"
    elif command_exists nginx; then
        WEB_SERVER="nginx"
        print_status "Detected Nginx web server"
    elif command_exists httpd; then
        WEB_SERVER="httpd"
        print_status "Detected Apache HTTPD web server"
    else
        print_warning "No web server detected. You may need to install Apache or Nginx"
        WEB_SERVER="none"
    fi
}

# Function to check system requirements
check_system_requirements() {
    print_header "=== System Requirements Check ==="
    
    # Check OS
    print_step "Checking operating system..."
    OS=$(uname -s)
    ARCH=$(uname -m)
    print_status "Operating System: $OS ($ARCH)"
    
    # Check disk space
    print_step "Checking disk space..."
    AVAILABLE_SPACE=$(df . | awk 'NR==2 {print $4}')
    REQUIRED_SPACE=102400  # 100MB in KB
    
    if [ "$AVAILABLE_SPACE" -gt "$REQUIRED_SPACE" ]; then
        print_status "Sufficient disk space available: $(($AVAILABLE_SPACE / 1024))MB"
    else
        print_error "Insufficient disk space. Required: 100MB, Available: $(($AVAILABLE_SPACE / 1024))MB"
        return 1
    fi
    
    # Check memory
    if [ -f /proc/meminfo ]; then
        TOTAL_RAM=$(grep MemTotal /proc/meminfo | awk '{print $2}')
        print_status "Total RAM: $(($TOTAL_RAM / 1024))MB"
    fi
    
    # Check PHP
    check_php_version || return 1
    check_php_extensions || return 1
    
    # Detect web server
    detect_web_server
    
    print_status "System requirements check completed"
}

# Function to create directories
create_directories() {
    print_step "Creating application directories..."
    
    DIRECTORIES=("data" "backup" "logs" "temp" "archive")
    
    for dir in "${DIRECTORIES[@]}"; do
        if [ ! -d "$dir" ]; then
            mkdir -p "$dir"
            chmod 755 "$dir"
            print_status "Created directory: $dir"
        else
            print_status "Directory already exists: $dir"
        fi
    done
    
    # Create .htaccess files for security
    for dir in "${DIRECTORIES[@]}"; do
        if [ ! -f "$dir/.htaccess" ]; then
            cat > "$dir/.htaccess" << 'EOF'
Order deny,allow
Deny from all
EOF
            print_status "Created .htaccess security file in $dir"
        fi
    done
}

# Function to set file permissions
set_permissions() {
    print_step "Setting file permissions..."
    
    # PHP files
    find . -name "*.php" -exec chmod 644 {} \;
    print_status "Set permissions for PHP files"
    
    # Configuration files
    if [ -f ".htaccess" ]; then
        chmod 644 .htaccess
        print_status "Set permissions for .htaccess"
    fi
    
    # Data directories
    chmod -R 755 data backup logs temp
    print_status "Set permissions for data directories"
    
    # Make sure web server can write to directories
    if [ "$WEB_SERVER" != "none" ]; then
        WEB_USER=$(ps -ef | egrep '(httpd|apache2|nginx)' | grep -v root | head -n1 | awk '{print $1}')
        if [ -n "$WEB_USER" ]; then
            chown -R "$WEB_USER:$WEB_USER" data backup logs temp 2>/dev/null || print_warning "Could not change ownership to web user"
        fi
    fi
}

# Function to configure web server
configure_web_server() {
    print_step "Configuring web server..."
    
    case $WEB_SERVER in
        "apache2"|"httpd")
            configure_apache
            ;;
        "nginx")
            configure_nginx
            ;;
        *)
            print_warning "Web server configuration skipped - no supported web server detected"
            ;;
    esac
}

# Function to configure Apache
configure_apache() {
    print_status "Configuring Apache web server..."
    
    # Enable required modules
    APACHE_MODULES=("rewrite" "headers" "deflate" "expires")
    
    for module in "${APACHE_MODULES[@]}"; do
        if command_exists a2enmod; then
            a2enmod "$module" 2>/dev/null || print_warning "Could not enable Apache module: $module"
        fi
    done
    
    # Create virtual host if domain is provided
    if [ -n "$DOMAIN_NAME" ]; then
        create_apache_vhost
    fi
    
    # Restart Apache
    if command_exists systemctl; then
        systemctl reload apache2 2>/dev/null || systemctl reload httpd 2>/dev/null || print_warning "Could not reload Apache"
    fi
}

# Function to configure Nginx
configure_nginx() {
    print_status "Configuring Nginx web server..."
    
    if [ -n "$DOMAIN_NAME" ]; then
        create_nginx_config
    fi
    
    # Test and reload Nginx
    if command_exists nginx; then
        nginx -t && systemctl reload nginx 2>/dev/null || print_warning "Could not reload Nginx"
    fi
}

# Function to create Apache virtual host
create_apache_vhost() {
    VHOST_FILE="/etc/apache2/sites-available/${DOMAIN_NAME}.conf"
    
    cat > "$VHOST_FILE" << EOF
<VirtualHost *:80>
    ServerName $DOMAIN_NAME
    DocumentRoot $(pwd)
    
    <Directory $(pwd)>
        AllowOverride All
        Require all granted
    </Directory>
    
    ErrorLog \${APACHE_LOG_DIR}/${DOMAIN_NAME}_error.log
    CustomLog \${APACHE_LOG_DIR}/${DOMAIN_NAME}_access.log combined
</VirtualHost>
EOF

    a2ensite "$DOMAIN_NAME" 2>/dev/null
    print_status "Created Apache virtual host for $DOMAIN_NAME"
}

# Function to create Nginx config
create_nginx_config() {
    NGINX_CONFIG="/etc/nginx/sites-available/${DOMAIN_NAME}"
    
    cat > "$NGINX_CONFIG" << EOF
server {
    listen 80;
    server_name $DOMAIN_NAME;
    root $(pwd);
    index index.php index.html;
    
    location / {
        try_files \$uri \$uri/ /index.php?\$query_string;
    }
    
    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php7.4-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME \$document_root\$fastcgi_script_name;
        include fastcgi_params;
    }
    
    location ~ /\. {
        deny all;
    }
    
    location ~* \.(json|log|bak|backup)$ {
        deny all;
    }
}
EOF

    ln -sf "$NGINX_CONFIG" "/etc/nginx/sites-enabled/"
    print_status "Created Nginx configuration for $DOMAIN_NAME"
}

# Function to run application setup
run_app_setup() {
    print_step "Running application setup..."
    
    if [ -f "setup.php" ]; then
        # Run setup via command line if possible
        if command_exists php; then
            print_status "Running PHP setup script..."
            echo "Running automated setup..." | php setup.php 2>&1 | tee -a "$LOG_FILE"
        else
            print_status "Please run setup.php in your web browser to complete the installation"
        fi
    else
        print_error "setup.php not found in current directory"
        return 1
    fi
}

# Function to create sample data
create_sample_data() {
    print_step "Creating sample data for demonstration..."
    
    SAMPLE_DATA='[
  {
    "date": "'$(date -d "3 days ago" +%Y-%m-%d)'",
    "sleepSessions": [
      {
        "type": "malam",
        "bedTime": "22:30",
        "wakeTime": "06:30",
        "duration": 8.0,
        "cycles": 5,
        "quality": "Excellent",
        "createdAt": "'$(date -d "3 days ago" +%Y-%m-%d)' 22:30:00"
      }
    ],
    "workActivity": "sedang",
    "location": "rumah",
    "notes": "Data contoh - hari produktif",
    "createdAt": "'$(date -d "3 days ago" +%Y-%m-%d)' 08:00:00",
    "updatedAt": "'$(date -d "3 days ago" +%Y-%m-%d)' 22:31:00"
  },
  {
    "date": "'$(date -d "2 days ago" +%Y-%m-%d)'",
    "sleepSessions": [
      {
        "type": "malam",
        "bedTime": "23:00",
        "wakeTime": "07:00",
        "duration": 8.0,
        "cycles": 5,
        "quality": "Good",
        "createdAt": "'$(date -d "2 days ago" +%Y-%m-%d)' 23:00:00"
      },
      {
        "type": "siang",
        "bedTime": "13:30",
        "wakeTime": "14:30",
        "duration": 1.0,
        "cycles": 1,
        "quality": "Good",
        "createdAt": "'$(date -d "2 days ago" +%Y-%m-%d)' 13:30:00"
      }
    ],
    "workActivity": "ringan",
    "location": "rumah",
    "notes": "Data contoh - termasuk tidur siang",
    "createdAt": "'$(date -d "2 days ago" +%Y-%m-%d)' 08:00:00",
    "updatedAt": "'$(date -d "2 days ago" +%Y-%m-%d)' 23:01:00"
  },
  {
    "date": "'$(date -d "1 day ago" +%Y-%m-%d)'",
    "sleepSessions": [
      {
        "type": "malam",
        "bedTime": "22:00",
        "wakeTime": "06:00",
        "duration": 8.0,
        "cycles": 5,
        "quality": "Excellent",
        "createdAt": "'$(date -d "1 day ago" +%Y-%m-%d)' 22:00:00"
      }
    ],
    "workActivity": "berat",
    "location": "keluar",
    "notes": "Data contoh - aktivitas berat",
    "createdAt": "'$(date -d "1 day ago" +%Y-%m-%d)' 08:00:00",
    "updatedAt": "'$(date -d "1 day ago" +%Y-%m-%d)' 22:01:00"
  }
]'

    CURRENT_MONTH=$(date +%Y-%m)
    echo "$SAMPLE_DATA" > "data/${CURRENT_MONTH}.json"
    print_status "Created sample data file: data/${CURRENT_MONTH}.json"
}

# Function to setup cron jobs
setup_cron_jobs() {
    print_step "Setting up automated maintenance cron jobs..."
    
    # Create cron script
    CRON_SCRIPT="cron_maintenance.sh"
    cat > "$CRON_SCRIPT" << 'EOF'
#!/bin/bash
# Sleep Tracker Automated Maintenance
# This script runs maintenance tasks automatically

cd "$(dirname "$0")"

# Log file
LOG_FILE="logs/cron_$(date +%Y%m%d).log"

echo "[$(date)] Starting automated maintenance..." >> "$LOG_FILE"

# Run PHP maintenance script
if [ -f "maintenance.php" ]; then
    php maintenance.php >> "$LOG_FILE" 2>&1
    echo "[$(date)] Maintenance completed" >> "$LOG_FILE"
else
    echo "[$(date)] ERROR: maintenance.php not found" >> "$LOG_FILE"
fi

# Cleanup old log files (keep last 30 days)
find logs/ -name "*.log" -mtime +30 -delete 2>/dev/null

echo "[$(date)] Automated maintenance finished" >> "$LOG_FILE"
EOF

    chmod +x "$CRON_SCRIPT"
    print_status "Created cron maintenance script: $CRON_SCRIPT"
    
    # Add to crontab
    (crontab -l 2>/dev/null; echo "0 2 * * * $(pwd)/$CRON_SCRIPT") | crontab -
    print_status "Added daily maintenance to crontab (runs at 2:00 AM)"
}

# Function to create PWA manifest
create_pwa_manifest() {
    print_step "Creating PWA manifest..."
    
    cat > "manifest.json" << EOF
{
  "name": "Sleep Tracker - Rizaldi",
  "short_name": "Sleep Tracker",
  "description": "Personal sleep management with AI predictions",
  "start_url": "/",
  "display": "standalone",
  "background_color": "#667eea",
  "theme_color": "#764ba2",
  "orientation": "portrait-primary",
  "categories": ["health", "productivity", "lifestyle"],
  "lang": "id",
  "icons": [
    {
      "src": "data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 100'><text y='.9em' font-size='90'>🌙</text></svg>",
      "sizes": "192x192",
      "type": "image/svg+xml",
      "purpose": "any maskable"
    },
    {
      "src": "data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 100'><text y='.9em' font-size='90'>🌙</text></svg>",
      "sizes": "512x512",
      "type": "image/svg+xml",
      "purpose": "any maskable"
    }
  ],
  "shortcuts": [
    {
      "name": "Start Sleep",
      "short_name": "Sleep",
      "description": "Start a new sleep session",
      "url": "/mobile.php?action=sleep",
      "icons": [{ "src": "data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 100'><text y='.9em' font-size='90'>🛏️</text></svg>", "sizes": "96x96" }]
    },
    {
      "name": "View Analytics",
      "short_name": "Analytics",
      "description": "View sleep analytics",
      "url": "/analytics.php",
      "icons": [{ "src": "data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 100'><text y='.9em' font-size='90'>📊</text></svg>", "sizes": "96x96" }]
    }
  ]
}
EOF

    print_status "Created PWA manifest.json"
}

# Function to run security hardening
security_hardening() {
    print_step "Applying security hardening..."
    
    # Create secure .htaccess if not exists
    if [ ! -f ".htaccess" ]; then
        print_status "Creating secure .htaccess file..."
        cp ".htaccess" ".htaccess.backup" 2>/dev/null || true
    fi
    
    # Set secure file permissions
    chmod 600 config.php 2>/dev/null || true
    chmod 644 .htaccess 2>/dev/null || true
    
    # Create robots.txt
    cat > "robots.txt" << 'EOF'
User-agent: *
Disallow: /data/
Disallow: /backup/
Disallow: /logs/
Disallow: /temp/
Disallow: /config.php
Disallow: /setup.php
Disallow: /maintenance.php

Sitemap: /sitemap.xml
EOF

    print_status "Applied security hardening measures"
}

# Function to verify installation
verify_installation() {
    print_step "Verifying installation..."
    
    REQUIRED_FILES=("index.php" "sleep_tracker.php" "config.php" "mobile.php" ".htaccess")
    MISSING_FILES=()
    
    for file in "${REQUIRED_FILES[@]}"; do
        if [ -f "$file" ]; then
            print_status "✓ $file exists"
        else
            MISSING_FILES+=("$file")
            print_error "✗ $file is missing"
        fi
    done
    
    # Check directories
    REQUIRED_DIRS=("data" "backup" "logs" "temp")
    for dir in "${REQUIRED_DIRS[@]}"; do
        if [ -d "$dir" ] && [ -w "$dir" ]; then
            print_status "✓ $dir directory is writable"
        else
            print_error "✗ $dir directory issue"
        fi
    done
    
    # Check PHP syntax
    if command_exists php; then
        for file in *.php; do
            if [ -f "$file" ]; then
                if php -l "$file" >/dev/null 2>&1; then
                    print_status "✓ $file syntax OK"
                else
                    print_error "✗ $file syntax error"
                fi
            fi
        done
    fi
    
    if [ ${#MISSING_FILES[@]} -eq 0 ]; then
        print_status "Installation verification completed successfully"
        return 0
    else
        print_error "Installation verification failed - missing files: ${MISSING_FILES[*]}"
        return 1
    fi
}

# Function to display post-installation instructions
show_post_install_info() {
    print_header "=== Installation Completed Successfully! ==="
    echo
    print_status "Sleep Tracker v$APP_VERSION has been installed for $APP_OWNER"
    echo
    print_header "Next Steps:"
    echo -e "${WHITE}1.${NC} Open your web browser and navigate to your installation URL"
    echo -e "${WHITE}2.${NC} Complete the setup process through the web interface"
    echo -e "${WHITE}3.${NC} Start adding your sleep data"
    echo
    print_header "Important URLs:"
    if [ -n "$DOMAIN_NAME" ]; then
        echo -e "${WHITE}Main App:${NC} http://$DOMAIN_NAME/"
        echo -e "${WHITE}Mobile:${NC} http://$DOMAIN_NAME/mobile.php"
        echo -e "${WHITE}Analytics:${NC} http://$DOMAIN_NAME/analytics.php"
        echo -e "${WHITE}API:${NC} http://$DOMAIN_NAME/api.php"
    else
        echo -e "${WHITE}Main App:${NC} http://localhost/sleep-tracker/"
        echo -e "${WHITE}Mobile:${NC} http://localhost/sleep-tracker/mobile.php"
        echo -e "${WHITE}Analytics:${NC} http://localhost/sleep-tracker/analytics.php"
        echo -e "${WHITE}API:${NC} http://localhost/sleep-tracker/api.php"
    fi
    echo
    print_header "Maintenance:"
    echo -e "${WHITE}•${NC} Automated maintenance runs daily at 2:00 AM"
    echo -e "${WHITE}•${NC} Manual maintenance: php maintenance.php"
    echo -e "${WHITE}•${NC} View logs in: logs/ directory"
    echo
    print_header "Support:"
    echo -e "${WHITE}•${NC} Documentation: README.md"
    echo -e "${WHITE}•${NC} Configuration: config.php"
    echo -e "${WHITE}•${NC} Health check: config.php"
    echo
    print_status "Installation log saved to: $LOG_FILE"
}

# Function to handle cleanup on error
cleanup_on_error() {
    print_error "Installation failed. Cleaning up..."
    
    # Remove created directories if they're empty
    for dir in data backup logs temp; do
        if [ -d "$dir" ] && [ -z "$(ls -A $dir)" ]; then
            rmdir "$dir" 2>/dev/null || true
        fi
    done
    
    print_error "Installation aborted. Check $LOG_FILE for details."
    exit 1
}

# Main installation function
main() {
    # Trap errors for cleanup
    trap cleanup_on_error ERR
    
    print_header "🌙 Sleep Tracker Installation Script"
    print_header "Version: $APP_VERSION"
    print_header "Owner: $APP_OWNER"
    print_header "Starting installation at: $(date)"
    echo
    
    # Parse command line arguments
    while [[ $# -gt 0 ]]; do
        case $1 in
            --domain)
                DOMAIN_NAME="$2"
                shift 2
                ;;
            --ssl)
                SSL_ENABLED=true
                shift
                ;;
            --web-root)
                WEB_ROOT="$2"
                shift 2
                ;;
            --sample-data)
                CREATE_SAMPLE_DATA=true
                shift
                ;;
            --help)
                echo "Usage: $0 [options]"
                echo "Options:"
                echo "  --domain DOMAIN     Set domain name for virtual host"
                echo "  --ssl              Enable SSL configuration"
                echo "  --web-root PATH    Set web root directory"
                echo "  --sample-data      Create sample data for testing"
                echo "  --help             Show this help message"
                exit 0
                ;;
            *)
                print_error "Unknown option: $1"
                exit 1
                ;;
        esac
    done
    
    # Check if running as root for some operations
    if [ "$EUID" -eq 0 ]; then
        print_warning "Running as root. This is not recommended for security reasons."
    fi
    
    # Main installation steps
    check_system_requirements
    create_directories
    set_permissions
    configure_web_server
    
    if [ "$CREATE_SAMPLE_DATA" = true ]; then
        create_sample_data
    fi
    
    setup_cron_jobs
    create_pwa_manifest
    security_hardening
    
    # Verify installation
    if verify_installation; then
        show_post_install_info
    else
        print_error "Installation verification failed"
        exit 1
    fi
    
    print_header "Installation completed successfully! 🎉"
}

# Run main function with all arguments
main "$@"