<?php
// cron.php - Automated Tasks & Scheduler for Sleep Tracker

require_once 'config.php';

class SleepTrackerCron {
    
    private $logFile;
    private $lockFile;
    private $results = [];
    
    public function __construct() {
        $this->logFile = 'logs/cron_' . date('Y-m-d') . '.log';
        $this->lockFile = 'temp/cron.lock';
        
        // Ensure we can write logs
        if (!file_exists('logs')) {
            mkdir('logs', 0755, true);
        }
        
        $this->log("Cron system initialized");
    }
    
    /**
     * Main cron execution
     */
    public function run($tasks = 'all') {
        // Check if another cron is running
        if ($this->isLocked()) {
            $this->log("Another cron process is running. Exiting.");
            return false;
        }
        
        // Create lock file
        $this->lock();
        
        try {
            $this->log("Starting cron execution - Tasks: $tasks");
            
            switch ($tasks) {
                case 'all':
                    $this->runAllTasks();
                    break;
                case 'maintenance':
                    $this->runMaintenance();
                    break;
                case 'backup':
                    $this->runBackup();
                    break;
                case 'cleanup':
                    $this->runCleanup();
                    break;
                case 'notifications':
                    $this->runNotifications();
                    break;
                case 'analytics':
                    $this->runAnalyticsUpdate();
                    break;
                default:
                    $this->log("Unknown task: $tasks");
                    return false;
            }
            
            $this->log("Cron execution completed successfully");
            return true;
            
        } catch (Exception $e) {
            $this->log("Cron execution failed: " . $e->getMessage());
            return false;
        } finally {
            // Remove lock file
            $this->unlock();
        }
    }
    
    /**
     * Run all scheduled tasks
     */
    private function runAllTasks() {
        $currentHour = (int)date('H');
        $currentMinute = (int)date('i');
        $dayOfWeek = (int)date('w');
        
        // Every 15 minutes: Quick cleanup
        if ($currentMinute % 15 === 0) {
            $this->runQuickCleanup();
        }
        
        // Every hour: Update analytics cache
        if ($currentMinute === 0) {
            $this->runAnalyticsUpdate();
        }
        
        // Daily at 2:00 AM: Full maintenance
        if ($currentHour === 2 && $currentMinute === 0) {
            $this->runMaintenance();
        }
        
        // Daily at 3:00 AM: Backup
        if ($currentHour === 3 && $currentMinute === 0) {
            $this->runBackup();
        }
        
        // Weekly on Sunday at 4:00 AM: Deep cleanup
        if ($dayOfWeek === 0 && $currentHour === 4 && $currentMinute === 0) {
            $this->runDeepCleanup();
        }
        
        // Evening notifications (8:00 PM - 11:00 PM every 30 minutes)
        if ($currentHour >= 20 && $currentHour <= 23 && $currentMinute % 30 === 0) {
            $this->runNotifications();
        }
        
        // Monthly on 1st at 1:00 AM: Archive old data
        if (date('d') === '01' && $currentHour === 1 && $currentMinute === 0) {
            $this->runArchiving();
        }
    }
    
    /**
     * Run maintenance tasks
     */
    private function runMaintenance() {
        $this->log("Starting maintenance tasks");
        
        try {
            require_once 'maintenance.php';
            $maintenance = new SleepTrackerMaintenance();
            $results = $maintenance->runAll();
            
            $successful = count(array_filter($results, function($r) { 
                return $r['status'] === 'success'; 
            }));
            $total = count($results);
            
            $this->log("Maintenance completed: $successful/$total tasks successful");
            $this->results['maintenance'] = $results;
            
        } catch (Exception $e) {
            $this->log("Maintenance failed: " . $e->getMessage());
        }
    }
    
    /**
     * Run backup tasks
     */
    private function runBackup() {
        $this->log("Starting backup tasks");
        
        try {
            $backupFile = 'backup/auto_backup_' . date('Y-m-d_H-i-s') . '.json';
            
            // Collect all data
            $allData = [];
            $dataFiles = glob('data/*.json');
            
            foreach ($dataFiles as $file) {
                $monthData = json_decode(file_get_contents($file), true) ?: [];
                $allData[basename($file, '.json')] = $monthData;
            }
            
            $backupData = [
                'created_at' => date('Y-m-d H:i:s'),
                'created_by' => 'cron_backup',
                'version' => SleepTrackerConfig::APP_VERSION,
                'owner' => SleepTrackerConfig::APP_OWNER,
                'total_months' => count($allData),
                'total_entries' => array_sum(array_map('count', $allData)),
                'data' => $allData
            ];
            
            if (file_put_contents($backupFile, json_encode($backupData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
                $this->log("Backup created successfully: " . basename($backupFile));
                
                // Keep only last 7 auto backups
                $this->cleanupOldBackups();
                
                $this->results['backup'] = [
                    'status' => 'success',
                    'file' => basename($backupFile),
                    'size' => filesize($backupFile)
                ];
            } else {
                throw new Exception("Failed to create backup file");
            }
            
        } catch (Exception $e) {
            $this->log("Backup failed: " . $e->getMessage());
            $this->results['backup'] = [
                'status' => 'error',
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Run quick cleanup
     */
    private function runQuickCleanup() {
        $this->log("Running quick cleanup");
        
        try {
            $cleaned = 0;
            
            // Clean temp files older than 1 hour
            $tempFiles = glob('temp/*');
            $cutoffTime = time() - 3600;
            
            foreach ($tempFiles as $file) {
                if (is_file($file) && filemtime($file) < $cutoffTime) {
                    if (unlink($file)) {
                        $cleaned++;
                    }
                }
            }
            
            // Clean rate limit data
            $rateLimitFile = 'temp/rate_limits.json';
            if (file_exists($rateLimitFile)) {
                $rateLimits = json_decode(file_get_contents($rateLimitFile), true) ?: [];
                $currentTime = time();
                $originalCount = count($rateLimits);
                
                $rateLimits = array_filter($rateLimits, function($timestamp) use ($currentTime) {
                    return ($currentTime - $timestamp) < 3600;
                });
                
                if (count($rateLimits) < $originalCount) {
                    file_put_contents($rateLimitFile, json_encode($rateLimits));
                    $cleaned += ($originalCount - count($rateLimits));
                }
            }
            
            $this->log("Quick cleanup completed: $cleaned items removed");
            
        } catch (Exception $e) {
            $this->log("Quick cleanup failed: " . $e->getMessage());
        }
    }
    
    /**
     * Run deep cleanup (weekly)
     */
    private function runDeepCleanup() {
        $this->log("Running deep cleanup");
        
        try {
            $results = [
                'old_logs' => $this->cleanupOldLogs(),
                'old_backups' => $this->cleanupOldBackups(),
                'temp_files' => $this->cleanupTempFiles(),
                'cache_optimization' => $this->optimizeCache()
            ];
            
            $this->log("Deep cleanup completed: " . json_encode($results));
            $this->results['deep_cleanup'] = $results;
            
        } catch (Exception $e) {
            $this->log("Deep cleanup failed: " . $e->getMessage());
        }
    }
    
    /**
     * Run notifications
     */
    private function runNotifications() {
        $this->log("Processing notifications");
        
        try {
            $notifications = $this->generateNotifications();
            
            foreach ($notifications as $notification) {
                $this->sendNotification($notification);
            }
            
            $this->log("Processed " . count($notifications) . " notifications");
            
        } catch (Exception $e) {
            $this->log("Notifications failed: " . $e->getMessage());
        }
    }
    
    /**
     * Run analytics update
     */
    private function runAnalyticsUpdate() {
        $this->log("Updating analytics cache");
        
        try {
            $cacheFile = 'temp/analytics_cache.json';
            
            // Load all data
            $allData = [];
            $dataFiles = glob('data/*.json');
            
            foreach ($dataFiles as $file) {
                $monthData = json_decode(file_get_contents($file), true) ?: [];
                $allData = array_merge($allData, $monthData);
            }
            
            // Generate analytics
            $analytics = $this->generateAnalytics($allData);
            $analytics['cached_at'] = date('Y-m-d H:i:s');
            $analytics['expires_at'] = date('Y-m-d H:i:s', strtotime('+1 hour'));
            
            if (file_put_contents($cacheFile, json_encode($analytics, JSON_PRETTY_PRINT))) {
                $this->log("Analytics cache updated successfully");
            }
            
        } catch (Exception $e) {
            $this->log("Analytics update failed: " . $e->getMessage());
        }
    }
    
    /**
     * Run archiving (monthly)
     */
    private function runArchiving() {
        $this->log("Running monthly archiving");
        
        try {
            $cutoffDate = date('Y-m-d', strtotime('-24 months'));
            $archived = 0;
            
            $dataFiles = glob('data/*.json');
            
            foreach ($dataFiles as $file) {
                $month = basename($file, '.json');
                
                if ($month < substr($cutoffDate, 0, 7)) {
                    // Create archive directory
                    if (!file_exists('archive')) {
                        mkdir('archive', 0755, true);
                    }
                    
                    $archiveFile = "archive/" . basename($file);
                    
                    if (rename($file, $archiveFile)) {
                        $archived++;
                        $this->log("Archived: " . basename($file));
                    }
                }
            }
            
            $this->log("Archiving completed: $archived files archived");
            
        } catch (Exception $e) {
            $this->log("Archiving failed: " . $e->getMessage());
        }
    }
    
    /**
     * Generate notifications based on sleep patterns
     */
    private function generateNotifications() {
        $notifications = [];
        
        // Load recent data to analyze patterns
        $currentMonth = date('Y-m');
        $dataFile = "data/{$currentMonth}.json";
        
        if (!file_exists($dataFile)) {
            return $notifications;
        }
        
        $monthData = json_decode(file_get_contents($dataFile), true) ?: [];
        
        // Check if no sleep recorded today
        $today = date('Y-m-d');
        $todayData = null;
        
        foreach ($monthData as $entry) {
            if ($entry['date'] === $today) {
                $todayData = $entry;
                break;
            }
        }
        
        $currentHour = (int)date('H');
        
        // Evening reminder if no sleep data today and it's after 9 PM
        if (!$todayData && $currentHour >= 21) {
            $notifications[] = [
                'type' => 'sleep_reminder',
                'title' => 'Sleep Tracker Reminder',
                'message' => 'Hey Rizaldi! Don\'t forget to track your sleep tonight 🌙',
                'priority' => 'normal'
            ];
        }
        
        // Late night warning if it's after midnight and no sleep recorded
        if (!$todayData && $currentHour >= 0 && $currentHour <= 2) {
            $notifications[] = [
                'type' => 'late_night_warning',
                'title' => 'Sleep Tracker Alert',
                'message' => 'It\'s getting late! Consider starting your sleep session 😴',
                'priority' => 'high'
            ];
        }
        
        // Weekly summary (Sunday evening)
        if (date('w') === '0' && $currentHour === 20) {
            $weeklyStats = $this->calculateWeeklyStats($monthData);
            if ($weeklyStats) {
                $notifications[] = [
                    'type' => 'weekly_summary',
                    'title' => 'Weekly Sleep Summary',
                    'message' => "This week: {$weeklyStats['avg_duration']}h avg sleep, {$weeklyStats['total_sessions']} sessions",
                    'priority' => 'normal'
                ];
            }
        }
        
        return $notifications;
    }
    
    /**
     * Send notification (placeholder for actual notification system)
     */
    private function sendNotification($notification) {
        // In a real implementation, this would send push notifications
        // For now, we'll log them
        
        $this->log("NOTIFICATION [{$notification['type']}]: {$notification['message']}");
        
        // Could integrate with:
        // - Push notification services
        // - Email notifications
        // - SMS services
        // - Slack/Discord webhooks
        // - etc.
    }
    
    /**
     * Calculate weekly statistics
     */
    private function calculateWeeklyStats($monthData) {
        $weekAgo = date('Y-m-d', strtotime('-7 days'));
        $today = date('Y-m-d');
        
        $weekData = array_filter($monthData, function($entry) use ($weekAgo, $today) {
            return $entry['date'] >= $weekAgo && $entry['date'] <= $today;
        });
        
        if (empty($weekData)) {
            return null;
        }
        
        $totalDuration = 0;
        $totalSessions = 0;
        
        foreach ($weekData as $entry) {
            if (!empty($entry['sleepSessions'])) {
                foreach ($entry['sleepSessions'] as $session) {
                    $totalDuration += $session['duration'];
                    $totalSessions++;
                }
            }
        }
        
        return [
            'avg_duration' => $totalSessions > 0 ? round($totalDuration / $totalSessions, 1) : 0,
            'total_sessions' => $totalSessions,
            'total_days' => count($weekData)
        ];
    }
    
    /**
     * Generate analytics data
     */
    private function generateAnalytics($allData) {
        $analytics = [
            'total_entries' => count($allData),
            'total_sessions' => 0,
            'total_duration' => 0,
            'avg_duration' => 0,
            'quality_distribution' => [],
            'type_distribution' => [],
            'recent_trend' => []
        ];
        
        foreach ($allData as $entry) {
            if (!empty($entry['sleepSessions'])) {
                foreach ($entry['sleepSessions'] as $session) {
                    $analytics['total_sessions']++;
                    $analytics['total_duration'] += $session['duration'];
                    
                    $quality = $session['quality'];
                    $type = $session['type'];
                    
                    $analytics['quality_distribution'][$quality] = 
                        ($analytics['quality_distribution'][$quality] ?? 0) + 1;
                    
                    $analytics['type_distribution'][$type] = 
                        ($analytics['type_distribution'][$type] ?? 0) + 1;
                }
            }
        }
        
        if ($analytics['total_sessions'] > 0) {
            $analytics['avg_duration'] = round($analytics['total_duration'] / $analytics['total_sessions'], 2);
        }
        
        // Recent 7-day trend
        $recent = array_slice($allData, -7);
        foreach ($recent as $entry) {
            $dayTotal = 0;
            if (!empty($entry['sleepSessions'])) {
                foreach ($entry['sleepSessions'] as $session) {
                    $dayTotal += $session['duration'];
                }
            }
            $analytics['recent_trend'][] = [
                'date' => $entry['date'],
                'duration' => $dayTotal
            ];
        }
        
        return $analytics;
    }
    
    /**
     * Cleanup old log files
     */
    private function cleanupOldLogs() {
        $logFiles = glob('logs/*.log');
        $cutoffTime = time() - (30 * 24 * 3600); // 30 days
        $cleaned = 0;
        
        foreach ($logFiles as $file) {
            if (filemtime($file) < $cutoffTime) {
                if (unlink($file)) {
                    $cleaned++;
                }
            }
        }
        
        return $cleaned;
    }
    
    /**
     * Cleanup old backup files
     */
    private function cleanupOldBackups() {
        $backupFiles = glob('backup/auto_backup_*.json');
        
        // Keep only last 7 auto backups
        if (count($backupFiles) > 7) {
            // Sort by modification time
            usort($backupFiles, function($a, $b) {
                return filemtime($a) - filemtime($b);
            });
            
            $toDelete = array_slice($backupFiles, 0, -7);
            $cleaned = 0;
            
            foreach ($toDelete as $file) {
                if (unlink($file)) {
                    $cleaned++;
                }
            }
            
            return $cleaned;
        }
        
        return 0;
    }
    
    /**
     * Cleanup temporary files
     */
    private function cleanupTempFiles() {
        $tempFiles = glob('temp/*');
        $cutoffTime = time() - 3600; // 1 hour
        $cleaned = 0;
        
        foreach ($tempFiles as $file) {
            if (is_file($file) && filemtime($file) < $cutoffTime) {
                if (unlink($file)) {
                    $cleaned++;
                }
            }
        }
        
        return $cleaned;
    }
    
    /**
     * Optimize cache files
     */
    private function optimizeCache() {
        $cacheFiles = glob('temp/*_cache.json');
        $optimized = 0;
        
        foreach ($cacheFiles as $file) {
            $data = json_decode(file_get_contents($file), true);
            if ($data) {
                // Re-encode to optimize format
                if (file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT))) {
                    $optimized++;
                }
            }
        }
        
        return $optimized;
    }
    
    /**
     * Check if cron is locked
     */
    private function isLocked() {
        if (!file_exists($this->lockFile)) {
            return false;
        }
        
        $lockTime = filemtime($this->lockFile);
        $currentTime = time();
        
        // Consider lock stale after 1 hour
        if (($currentTime - $lockTime) > 3600) {
            unlink($this->lockFile);
            return false;
        }
        
        return true;
    }
    
    /**
     * Create lock file
     */
    private function lock() {
        file_put_contents($this->lockFile, getmypid() . "\n" . date('Y-m-d H:i:s'));
    }
    
    /**
     * Remove lock file
     */
    private function unlock() {
        if (file_exists($this->lockFile)) {
            unlink($this->lockFile);
        }
    }
    
    /**
     * Log message
     */
    private function log($message) {
        $timestamp = date('Y-m-d H:i:s');
        $logMessage = "[$timestamp] $message" . PHP_EOL;
        file_put_contents($this->logFile, $logMessage, FILE_APPEND | LOCK_EX);
        
        // Also output to stdout if running in CLI
        if (php_sapi_name() === 'cli') {
            echo $logMessage;
        }
    }
    
    /**
     * Get results
     */
    public function getResults() {
        return $this->results;
    }
}

// CLI execution
if (php_sapi_name() === 'cli') {
    $tasks = $argv[1] ?? 'all';
    
    echo "Sleep Tracker Cron System\n";
    echo "=========================\n";
    echo "Starting cron execution: $tasks\n\n";
    
    $cron = new SleepTrackerCron();
    $success = $cron->run($tasks);
    
    if ($success) {
        echo "\nCron execution completed successfully!\n";
        exit(0);
    } else {
        echo "\nCron execution failed!\n";
        exit(1);
    }
}

// Web interface for manual execution
if (basename($_SERVER['PHP_SELF']) === 'cron.php' && $_SERVER['REQUEST_METHOD'] === 'GET') {
    ?>
    <!DOCTYPE html>
    <html lang="id">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Cron Management - Sleep Tracker</title>
        <style>
            body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
            .container { max-width: 800px; margin: 0 auto; background: white; padding: 20px; border-radius: 10px; }
            .btn { background: #667eea; color: white; padding: 10px 20px; border: none; border-radius: 5px; margin: 5px; cursor: pointer; }
            .btn:hover { background: #5a6fd8; }
            .result { background: #f8f9fa; padding: 15px; margin: 10px 0; border-radius: 5px; border-left: 4px solid #667eea; }
            .success { border-left-color: #28a745; }
            .error { border-left-color: #dc3545; }
            h1, h2 { color: #333; }
        </style>
    </head>
    <body>
        <div class="container">
            <h1>🕐 Cron Management</h1>
            <p>Manual execution of automated tasks</p>
            
            <h2>Available Tasks</h2>
            <button class="btn" onclick="runTask('all')">Run All Tasks</button>
            <button class="btn" onclick="runTask('maintenance')">Run Maintenance</button>
            <button class="btn" onclick="runTask('backup')">Run Backup</button>
            <button class="btn" onclick="runTask('cleanup')">Run Cleanup</button>
            <button class="btn" onclick="runTask('notifications')">Run Notifications</button>
            <button class="btn" onclick="runTask('analytics')">Update Analytics</button>
            
            <div id="results"></div>
        </div>
        
        <script>
            async function runTask(task) {
                const resultsDiv = document.getElementById('results');
                resultsDiv.innerHTML = '<div class="result">Running task: ' + task + '...</div>';
                
                try {
                    const response = await fetch('cron.php?action=run&task=' + task);
                    const result = await response.text();
                    
                    resultsDiv.innerHTML = '<div class="result success"><h3>Task Completed: ' + task + '</h3><pre>' + result + '</pre></div>';
                } catch (error) {
                    resultsDiv.innerHTML = '<div class="result error"><h3>Task Failed: ' + task + '</h3><p>' + error.message + '</p></div>';
                }
            }
        </script>
    </body>
    </html>
    <?php
}

// Handle AJAX requests
if (isset($_GET['action']) && $_GET['action'] === 'run') {
    $task = $_GET['task'] ?? 'all';
    
    header('Content-Type: text/plain');
    
    $cron = new SleepTrackerCron();
    $success = $cron->run($task);
    
    if ($success) {
        echo "Cron task '$task' completed successfully\n";
        $results = $cron->getResults();
        if (!empty($results)) {
            echo "\nResults:\n";
            echo json_encode($results, JSON_PRETTY_PRINT);
        }
    } else {
        echo "Cron task '$task' failed\n";
    }
    exit;
}
?>