<?php
// config.php - Sleep Tracker Configuration System

// Prevent direct access
if (!defined('SLEEP_TRACKER_ACCESS')) {
    define('SLEEP_TRACKER_ACCESS', true);
}

// Application Configuration
class SleepTrackerConfig {
    
    // Application Settings
    const APP_NAME = 'Sleep Tracker';
    const APP_VERSION = '2.0';
    const APP_OWNER = 'Rizaldi';
    const APP_DESCRIPTION = 'Aplikasi Pengelola Jam Tidur Pribadi dengan Prediksi Cerdas';
    
    // File & Directory Settings
    const DATA_DIR = 'data';
    const BACKUP_DIR = 'backup';
    const TEMP_DIR = 'temp';
    const LOG_DIR = 'logs';
    
    // Data Settings
    const MAX_BACKUP_FILES_PER_MONTH = 10;
    const AUTO_BACKUP_INTERVAL = 300; // seconds (5 minutes)
    const DATA_RETENTION_MONTHS = 24; // Keep data for 24 months
    const MAX_SESSIONS_PER_DAY = 10;
    const MIN_SLEEP_DURATION = 0.5; // hours
    const MAX_SLEEP_DURATION = 15; // hours
    
    // AI Prediction Settings
    const MIN_DATA_FOR_PREDICTION = 3; // minimum days of data needed
    const PREDICTION_DAYS = 3; // how many days to predict
    const LEARNING_WEIGHT_RECENT = 2; // weight for recent data
    const CONFIDENCE_BASE = 40; // base confidence percentage
    const CONFIDENCE_INCREMENT = 8; // increment per data point
    const MAX_CONFIDENCE = 95; // maximum confidence percentage
    
    // Sleep Quality Thresholds
    const SLEEP_CYCLES = [
        'excellent' => ['min' => 5, 'max' => 6],
        'good' => ['min' => 4, 'max' => 7],
        'fair' => ['min' => 3, 'max' => 8],
        'poor' => ['min' => 0, 'max' => 15]
    ];
    
    // Sleep Type Time Ranges (24-hour format)
    const SLEEP_TYPE_RANGES = [
        'pagi' => [4, 5, 6, 7, 8, 9, 10],
        'siang' => [11, 12, 13, 14, 15],
        'sore' => [16, 17, 18, 19],
        'malam' => [20, 21, 22, 23, 0, 1, 2, 3]
    ];
    
    // Activity Impact on Sleep Need (additional hours)
    const ACTIVITY_IMPACT = [
        'ringan' => 0,
        'sedang' => 0.2,
        'berat' => 0.5
    ];
    
    // Location Impact on Sleep Need (additional hours)
    const LOCATION_IMPACT = [
        'rumah' => 0,
        'keluar' => 0.3,
        'jauh' => 1.0
    ];
    
    // Weekend Adjustments
    const WEEKEND_BED_TIME_DELAY = 30; // minutes later
    const WEEKEND_SLEEP_BONUS = 0.5; // additional hours
    
    // Security Settings
    const SESSION_TIMEOUT = 3600; // 1 hour
    const MAX_LOGIN_ATTEMPTS = 5;
    const RATE_LIMIT_REQUESTS = 100; // per hour
    
    // Display Settings
    const TIMEZONE = 'Asia/Jakarta';
    const DATE_FORMAT = 'Y-m-d';
    const TIME_FORMAT = 'H:i';
    const DATETIME_FORMAT = 'Y-m-d H:i:s';
    const LOCALE = 'id_ID';
    
    // Export Settings
    const EXPORT_FORMATS = ['csv', 'json', 'summary', 'backup'];
    const MAX_EXPORT_SIZE = 50000000; // 50MB
    const EXPORT_BATCH_SIZE = 1000; // records per batch
    
    // Analytics Settings
    const ANALYTICS_ENABLED = true;
    const CHART_COLORS = [
        'primary' => '#667eea',
        'secondary' => '#764ba2',
        'accent' => '#a8d8f0',
        'success' => '#28a745',
        'warning' => '#ffc107',
        'danger' => '#dc3545',
        'info' => '#17a2b8'
    ];
    
    // Notification Settings
    const NOTIFICATIONS_ENABLED = true;
    const NOTIFICATION_DURATION = 3000; // milliseconds
    
    private static $customConfig = [];
    
    /**
     * Initialize configuration
     */
    public static function init() {
        // Set timezone
        date_default_timezone_set(self::TIMEZONE);
        
        // Create necessary directories
        self::createDirectories();
        
        // Load custom configuration if exists
        self::loadCustomConfig();
        
        // Setup error handling
        self::setupErrorHandling();
    }
    
    /**
     * Create necessary directories
     */
    private static function createDirectories() {
        $directories = [
            self::DATA_DIR,
            self::BACKUP_DIR,
            self::TEMP_DIR,
            self::LOG_DIR
        ];
        
        foreach ($directories as $dir) {
            if (!file_exists($dir)) {
                mkdir($dir, 0755, true);
                
                // Create .htaccess for security
                file_put_contents($dir . '/.htaccess', "Order deny,allow\nDeny from all\n");
            }
        }
    }
    
    /**
     * Load custom configuration from file
     */
    private static function loadCustomConfig() {
        $configFile = 'config_custom.json';
        if (file_exists($configFile)) {
            $customConfig = json_decode(file_get_contents($configFile), true);
            if (is_array($customConfig)) {
                self::$customConfig = $customConfig;
            }
        }
    }
    
    /**
     * Setup error handling and logging
     */
    private static function setupErrorHandling() {
        // Custom error handler
        set_error_handler([__CLASS__, 'errorHandler']);
        
        // Custom exception handler
        set_exception_handler([__CLASS__, 'exceptionHandler']);
        
        // Register shutdown function for fatal errors
        register_shutdown_function([__CLASS__, 'shutdownHandler']);
    }
    
    /**
     * Get configuration value
     */
    public static function get($key, $default = null) {
        // Check custom config first
        if (isset(self::$customConfig[$key])) {
            return self::$customConfig[$key];
        }
        
        // Check class constants
        $constant = 'self::' . strtoupper($key);
        if (defined($constant)) {
            return constant($constant);
        }
        
        return $default;
    }
    
    /**
     * Set custom configuration value
     */
    public static function set($key, $value) {
        self::$customConfig[$key] = $value;
        self::saveCustomConfig();
    }
    
    /**
     * Save custom configuration to file
     */
    private static function saveCustomConfig() {
        $configFile = 'config_custom.json';
        file_put_contents($configFile, json_encode(self::$customConfig, JSON_PRETTY_PRINT));
    }
    
    /**
     * Get sleep quality based on cycles
     */
    public static function getSleepQuality($cycles, $type = 'malam') {
        if ($type === 'malam') {
            if ($cycles >= 5 && $cycles <= 6) return 'Excellent';
            if ($cycles >= 4 && $cycles <= 7) return 'Good';
            if ($cycles >= 3 && $cycles <= 8) return 'Fair';
            return 'Poor';
        } else {
            // For naps (siang, pagi, sore)
            if ($cycles >= 1 && $cycles <= 2) return 'Excellent';
            if ($cycles >= 1 && $cycles <= 3) return 'Good';
            return 'Fair';
        }
    }
    
    /**
     * Validate sleep session data
     */
    public static function validateSleepSession($data) {
        $errors = [];
        
        // Validate required fields
        $required = ['type', 'bedTime', 'wakeTime'];
        foreach ($required as $field) {
            if (empty($data[$field])) {
                $errors[] = "Field {$field} is required";
            }
        }
        
        if (!empty($errors)) {
            return $errors;
        }
        
        // Validate sleep type
        if (!array_key_exists($data['type'], self::SLEEP_TYPE_RANGES)) {
            $errors[] = "Invalid sleep type: {$data['type']}";
        }
        
        // Validate time format
        if (!preg_match('/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/', $data['bedTime'])) {
            $errors[] = "Invalid bed time format";
        }
        
        if (!preg_match('/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/', $data['wakeTime'])) {
            $errors[] = "Invalid wake time format";
        }
        
        // Validate sleep type vs time
        $hour = (int)substr($data['bedTime'], 0, 2);
        if (!in_array($hour, self::SLEEP_TYPE_RANGES[$data['type']])) {
            $errors[] = "Bed time doesn't match sleep type {$data['type']}";
        }
        
        // Calculate and validate duration
        $duration = self::calculateSleepDuration($data['bedTime'], $data['wakeTime']);
        
        if ($duration < self::MIN_SLEEP_DURATION) {
            $errors[] = "Sleep duration too short (minimum " . self::MIN_SLEEP_DURATION . "h)";
        }
        
        if ($duration > self::MAX_SLEEP_DURATION) {
            $errors[] = "Sleep duration too long (maximum " . self::MAX_SLEEP_DURATION . "h)";
        }
        
        return $errors;
    }
    
    /**
     * Calculate sleep duration between two times
     */
    public static function calculateSleepDuration($bedTime, $wakeTime) {
        $bedDateTime = new DateTime("2000-01-01 $bedTime");
        $wakeDateTime = new DateTime("2000-01-01 $wakeTime");
        
        // Handle next day wake time
        if ($wakeDateTime <= $bedDateTime) {
            $wakeDateTime->add(new DateInterval('P1D'));
        }
        
        $interval = $bedDateTime->diff($wakeDateTime);
        return round($interval->h + ($interval->i / 60), 1);
    }
    
    /**
     * Log message to file
     */
    public static function log($message, $level = 'INFO') {
        $logFile = self::LOG_DIR . '/app_' . date('Y-m-d') . '.log';
        $timestamp = date(self::DATETIME_FORMAT);
        $logMessage = "[{$timestamp}] [{$level}] {$message}" . PHP_EOL;
        
        file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
        
        // Keep only last 30 days of logs
        self::cleanupOldLogs();
    }
    
    /**
     * Clean up old log files
     */
    private static function cleanupOldLogs() {
        $logFiles = glob(self::LOG_DIR . '/app_*.log');
        $cutoffDate = strtotime('-30 days');
        
        foreach ($logFiles as $logFile) {
            if (filemtime($logFile) < $cutoffDate) {
                unlink($logFile);
            }
        }
    }
    
    /**
     * Error handler
     */
    public static function errorHandler($severity, $message, $file, $line) {
        $errorMessage = "Error: {$message} in {$file} on line {$line}";
        self::log($errorMessage, 'ERROR');
        
        // Don't execute PHP internal error handler
        return true;
    }
    
    /**
     * Exception handler
     */
    public static function exceptionHandler($exception) {
        $errorMessage = "Uncaught exception: " . $exception->getMessage() . 
                       " in " . $exception->getFile() . 
                       " on line " . $exception->getLine();
        self::log($errorMessage, 'FATAL');
    }
    
    /**
     * Shutdown handler for fatal errors
     */
    public static function shutdownHandler() {
        $error = error_get_last();
        if ($error && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
            $errorMessage = "Fatal error: {$error['message']} in {$error['file']} on line {$error['line']}";
            self::log($errorMessage, 'FATAL');
        }
    }
    
    /**
     * Get system information
     */
    public static function getSystemInfo() {
        return [
            'app_name' => self::APP_NAME,
            'app_version' => self::APP_VERSION,
            'app_owner' => self::APP_OWNER,
            'php_version' => phpversion(),
            'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
            'timezone' => self::TIMEZONE,
            'memory_limit' => ini_get('memory_limit'),
            'max_execution_time' => ini_get('max_execution_time'),
            'upload_max_filesize' => ini_get('upload_max_filesize'),
            'data_directory' => self::DATA_DIR,
            'backup_directory' => self::BACKUP_DIR,
            'writable_data' => is_writable(self::DATA_DIR),
            'writable_backup' => is_writable(self::BACKUP_DIR)
        ];
    }
    
    /**
     * Check system health
     */
    public static function healthCheck() {
        $checks = [
            'directories_exist' => file_exists(self::DATA_DIR) && file_exists(self::BACKUP_DIR),
            'directories_writable' => is_writable(self::DATA_DIR) && is_writable(self::BACKUP_DIR),
            'php_version_ok' => version_compare(phpversion(), '7.0', '>='),
            'json_support' => function_exists('json_encode') && function_exists('json_decode'),
            'file_functions' => function_exists('file_get_contents') && function_exists('file_put_contents'),
            'date_functions' => class_exists('DateTime') && function_exists('date'),
            'memory_adequate' => (int)ini_get('memory_limit') >= 128 || ini_get('memory_limit') == -1
        ];
        
        $allPassed = true;
        foreach ($checks as $check => $result) {
            if (!$result) {
                $allPassed = false;
                self::log("Health check failed: {$check}", 'WARNING');
            }
        }
        
        return [
            'status' => $allPassed ? 'healthy' : 'issues_detected',
            'checks' => $checks,
            'timestamp' => date(self::DATETIME_FORMAT)
        ];
    }
    
    /**
     * Get application statistics
     */
    public static function getAppStats() {
        $dataFiles = glob(self::DATA_DIR . '/*.json');
        $backupFiles = glob(self::BACKUP_DIR . '/*.json');
        $logFiles = glob(self::LOG_DIR . '/*.log');
        
        $totalSessions = 0;
        $totalDays = 0;
        
        foreach ($dataFiles as $file) {
            $data = json_decode(file_get_contents($file), true) ?: [];
            $totalDays += count($data);
            
            foreach ($data as $entry) {
                if (isset($entry['sleepSessions'])) {
                    $totalSessions += count($entry['sleepSessions']);
                }
            }
        }
        
        return [
            'total_data_files' => count($dataFiles),
            'total_backup_files' => count($backupFiles),
            'total_log_files' => count($logFiles),
            'total_days_recorded' => $totalDays,
            'total_sleep_sessions' => $totalSessions,
            'data_directory_size' => self::getDirectorySize(self::DATA_DIR),
            'backup_directory_size' => self::getDirectorySize(self::BACKUP_DIR),
            'oldest_record' => self::getOldestRecord(),
            'newest_record' => self::getNewestRecord()
        ];
    }
    
    /**
     * Get directory size in bytes
     */
    private static function getDirectorySize($directory) {
        $size = 0;
        foreach (new RecursiveIteratorIterator(new RecursiveDirectoryIterator($directory)) as $file) {
            $size += $file->getSize();
        }
        return $size;
    }
    
    /**
     * Get oldest record date
     */
    private static function getOldestRecord() {
        $dataFiles = glob(self::DATA_DIR . '/*.json');
        $oldest = null;
        
        foreach ($dataFiles as $file) {
            $data = json_decode(file_get_contents($file), true) ?: [];
            foreach ($data as $entry) {
                if (!$oldest || $entry['date'] < $oldest) {
                    $oldest = $entry['date'];
                }
            }
        }
        
        return $oldest;
    }
    
    /**
     * Get newest record date
     */
    private static function getNewestRecord() {
        $dataFiles = glob(self::DATA_DIR . '/*.json');
        $newest = null;
        
        foreach ($dataFiles as $file) {
            $data = json_decode(file_get_contents($file), true) ?: [];
            foreach ($data as $entry) {
                if (!$newest || $entry['date'] > $newest) {
                    $newest = $entry['date'];
                }
            }
        }
        
        return $newest;
    }
}

// Initialize configuration
SleepTrackerConfig::init();

// Helper functions for backward compatibility
if (!function_exists('sleepConfig')) {
    function sleepConfig($key, $default = null) {
        return SleepTrackerConfig::get($key, $default);
    }
}

if (!function_exists('sleepLog')) {
    function sleepLog($message, $level = 'INFO') {
        SleepTrackerConfig::log($message, $level);
    }
}

if (!function_exists('sleepValidate')) {
    function sleepValidate($data) {
        return SleepTrackerConfig::validateSleepSession($data);
    }
}

if (!function_exists('sleepDuration')) {
    function sleepDuration($bedTime, $wakeTime) {
        return SleepTrackerConfig::calculateSleepDuration($bedTime, $wakeTime);
    }
}

if (!function_exists('sleepQuality')) {
    function sleepQuality($cycles, $type = 'malam') {
        return SleepTrackerConfig::getSleepQuality($cycles, $type);
    }
}

?>

<!-- Configuration Interface (only accessible when called directly) -->
<?php if (basename($_SERVER['PHP_SELF']) === 'config.php'): ?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Configuration - Sleep Tracker</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
            margin: 0;
        }

        .container {
            max-width: 1000px;
            margin: 0 auto;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 20px;
            padding: 30px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
        }

        .header {
            text-align: center;
            margin-bottom: 30px;
        }

        .header h1 {
            color: #667eea;
            margin-bottom: 10px;
        }

        .config-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 25px;
        }

        .config-card {
            background: white;
            border-radius: 15px;
            padding: 20px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }

        .config-card h3 {
            color: #667eea;
            margin-bottom: 15px;
            border-bottom: 2px solid #f0f0f0;
            padding-bottom: 8px;
        }

        .config-item {
            margin-bottom: 10px;
            padding: 8px;
            background: #f8f9fa;
            border-radius: 5px;
            font-family: monospace;
            font-size: 0.9em;
        }

        .status-indicator {
            display: inline-block;
            width: 10px;
            height: 10px;
            border-radius: 50%;
            margin-right: 8px;
        }

        .status-ok { background: #28a745; }
        .status-warning { background: #ffc107; }
        .status-error { background: #dc3545; }

        .nav-links {
            text-align: center;
            margin-bottom: 20px;
        }

        .nav-links a {
            display: inline-block;
            padding: 8px 15px;
            margin: 0 5px;
            background: #667eea;
            color: white;
            text-decoration: none;
            border-radius: 20px;
            font-size: 0.9em;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1><i class="fas fa-cog"></i> Sleep Tracker Configuration</h1>
            <p>System Configuration & Health Status</p>
        </div>

        <div class="nav-links">
            <a href="index.php"><i class="fas fa-home"></i> Dashboard</a>
            <a href="analytics.php"><i class="fas fa-chart-bar"></i> Analytics</a>
            <a href="export.php"><i class="fas fa-download"></i> Export</a>
        </div>

        <div class="config-grid">
            <!-- System Information -->
            <div class="config-card">
                <h3><i class="fas fa-info-circle"></i> System Information</h3>
                <?php
                $sysInfo = SleepTrackerConfig::getSystemInfo();
                foreach ($sysInfo as $key => $value) {
                    echo "<div class='config-item'><strong>" . ucwords(str_replace('_', ' ', $key)) . ":</strong> " . 
                         (is_bool($value) ? ($value ? 'Yes' : 'No') : $value) . "</div>";
                }
                ?>
            </div>

            <!-- Health Check -->
            <div class="config-card">
                <h3><i class="fas fa-heartbeat"></i> Health Check</h3>
                <?php
                $health = SleepTrackerConfig::healthCheck();
                echo "<div class='config-item'><strong>Overall Status:</strong> " . 
                     "<span class='status-indicator " . ($health['status'] === 'healthy' ? 'status-ok' : 'status-warning') . "'></span>" .
                     ucfirst(str_replace('_', ' ', $health['status'])) . "</div>";
                
                foreach ($health['checks'] as $check => $result) {
                    echo "<div class='config-item'>" .
                         "<span class='status-indicator " . ($result ? 'status-ok' : 'status-error') . "'></span>" .
                         ucwords(str_replace('_', ' ', $check)) . "</div>";
                }
                ?>
            </div>

            <!-- Application Statistics -->
            <div class="config-card">
                <h3><i class="fas fa-chart-pie"></i> Application Statistics</h3>
                <?php
                $stats = SleepTrackerConfig::getAppStats();
                foreach ($stats as $key => $value) {
                    if (strpos($key, 'size') !== false) {
                        $value = number_format($value / 1024, 1) . ' KB';
                    }
                    echo "<div class='config-item'><strong>" . ucwords(str_replace('_', ' ', $key)) . ":</strong> {$value}</div>";
                }
                ?>
            </div>

            <!-- Configuration Values -->
            <div class="config-card">
                <h3><i class="fas fa-sliders-h"></i> Configuration Values</h3>
                <div class='config-item'><strong>Min Sleep Duration:</strong> <?php echo SleepTrackerConfig::MIN_SLEEP_DURATION; ?>h</div>
                <div class='config-item'><strong>Max Sleep Duration:</strong> <?php echo SleepTrackerConfig::MAX_SLEEP_DURATION; ?>h</div>
                <div class='config-item'><strong>Prediction Days:</strong> <?php echo SleepTrackerConfig::PREDICTION_DAYS; ?></div>
                <div class='config-item'><strong>Max Confidence:</strong> <?php echo SleepTrackerConfig::MAX_CONFIDENCE; ?>%</div>
                <div class='config-item'><strong>Auto Backup:</strong> <?php echo SleepTrackerConfig::AUTO_BACKUP_INTERVAL; ?>s</div>
                <div class='config-item'><strong>Data Retention:</strong> <?php echo SleepTrackerConfig::DATA_RETENTION_MONTHS; ?> months</div>
            </div>
        </div>
    </div>
</body>
</html>
<?php endif; ?>