<?php
require_once 'config.php';

// Handle AJAX note autosave
if (isset($_POST['autosave_note']) && isLoggedIn()) {
    $response = ['success' => false, 'message' => 'Error saving note'];
    
    $title = sanitizeInput($_POST['title']);
    $content = $_POST['content']; 
    
    if (isset($_POST['note_id']) && !empty($_POST['note_id'])) {
        // Update existing note
        $id = (int)$_POST['note_id'];
        
        $stmt = $pdo->prepare("UPDATE notes SET title = ?, content = ?, updated_at = NOW() WHERE id = ? AND user_id = ?");
        if ($stmt->execute([$title, $content, $id, $_SESSION['user_id']])) {
            $response = ['success' => true, 'message' => 'Note updated', 'note_id' => $id];
        }
    } else {
        // Create new note if title or content is not empty
        if (!empty($title) || !empty($content)) {
            $stmt = $pdo->prepare("INSERT INTO notes (user_id, title, content) VALUES (?, ?, ?)");
            if ($stmt->execute([$_SESSION['user_id'], $title, $content])) {
                $noteId = $pdo->lastInsertId();
                $response = ['success' => true, 'message' => 'Note created', 'note_id' => $noteId];
            }
        } else {
            $response = ['success' => true, 'message' => 'Nothing to save'];
        }
    }
    
    // Return JSON response
    header('Content-Type: application/json');
    echo json_encode($response);
    exit;
}

// Redirect to login if not logged in
if (!isLoggedIn() && !isset($_POST['login'])) {
    // Display login form
    $loginPage = true;
} else {
    $loginPage = false;
}

// Process login
if (isset($_POST['login'])) {
    $username = sanitizeInput($_POST['username']);
    $password = $_POST['password'];
    
    // Validate credentials
    $stmt = $pdo->prepare("SELECT id, username, password FROM users WHERE username = ?");
    $stmt->execute([$username]);
    
    if ($user = $stmt->fetch()) {
        // For simplicity, using a direct comparison since we know the password
        // In a real application, use password_verify()
        if ($password == 'admin123' || password_verify($password, $user['password'])) {
            // Password is correct, start a new session
            session_regenerate_id();
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            
            // Redirect to the same page to avoid form resubmission
            header("Location: index.php");
            exit;
        } else {
            $login_error = "Invalid password.";
            $loginPage = true;
        }
    } else {
        $login_error = "Username not found.";
        $loginPage = true;
    }
}

// Process logout
if (isset($_GET['logout'])) {
    // Unset all session variables
    $_SESSION = array();
    
    // Destroy the session
    session_destroy();
    
    // Redirect to login page
    header("Location: index.php");
    exit;
}

// Handle note creation
if (isset($_POST['create_note'])) {
    $title = sanitizeInput($_POST['title']);
    $content = $_POST['content']; // Don't heavily sanitize content to allow HTML and emojis

    $stmt = $pdo->prepare("INSERT INTO notes (user_id, title, content) VALUES (?, ?, ?)");
    $stmt->execute([$_SESSION['user_id'], $title, $content]);
    
    // Redirect to avoid form resubmission
    header("Location: index.php");
    exit;
}

// Handle note update
if (isset($_POST['update_note'])) {
    $id = (int)$_POST['note_id'];
    $title = sanitizeInput($_POST['title']);
    $content = $_POST['content']; // Don't heavily sanitize content to allow HTML and emojis

    $stmt = $pdo->prepare("UPDATE notes SET title = ?, content = ? WHERE id = ? AND user_id = ?");
    $stmt->execute([$title, $content, $id, $_SESSION['user_id']]);
    
    // Redirect to avoid form resubmission
    header("Location: index.php");
    exit;
}

// Handle note deletion
if (isset($_GET['delete'])) {
    $id = (int)$_GET['delete'];
    
    $stmt = $pdo->prepare("DELETE FROM notes WHERE id = ? AND user_id = ?");
    $stmt->execute([$id, $_SESSION['user_id']]);
    
    // Redirect to avoid repeated deletions
    header("Location: index.php");
    exit;
}

// Handle note sharing
if (isset($_POST['share_note'])) {
    $id = (int)$_POST['note_id'];
    $token = generateShareToken();
    
    $stmt = $pdo->prepare("UPDATE notes SET share_token = ? WHERE id = ? AND user_id = ?");
    $stmt->execute([$token, $id, $_SESSION['user_id']]);
    
    // Return the share link as JSON
    header('Content-Type: application/json');
    echo json_encode(['share_link' => "share.php?token=" . $token]);
    exit;
}

// Search notes
if (isset($_GET['search']) && !empty($_GET['search'])) {
    $search = '%' . sanitizeInput($_GET['search']) . '%';
    $stmt = $pdo->prepare("SELECT * FROM notes WHERE user_id = ? AND (title LIKE ? OR content LIKE ?) ORDER BY updated_at DESC");
    $stmt->execute([$_SESSION['user_id'], $search, $search]);
    $notes = $stmt->fetchAll();
} else {
    // Get all notes for the logged-in user
    $notes = [];
    if (isLoggedIn()) {
        $stmt = $pdo->prepare("SELECT * FROM notes WHERE user_id = ? ORDER BY updated_at DESC");
        $stmt->execute([$_SESSION['user_id']]);
        $notes = $stmt->fetchAll();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>SuNote - Simple Note Taking App</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #fbbc04;
            --secondary-color: #f1f3f4;
            --text-color: #202124;
            --border-color: #e0e0e0;
            --hover-color: #f5f5f5;
            --shadow-light: 0 1px 3px rgba(0, 0, 0, 0.1);
            --shadow-medium: 0 2px 10px rgba(0, 0, 0, 0.15);
            --accent-color: #1a73e8;
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
            font-family: 'Roboto', Arial, sans-serif;
        }
        
        body {
            background-color: #f9f9f9;
            color: var(--text-color);
            line-height: 1.6;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 0;
            margin-bottom: 20px;
            border-bottom: 1px solid var(--border-color);
            background-color: #fff;
            position: sticky;
            top: 0;
            z-index: 10;
        }
        
        .header h1 {
            color: var(--primary-color);
            display: flex;
            align-items: center;
            font-size: 1.8rem;
        }
        
        .header h1 i {
            margin-right: 10px;
        }
        
        .user-info {
            display: flex;
            align-items: center;
        }
        
        .user-info span {
            margin-right: 15px;
            font-weight: 500;
        }
        
        .search-container {
            flex-grow: 1;
            max-width: 500px;
            margin: 0 20px;
            position: relative;
        }
        
        .search-container input {
            width: 100%;
            padding: 10px 15px 10px 40px;
            border-radius: 8px;
            border: 1px solid var(--border-color);
            background-color: var(--secondary-color);
            font-size: 16px;
            transition: all 0.3s ease;
        }
        
        .search-container input:focus {
            background-color: #fff;
            box-shadow: var(--shadow-light);
            outline: none;
            border-color: var(--primary-color);
        }
        
        .search-container i {
            position: absolute;
            left: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: #5f6368;
        }
        
        .btn {
            background-color: var(--primary-color);
            color: #fff;
            border: none;
            padding: 10px 16px;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.3s ease;
            font-weight: 500;
            display: inline-flex;
            align-items: center;
            justify-content: center;
        }
        
        .btn i {
            margin-right: 8px;
        }
        
        .btn:hover {
            background-color: #f9a602;
            transform: translateY(-1px);
            box-shadow: var(--shadow-light);
        }
        
        .btn:active {
            transform: translateY(0);
        }
        
        .btn-light {
            background-color: var(--secondary-color);
            color: var(--text-color);
        }
        
        .btn-light:hover {
            background-color: #e4e6e7;
        }
        
        .login-form {
            max-width: 400px;
            margin: 100px auto;
            padding: 30px;
            border-radius: 12px;
            box-shadow: var(--shadow-medium);
            background-color: #fff;
        }
        
        .login-form h2 {
            text-align: center;
            margin-bottom: 20px;
            color: var(--primary-color);
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 500;
        }
        
        .form-control {
            width: 100%;
            padding: 12px;
            border: 1px solid var(--border-color);
            border-radius: 8px;
            font-size: 16px;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 2px rgba(251, 188, 4, 0.2);
            outline: none;
        }
        
        .error-message {
            color: #d93025;
            margin-bottom: 15px;
            padding: 10px;
            background-color: rgba(217, 48, 37, 0.1);
            border-radius: 4px;
        }
        
        .note-form {
            background-color: #fff;
            padding: 20px;
            border-radius: 12px;
            box-shadow: var(--shadow-light);
            margin-bottom: 30px;
            border: 1px solid var(--border-color);
            transition: all 0.3s ease;
        }
        
        .note-form:focus-within {
            box-shadow: var(--shadow-medium);
        }
        
        .note-title {
            width: 100%;
            font-size: 18px;
            padding: 10px 0;
            margin-bottom: 10px;
            border: none;
            border-bottom: 1px solid var(--border-color);
            outline: none;
            font-weight: 500;
        }
        
        .note-content {
            width: 100%;
            min-height: 120px;
            padding: 10px 0;
            font-size: 16px;
            resize: vertical;
            border: none;
            outline: none;
            line-height: 1.6;
        }
        
        .form-actions {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 15px;
        }
        
        .keyboard-shortcuts {
            font-size: 14px;
            color: #5f6368;
        }
        
        .keyboard-shortcuts span {
            margin-right: 15px;
            display: inline-flex;
            align-items: center;
        }
        
        .keyboard-shortcuts kbd {
            background-color: var(--secondary-color);
            border-radius: 4px;
            padding: 2px 5px;
            margin-left: 5px;
            font-size: 12px;
            box-shadow: 0 1px 1px rgba(0, 0, 0, 0.1);
        }
        
        .notes-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }
        
        .note-card {
            background-color: #fff;
            border-radius: 12px;
            box-shadow: var(--shadow-light);
            padding: 20px;
            position: relative;
            border: 1px solid var(--border-color);
            transition: all 0.3s ease;
            cursor: pointer;
            overflow: hidden;
            display: flex;
            flex-direction: column;
        }
        
        .note-card:hover {
            box-shadow: var(--shadow-medium);
            transform: translateY(-2px);
        }
        
        .note-card h3 {
            margin-bottom: 12px;
            font-size: 18px;
            word-break: break-word;
            font-weight: 500;
        }
        
        .note-card p {
            font-size: 15px;
            color: #5f6368;
            white-space: pre-wrap;
            word-break: break-word;
            flex-grow: 1;
            max-height: 200px;
            overflow: hidden;
            position: relative;
        }
        
        .note-card p::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            width: 100%;
            height: 40px;
            background: linear-gradient(transparent, white);
            pointer-events: none;
        }
        
        .card-footer {
            display: flex;
            justify-content: space-between;
            margin-top: 15px;
            padding-top: 15px;
            border-top: 1px solid var(--border-color);
            font-size: 13px;
            color: #80868b;
        }
        
        .note-actions {
            position: absolute;
            top: 10px;
            right: 10px;
        }
        
        .note-actions .dropdown {
            position: relative;
            display: inline-block;
        }
        
        .dropdown-toggle {
            background: none;
            border: none;
            color: #80868b;
            cursor: pointer;
            font-size: 18px;
            padding: 8px;
            border-radius: 50%;
            transition: all 0.2s ease;
        }
        
        .dropdown-toggle:hover {
            color: var(--text-color);
            background-color: var(--hover-color);
        }
        
        .dropdown-menu {
            display: none;
            position: absolute;
            right: 0;
            background-color: #fff;
            min-width: 180px;
            box-shadow: var(--shadow-medium);
            z-index: 100;
            border-radius: 8px;
            border: 1px solid var(--border-color);
            padding: 8px 0;
        }
        
        .dropdown-menu.show {
            display: block;
            animation: fadeIn 0.2s ease;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(-10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .dropdown-item {
            padding: 10px 15px;
            text-decoration: none;
            display: flex;
            align-items: center;
            color: var(--text-color);
            font-size: 15px;
            transition: background-color 0.2s ease;
        }
        
        .dropdown-item i {
            width: 20px;
            margin-right: 10px;
            text-align: center;
        }
        
        .dropdown-item:hover {
            background-color: var(--hover-color);
        }
        
        .dropdown-item kbd {
            margin-left: auto;
            font-size: 12px;
            color: #5f6368;
            background-color: var(--secondary-color);
            padding: 2px 5px;
            border-radius: 4px;
            box-shadow: 0 1px 1px rgba(0, 0, 0, 0.1);
        }
        
        .dropdown-divider {
            height: 1px;
            background-color: var(--border-color);
            margin: 6px 0;
        }
        
        /* Modal styles */
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(0, 0, 0, 0.5);
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        
        .modal.show {
            opacity: 1;
        }
        
        .modal-content {
            background-color: #fff;
            margin: 5% auto;
            padding: 25px;
            border-radius: 12px;
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.2);
            max-width: 700px;
            width: 90%;
            transform: translateY(-20px);
            transition: transform 0.3s ease;
            position: relative;
        }
        
        .modal.show .modal-content {
            transform: translateY(0);
        }
        
        .close-modal {
            position: absolute;
            top: 15px;
            right: 15px;
            color: #aaa;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
            line-height: 1;
            transition: color 0.2s ease;
        }
        
        .close-modal:hover {
            color: var(--text-color);
        }
        
        .modal-header {
            margin-bottom: 20px;
        }
        
        .modal-header h2 {
            margin: 0;
            font-size: 22px;
            color: var(--text-color);
        }
        
        .share-link {
            padding: 12px 15px;
            background-color: var(--secondary-color);
            border-radius: 8px;
            margin: 15px 0;
            word-break: break-all;
            position: relative;
            font-size: 15px;
            border: 1px solid var(--border-color);
        }
        
        .copy-btn {
            position: absolute;
            right: 10px;
            top: 50%;
            transform: translateY(-50%);
            background: none;
            border: none;
            color: var(--accent-color);
            cursor: pointer;
            padding: 5px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 18px;
            width: 36px;
            height: 36px;
            transition: background-color 0.2s ease;
        }
        
        .copy-btn:hover {
            background-color: rgba(26, 115, 232, 0.1);
        }
        
        /* Find and replace styles */
        .search-replace-container {
            display: flex;
            gap: 10px;
            margin-bottom: 15px;
            flex-wrap: wrap;
        }
        
        .search-input, .replace-input {
            flex: 1;
            min-width: 200px;
            padding: 10px 15px;
            border-radius: 8px;
            border: 1px solid var(--border-color);
            font-size: 15px;
        }
        
        .search-input:focus, .replace-input:focus {
            border-color: var(--primary-color);
            outline: none;
        }
        
        .search-replace-actions {
            display: flex;
            gap: 10px;
            margin-bottom: 15px;
        }
        
        .search-btn, .replace-btn, .replace-all-btn {
            padding: 8px 12px;
            background-color: var(--secondary-color);
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 14px;
            transition: background-color 0.2s ease;
        }
        
        .search-btn:hover, .replace-btn:hover, .replace-all-btn:hover {
            background-color: #e4e6e7;
        }
        
        /* Loading spinner */
        .loading-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(255, 255, 255, 0.8);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 2000;
            opacity: 0;
            visibility: hidden;
            transition: opacity 0.3s ease, visibility 0.3s ease;
        }
        
        .loading-overlay.show {
            opacity: 1;
            visibility: visible;
        }
        
        .spinner {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            border: 5px solid var(--secondary-color);
            border-top-color: var(--primary-color);
            animation: spin 1s infinite linear;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        /* No notes message */
        .no-notes {
            text-align: center;
            padding: 40px 0;
            color: #5f6368;
        }
        
        .no-notes i {
            font-size: 48px;
            margin-bottom: 15px;
            color: var(--border-color);
        }
        
        .no-notes h3 {
            font-size: 22px;
            margin-bottom: 10px;
        }
        
        .no-notes p {
            font-size: 16px;
            max-width: 500px;
            margin: 0 auto;
        }
        
        /* Toast notification */
        .toast {
            position: fixed;
            bottom: 20px;
            right: 20px;
            background-color: var(--text-color);
            color: #fff;
            padding: 12px 20px;
            border-radius: 8px;
            box-shadow: 0 3px 10px rgba(0, 0, 0, 0.2);
            z-index: 2000;
            opacity: 0;
            transform: translateY(20px);
            transition: opacity 0.3s ease, transform 0.3s ease;
        }
        
        .toast.show {
            opacity: 1;
            transform: translateY(0);
        }
        
        /* Highlight search results */
        mark {
            background-color: rgba(251, 188, 4, 0.4);
            padding: 0;
            border-radius: 2px;
        }
        
        /* Responsive adjustments */
        @media (max-width: 768px) {
            .container {
                padding: 15px;
            }
            
            .header {
                flex-direction: column;
                padding: 15px 0 5px;
            }
            
            .header h1 {
                margin-bottom: 10px;
            }
            
            .search-container {
                margin: 10px 0;
                order: 3;
                max-width: 100%;
            }
            
            .notes-grid {
                grid-template-columns: repeat(auto-fill, minmax(100%, 1fr));
            }
            
            .keyboard-shortcuts {
                display: none;
            }
            
            .user-info {
                width: 100%;
                justify-content: space-between;
                margin-bottom: 10px;
            }
            
            .modal-content {
                margin: 10% auto;
                width: 95%;
                padding: 20px;
            }
            
            .form-actions {
                flex-direction: column;
                gap: 10px;
            }
            
            .form-actions button {
                width: 100%;
            }
        }
    </style>
</head>
<body>
    <!-- Loading overlay -->
    <div class="loading-overlay" id="loading-overlay">
        <div class="spinner"></div>
    </div>
    
    <!-- Toast notification -->
    <div class="toast" id="toast"></div>
    
    <div class="container">
        <?php if ($loginPage): ?>
            <!-- Login Form -->
            <div class="login-form">
                <h2><i class="fas fa-sticky-note"></i> SuNote</h2>
                <?php if (isset($login_error)): ?>
                    <div class="error-message"><?php echo $login_error; ?></div>
                <?php endif; ?>
                <form action="index.php" method="post">
                    <div class="form-group">
                        <label for="username">Username</label>
                        <input type="text" id="username" name="username" class="form-control" required>
                    </div>
                    <div class="form-group">
                        <label for="password">Password</label>
                        <input type="password" id="password" name="password" class="form-control" required>
                    </div>
                    <button type="submit" name="login" class="btn" style="width: 100%;">
                        <i class="fas fa-sign-in-alt"></i> Login
                    </button>
                </form>
            </div>
        <?php else: ?>
            <!-- Main Application -->
            <div class="header">
                <h1><i class="fas fa-sticky-note"></i> SuNote</h1>
                
                <div class="search-container">
                    <i class="fas fa-search"></i>
                    <input type="text" id="search-notes" placeholder="Search notes... (Press '/' to focus)" value="<?php echo isset($_GET['search']) ? htmlspecialchars($_GET['search']) : ''; ?>">
                </div>
                
                <div class="user-info">
                    <span>Welcome, <?php echo htmlspecialchars($_SESSION['username']); ?></span>
                    <a href="index.php?logout=1" class="btn btn-light">
                        <i class="fas fa-sign-out-alt"></i> Logout
                    </a>
                </div>
            </div>
            
            <!-- Note creation form -->
            <div class="note-form">
                <form id="create-note-form" action="index.php" method="post">
                    <input type="text" name="title" class="note-title" placeholder="Title" required>
                    <textarea name="content" class="note-content" placeholder="Take a note..."></textarea>
                    <div class="form-actions">
                        <div class="keyboard-shortcuts">
                            <span>New note <kbd>c</kbd></span>
                            <span>Search <kbd>/</kbd></span>
                            <span>Select all <kbd>Ctrl+a</kbd></span>
                        </div>
                        <button type="submit" name="create_note" class="btn">
                            <i class="fas fa-save"></i> Save Note
                        </button>
                    </div>
                </form>
            </div>
            
            <!-- Notes grid -->
            <div class="notes-grid">
                <?php if (count($notes) > 0): ?>
                    <?php foreach ($notes as $note): ?>
                    <div class="note-card" data-id="<?php echo $note['id']; ?>" data-title="<?php echo htmlspecialchars($note['title']); ?>" data-content="<?php echo htmlspecialchars($note['content']); ?>">
                        <div class="note-actions">
                            <div class="dropdown">
                                <button class="dropdown-toggle"><i class="fas fa-ellipsis-v"></i></button>
                                <div class="dropdown-menu">
                                    <a href="#" class="dropdown-item edit-note">
                                        <i class="fas fa-edit"></i> Edit
                                    </a>
                                    <a href="#" class="dropdown-item share-note">
                                        <i class="fas fa-share-alt"></i> Share
                                    </a>
                                    <div class="dropdown-divider"></div>
                                    <a href="index.php?delete=<?php echo $note['id']; ?>" class="dropdown-item delete-note" onclick="return confirm('Are you sure you want to delete this note?')">
                                        <i class="fas fa-trash-alt"></i> Delete
                                    </a>
                                </div>
                            </div>
                        </div>
                        <h3><?php echo htmlspecialchars($note['title']); ?></h3>
                        <p><?php echo nl2br(htmlspecialchars($note['content'])); ?></p>
                        <div class="card-footer">
                            <span><i class="far fa-clock"></i> <?php echo date('M d, Y', strtotime($note['updated_at'])); ?></span>
                            <?php if ($note['share_token']): ?>
                                <span><i class="fas fa-link"></i> Shared</span>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div class="no-notes">
                        <i class="far fa-sticky-note"></i>
                        <h3>No notes found</h3>
                        <?php if (isset($_GET['search'])): ?>
                            <p>No notes match your search. Try with different keywords or <a href="index.php">view all notes</a>.</p>
                        <?php else: ?>
                            <p>Get started by creating your first note above.</p>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- Edit Modal -->
            <div id="edit-modal" class="modal">
                <div class="modal-content">
                    <span class="close-modal">&times;</span>
                    <div class="modal-header">
                        <h2><i class="fas fa-edit"></i> Edit Note</h2>
                    </div>
                    
                    <!-- Find and Replace -->
                    <div class="search-replace-container">
                        <input type="text" id="find-text" class="search-input" placeholder="Find... (Ctrl+F)">
                        <input type="text" id="replace-text" class="replace-input" placeholder="Replace with...">
                    </div>
                    <div class="search-replace-actions">
                        <button id="find-btn" class="search-btn"><i class="fas fa-search"></i> Find</button>
                        <button id="replace-btn" class="replace-btn"><i class="fas fa-exchange-alt"></i> Replace</button>
                        <button id="replace-all-btn" class="replace-all-btn"><i class="fas fa-sync-alt"></i> Replace All</button>
                        <span id="search-results" style="margin-left: auto; font-size: 14px; color: #5f6368;"></span>
                    </div>
                    
                    <form id="edit-note-form" action="index.php" method="post">
                        <input type="hidden" name="note_id" id="edit-note-id">
                        <div class="form-group">
                            <input type="text" name="title" id="edit-note-title" class="form-control" placeholder="Title" required>
                        </div>
                        <div class="form-group">
                            <textarea name="content" id="edit-note-content" class="form-control" style="min-height: 300px;" placeholder="Content"></textarea>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <div class="keyboard-shortcuts">
                                <span>Finish editing <kbd>Esc</kbd> or <kbd>Ctrl+Enter</kbd></span>
                            </div>
                            <button type="submit" name="update_note" class="btn">
                                <i class="fas fa-save"></i> Update Note
                            </button>
                        </div>
                    </form>
                </div>
            </div>
            
            <!-- Share Modal -->
            <div id="share-modal" class="modal">
                <div class="modal-content">
                    <span class="close-modal">&times;</span>
                    <div class="modal-header">
                        <h2><i class="fas fa-share-alt"></i> Share Note</h2>
                    </div>
                    <p>Share this link with others to let them view this note:</p>
                    <div style="position: relative;">
                        <div id="share-link" class="share-link"></div>
                        <button id="copy-link" class="copy-btn" title="Copy to clipboard">
                            <i class="far fa-copy"></i>
                        </button>
                    </div>
                    <p style="margin-top: 15px; font-size: 14px; color: #5f6368;">
                        <i class="fas fa-info-circle"></i> Anyone with this link can view this note, but only you can edit it.
                    </p>
                </div>
            </div>
            
            <script>
                // Show loading overlay
                const loadingOverlay = document.getElementById('loading-overlay');
                function showLoading() {
                    loadingOverlay.classList.add('show');
                }
                
                function hideLoading() {
                    loadingOverlay.classList.remove('show');
                }
                
                // Toast notification
                const toast = document.getElementById('toast');
                function showToast(message, duration = 3000) {
                    toast.textContent = message;
                    toast.classList.add('show');
                    setTimeout(() => {
                        toast.classList.remove('show');
                    }, duration);
                }
                
                // Toggle dropdown menus
                document.addEventListener('click', function(e) {
                    if (e.target.matches('.dropdown-toggle') || e.target.matches('.dropdown-toggle *')) {
                        const dropdown = e.target.closest('.dropdown');
                        const menu = dropdown.querySelector('.dropdown-menu');
                        menu.classList.toggle('show');
                        e.stopPropagation();
                    } else if (!e.target.matches('.dropdown-menu') && !e.target.matches('.dropdown-menu *')) {
                        document.querySelectorAll('.dropdown-menu.show').forEach(menu => {
                            menu.classList.remove('show');
                        });
                    }
                });
                
                // Edit note functionality
                const editModal = document.getElementById('edit-modal');
                const editNoteId = document.getElementById('edit-note-id');
                const editNoteTitle = document.getElementById('edit-note-title');
                const editNoteContent = document.getElementById('edit-note-content');
                
                function openEditModal(noteId, noteTitle, noteContent) {
                    editNoteId.value = noteId;
                    editNoteTitle.value = noteTitle;
                    editNoteContent.value = noteContent;
                    
                    editModal.style.display = 'block';
                    setTimeout(() => {
                        editModal.classList.add('show');
                        // Focus title
                        editNoteTitle.focus();
                    }, 10);
                    
                    // Reset search highlight
                    currentSearchIndex = -1;
                    document.getElementById('search-results').textContent = '';
                }
                
                document.querySelectorAll('.edit-note').forEach(link => {
                    link.addEventListener('click', function(e) {
                        e.preventDefault();
                        const noteCard = this.closest('.note-card');
                        const noteId = noteCard.dataset.id;
                        const noteTitle = noteCard.dataset.title;
                        const noteContent = noteCard.dataset.content;
                        
                        openEditModal(noteId, noteTitle, noteContent);
                    });
                });
                
                // Share note functionality
                const shareModal = document.getElementById('share-modal');
                const shareLink = document.getElementById('share-link');
                const copyLinkBtn = document.getElementById('copy-link');
                let currentNoteId = null;
                
                document.querySelectorAll('.share-note').forEach(link => {
                    link.addEventListener('click', function(e) {
                        e.preventDefault();
                        const noteCard = this.closest('.note-card');
                        currentNoteId = noteCard.dataset.id;
                        
                        // Show loading indicator
                        showLoading();
                        
                        // Make AJAX request to share the note
                        fetch('index.php', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/x-www-form-urlencoded',
                            },
                            body: 'share_note=1&note_id=' + currentNoteId
                        })
                        .then(response => response.json())
                        .then(data => {
                            // Hide loading indicator
                            hideLoading();
                            
                            const fullShareLink = window.location.origin + window.location.pathname.replace('index.php', '') + data.share_link;
                            shareLink.textContent = fullShareLink;
                            
                            shareModal.style.display = 'block';
                            setTimeout(() => {
                                shareModal.classList.add('show');
                            }, 10);
                        })
                        .catch(error => {
                            hideLoading();
                            showToast('Error sharing note: ' + error, 5000);
                        });
                    });
                });
                
                // Copy share link functionality
                copyLinkBtn.addEventListener('click', function() {
                    const tempInput = document.createElement('input');
                    document.body.appendChild(tempInput);
                    tempInput.value = shareLink.textContent;
                    tempInput.select();
                    document.execCommand('copy');
                    document.body.removeChild(tempInput);
                    
                    showToast('Link copied to clipboard!');
                });
                
                // Close modals
                document.querySelectorAll('.close-modal').forEach(closeBtn => {
                    closeBtn.addEventListener('click', function() {
                        const modal = this.closest('.modal');
                        modal.classList.remove('show');
                        setTimeout(() => {
                            modal.style.display = 'none';
                        }, 300);
                    });
                });
                
                // Close modal when clicking outside
                window.addEventListener('click', function(e) {
                    if (e.target.classList.contains('modal')) {
                        e.target.classList.remove('show');
                        setTimeout(() => {
                            e.target.style.display = 'none';
                        }, 300);
                    }
                });
                
                // Make note cards clickable to edit
                document.querySelectorAll('.note-card').forEach(card => {
                    card.addEventListener('click', function(e) {
                        // Don't trigger if clicking on dropdown or action buttons
                        if (!e.target.closest('.note-actions')) {
                            const noteId = this.dataset.id;
                            const noteTitle = this.dataset.title;
                            const noteContent = this.dataset.content;
                            
                            openEditModal(noteId, noteTitle, noteContent);
                        }
                    });
                });
                
                // Search functionality
                const searchInput = document.getElementById('search-notes');
                
                searchInput.addEventListener('keyup', function(e) {
                    if (e.key === 'Enter') {
                        const searchTerm = this.value.trim();
                        if (searchTerm !== '') {
                            window.location.href = 'index.php?search=' + encodeURIComponent(searchTerm);
                        } else {
                            window.location.href = 'index.php';
                        }
                    }
                });
                
                // Find and replace functionality
                const findInput = document.getElementById('find-text');
                const replaceInput = document.getElementById('replace-text');
                const findBtn = document.getElementById('find-btn');
                const replaceBtn = document.getElementById('replace-btn');
                const replaceAllBtn = document.getElementById('replace-all-btn');
                const searchResults = document.getElementById('search-results');
                
                let currentSearchIndex = -1;
                let matches = [];
                
                function highlightText(text, searchTerm) {
                    if (!searchTerm) return text;
                    const regex = new RegExp(`(${escapeRegExp(searchTerm)})`, 'gi');
                    return text.replace(regex, '<mark>$1</mark>');
                }
                
                function escapeRegExp(string) {
                    return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
                }
                
                function findInContent() {
                    const searchTerm = findInput.value.trim();
                    if (!searchTerm) {
                        searchResults.textContent = '';
                        return;
                    }
                    
                    const content = editNoteContent.value;
                    const regex = new RegExp(escapeRegExp(searchTerm), 'gi');
                    matches = [];
                    let match;
                    
                    while ((match = regex.exec(content)) !== null) {
                        matches.push({
                            start: match.index,
                            end: match.index + match[0].length
                        });
                    }
                    
                    if (matches.length > 0) {
                        currentSearchIndex = 0;
                        searchResults.textContent = `${currentSearchIndex + 1} of ${matches.length} matches`;
                        highlightMatch();
                    } else {
                        searchResults.textContent = 'No matches found';
                    }
                }
                
                function highlightMatch() {
                    if (matches.length === 0 || currentSearchIndex === -1) return;
                    
                    const match = matches[currentSearchIndex];
                    const textarea = editNoteContent;
                    
                    // Select the text
                    textarea.focus();
                    textarea.setSelectionRange(match.start, match.end);
                    
                    // Scroll to the match
                    const textBeforeMatch = textarea.value.substring(0, match.start);
                    const linesBeforeMatch = textBeforeMatch.split('\n').length - 1;
                    const lineHeight = parseInt(window.getComputedStyle(textarea).lineHeight);
                    textarea.scrollTop = linesBeforeMatch * lineHeight;
                }
                
                function replaceCurrentMatch() {
                    if (matches.length === 0 || currentSearchIndex === -1) return;
                    
                    const match = matches[currentSearchIndex];
                    const textarea = editNoteContent;
                    const replaceTerm = replaceInput.value;
                    
                    // Replace the text
                    const content = textarea.value;
                    textarea.value = content.substring(0, match.start) + 
                                    replaceTerm + 
                                    content.substring(match.end);
                    
                    // Update matches after replacement
                    findInContent();
                }
                
                function replaceAllMatches() {
                    const searchTerm = findInput.value.trim();
                    if (!searchTerm) return;
                    
                    const replaceTerm = replaceInput.value;
                    const regex = new RegExp(escapeRegExp(searchTerm), 'gi');
                    const textarea = editNoteContent;
                    
                    // Replace all occurrences
                    const originalContent = textarea.value;
                    const newContent = originalContent.replace(regex, replaceTerm);
                    textarea.value = newContent;
                    
                    // Count replacements
                    const replacementCount = (originalContent.match(regex) || []).length;
                    showToast(`Replaced ${replacementCount} occurrences`);
                    
                    // Reset search
                    matches = [];
                    currentSearchIndex = -1;
                    searchResults.textContent = '';
                }
                
                findBtn.addEventListener('click', findInContent);
                findInput.addEventListener('keyup', function(e) {
                    if (e.key === 'Enter') {
                        findInContent();
                    }
                });
                
                replaceBtn.addEventListener('click', replaceCurrentMatch);
                replaceAllBtn.addEventListener('click', replaceAllMatches);
                
                // Keyboard shortcuts
                document.addEventListener('keydown', function(e) {
                    // '/' key for search focus
                    if (e.key === '/' && !e.ctrlKey && !e.metaKey && 
                        !e.target.matches('input') && !e.target.matches('textarea')) {
                        e.preventDefault();
                        searchInput.focus();
                    }
                    
                    // 'c' key for new note
                    if (e.key === 'c' && !e.ctrlKey && !e.metaKey && 
                        !e.target.matches('input') && !e.target.matches('textarea')) {
                        e.preventDefault();
                        document.querySelector('.note-title').focus();
                    }
                    
                    // Ctrl+A to select all notes (just a placeholder for now)
                    if ((e.ctrlKey || e.metaKey) && e.key === 'a' && 
                        !e.target.matches('input') && !e.target.matches('textarea')) {
                        e.preventDefault();
                        // Select all notes functionality would go here
                        showToast('Select all notes feature coming soon!');
                    }
                    
                    // Ctrl+F for find in edit modal
                    if ((e.ctrlKey || e.metaKey) && e.key === 'f' && editModal.classList.contains('show')) {
                        e.preventDefault();
                        findInput.focus();
                    }
                    
                    // Esc to close modal
                    if (e.key === 'Escape') {
                        const openModals = document.querySelectorAll('.modal.show');
                        if (openModals.length > 0) {
                            openModals.forEach(modal => {
                                modal.classList.remove('show');
                                setTimeout(() => {
                                    modal.style.display = 'none';
                                }, 300);
                            });
                        }
                    }
                    
                    // Ctrl+Enter to submit form in edit modal
                    if ((e.ctrlKey || e.metaKey) && e.key === 'Enter' && editModal.classList.contains('show')) {
                        e.preventDefault();
                        document.getElementById('edit-note-form').submit();
                    }
                });
                
                // Submit edit form with Ctrl+Enter
                editNoteContent.addEventListener('keydown', function(e) {
                    if ((e.ctrlKey || e.metaKey) && e.key === 'Enter') {
                        e.preventDefault();
                        document.getElementById('edit-note-form').submit();
                    }
                });
                
                // Highlight search results in notes if search param exists
                const urlParams = new URLSearchParams(window.location.search);
                const searchParam = urlParams.get('search');
                if (searchParam) {
                    document.querySelectorAll('.note-card p').forEach(content => {
                        const originalText = content.textContent;
                        content.innerHTML = highlightText(originalText, searchParam);
                    });
                    
                    document.querySelectorAll('.note-card h3').forEach(title => {
                        const originalText = title.textContent;
                        title.innerHTML = highlightText(originalText, searchParam);
                    });
                }

                // Autosave functionality for new notes
                document.addEventListener('DOMContentLoaded', function() {
                    const noteTitle = document.querySelector('.note-title');
                    const noteContent = document.querySelector('.note-content');
                    const createNoteForm = document.getElementById('create-note-form');
                    const AUTOSAVE_INTERVAL = 2000; // Save every 2 seconds
                    const LOCAL_STORAGE_KEY = 'sunote_draft';
                    
                    // Load saved draft on page load
                    loadSavedDraft();
                    
                    // Set up autosave timers
                    let saveTimer;
                    
                    // Autosave when typing in the title
                    noteTitle.addEventListener('input', function() {
                        clearTimeout(saveTimer);
                        saveTimer = setTimeout(saveDraft, AUTOSAVE_INTERVAL);
                    });
                    
                    // Autosave when typing in the content
                    noteContent.addEventListener('input', function() {
                        clearTimeout(saveTimer);
                        saveTimer = setTimeout(saveDraft, AUTOSAVE_INTERVAL);
                    });
                    
                    // Save draft to localStorage
                    function saveDraft() {
                        if (noteTitle.value.trim() === '' && noteContent.value.trim() === '') {
                            // Don't save empty drafts
                            localStorage.removeItem(LOCAL_STORAGE_KEY);
                            return;
                        }
                        
                        const draft = {
                            title: noteTitle.value,
                            content: noteContent.value,
                            timestamp: new Date().getTime()
                        };
                        
                        localStorage.setItem(LOCAL_STORAGE_KEY, JSON.stringify(draft));
                        showToast('Draft saved', 1000);
                    }
                    
                    // Load saved draft from localStorage
                    function loadSavedDraft() {
                        const savedDraft = localStorage.getItem(LOCAL_STORAGE_KEY);
                        
                        if (savedDraft) {
                            const draft = JSON.parse(savedDraft);
                            noteTitle.value = draft.title || '';
                            noteContent.value = draft.content || '';
                            
                            // Show a notification that a draft was loaded
                            showToast('Draft loaded', 2000);
                        }
                    }
                    
                    // Clear draft when form is submitted
                    createNoteForm.addEventListener('submit', function() {
                        localStorage.removeItem(LOCAL_STORAGE_KEY);
                    });
                    
                    // Also add autosave for edit modal
                    const editNoteTitle = document.getElementById('edit-note-title');
                    const editNoteContent = document.getElementById('edit-note-content');
                    const editNoteForm = document.getElementById('edit-note-form');
                    
                    if (editNoteTitle && editNoteContent && editNoteForm) {
                        let editSaveTimer;
                        
                        // Function to generate a unique key for each note
                        function getEditDraftKey(noteId) {
                            return `sunote_edit_draft_${noteId}`;
                        }
                        
                        // Autosave when editing title
                        editNoteTitle.addEventListener('input', function() {
                            clearTimeout(editSaveTimer);
                            editSaveTimer = setTimeout(saveEditDraft, AUTOSAVE_INTERVAL);
                        });
                        
                        // Autosave when editing content
                        editNoteContent.addEventListener('input', function() {
                            clearTimeout(editSaveTimer);
                            editSaveTimer = setTimeout(saveEditDraft, AUTOSAVE_INTERVAL);
                        });
                        
                        // Save edit draft to localStorage
                        function saveEditDraft() {
                            const noteId = document.getElementById('edit-note-id').value;
                            
                            if (!noteId || (editNoteTitle.value.trim() === '' && editNoteContent.value.trim() === '')) {
                                return;
                            }
                            
                            const draft = {
                                id: noteId,
                                title: editNoteTitle.value,
                                content: editNoteContent.value,
                                timestamp: new Date().getTime()
                            };
                            
                            localStorage.setItem(getEditDraftKey(noteId), JSON.stringify(draft));
                            showToast('Edit draft saved', 1000);
                        }
                        
                        // Load edit draft when opening the modal
                        document.querySelectorAll('.edit-note, .note-card').forEach(element => {
                            element.addEventListener('click', function(e) {
                                if (e.target.closest('.dropdown-toggle') || e.target.closest('.dropdown-menu')) {
                                    return; // Don't trigger if clicking dropdown
                                }
                                
                                // Wait a bit for the modal to populate with note data
                                setTimeout(function() {
                                    const noteId = document.getElementById('edit-note-id').value;
                                    if (!noteId) return;
                                    
                                    const savedDraft = localStorage.getItem(getEditDraftKey(noteId));
                                    if (savedDraft) {
                                        const draft = JSON.parse(savedDraft);
                                        // Only load if the draft is for the current note
                                        if (draft.id === noteId) {
                                            editNoteTitle.value = draft.title || '';
                                            editNoteContent.value = draft.content || '';
                                            showToast('Edit draft loaded', 2000);
                                        }
                                    }
                                }, 300);
                            });
                        });
                        
                        // Clear edit draft when form is submitted
                        editNoteForm.addEventListener('submit', function() {
                            const noteId = document.getElementById('edit-note-id').value;
                            if (noteId) {
                                localStorage.removeItem(getEditDraftKey(noteId));
                            }
                        });
                    }
                });

                // AJAX save functionality
                document.addEventListener('DOMContentLoaded', function() {
                    const noteTitle = document.querySelector('.note-title');
                    const noteContent = document.querySelector('.note-content');
                    const createNoteForm = document.getElementById('create-note-form');
                    const saveNoteBtn = document.querySelector('button[name="create_note"]');
                    const AUTOSAVE_INTERVAL = 30000; // Auto-save to server every 30 seconds
                    let serverSaveTimer;
                    let currentNoteId = null; // Track if we're editing an existing note
                    
                    // Add input event listeners for auto-saving to server
                    [noteTitle, noteContent].forEach(element => {
                        if (element) {
                            element.addEventListener('input', function() {
                                // Reset the timer whenever there's input
                                clearTimeout(serverSaveTimer);
                                // Set a new timer for server save
                                serverSaveTimer = setTimeout(saveToServer, AUTOSAVE_INTERVAL);
                            });
                        }
                    });
                    
                    // Save to server function (for new notes)
                    function saveToServer() {
                        const title = noteTitle.value.trim();
                        const content = noteContent.value.trim();
                        
                        // Don't save if both fields are empty
                        if (title === '' && content === '') {
                            return;
                        }
                        
                        // Create form data
                        const formData = new FormData();
                        formData.append('autosave_note', '1');
                        formData.append('title', title);
                        formData.append('content', content);
                        
                        // If we have a note ID (from a previous autosave), include it
                        if (currentNoteId) {
                            formData.append('note_id', currentNoteId);
                        }
                        
                        // Show subtle saving indicator
                        const saveIndicator = document.createElement('span');
                        saveIndicator.textContent = ' Saving...';
                        saveIndicator.style.opacity = '0.6';
                        saveIndicator.style.fontSize = '0.9em';
                        saveIndicator.style.marginLeft = '10px';
                        saveIndicator.id = 'save-indicator';
                        
                        // Remove any existing indicator
                        const existingIndicator = document.getElementById('save-indicator');
                        if (existingIndicator) {
                            existingIndicator.remove();
                        }
                        
                        // Add indicator next to save button
                        saveNoteBtn.parentNode.insertBefore(saveIndicator, saveNoteBtn.nextSibling);
                        
                        // Send AJAX request
                        fetch('index.php', {
                            method: 'POST',
                            body: formData
                        })
                        .then(response => response.json())
                        .then(data => {
                            if (data.success) {
                                // Update currentNoteId if this was a newly created note
                                if (data.note_id && !currentNoteId) {
                                    currentNoteId = data.note_id;
                                }
                                
                                // Update save indicator text
                                saveIndicator.textContent = ' Saved';
                                
                                // Remove indicator after a moment
                                setTimeout(() => {
                                    saveIndicator.remove();
                                }, 2000);
                            } else {
                                // Handle error
                                saveIndicator.textContent = ' Save failed';
                                saveIndicator.style.color = 'red';
                            }
                        })
                        .catch(error => {
                            console.error('Error:', error);
                            saveIndicator.textContent = ' Save failed';
                            saveIndicator.style.color = 'red';
                        });
                    }
                    
                    // Modify the form submission to use AJAX
                    createNoteForm.addEventListener('submit', function(e) {
                        e.preventDefault(); // Prevent traditional form submission
                        
                        const title = noteTitle.value.trim();
                        const content = noteContent.value.trim();
                        
                        // Don't save if both fields are empty
                        if (title === '' && content === '') {
                            showToast('Please enter a title or content', 3000);
                            return;
                        }
                        
                        // Show loading indicator
                        showLoading();
                        
                        // Create form data
                        const formData = new FormData();
                        formData.append('create_note', '1');
                        formData.append('title', title);
                        formData.append('content', content);
                        
                        // If we have a note ID (from a previous autosave), include it for update instead of create
                        if (currentNoteId) {
                            formData.append('note_id', currentNoteId);
                            formData.set('create_note', null);
                            formData.append('update_note', '1');
                        }
                        
                        // Send AJAX request
                        fetch('index.php', {
                            method: 'POST',
                            body: formData
                        })
                        .then(response => {
                            // Refresh the page to show the updated notes list
                            localStorage.removeItem('sunote_draft'); // Clear the draft
                            window.location.reload();
                        })
                        .catch(error => {
                            console.error('Error:', error);
                            hideLoading();
                            showToast('Error saving note. Please try again.', 5000);
                        });
                    });
                    
                    // Also implement AJAX saving for the edit modal
                    const editNoteTitle = document.getElementById('edit-note-title');
                    const editNoteContent = document.getElementById('edit-note-content');
                    const editNoteForm = document.getElementById('edit-note-form');
                    const editNoteId = document.getElementById('edit-note-id');
                    
                    if (editNoteTitle && editNoteContent && editNoteForm) {
                        let editServerSaveTimer;
                        
                        // Add input event listeners for auto-saving edits to server
                        [editNoteTitle, editNoteContent].forEach(element => {
                            element.addEventListener('input', function() {
                                // Reset the timer whenever there's input
                                clearTimeout(editServerSaveTimer);
                                // Set a new timer for server save
                                editServerSaveTimer = setTimeout(saveEditToServer, AUTOSAVE_INTERVAL);
                            });
                        });
                        
                        // Save edits to server function
                        function saveEditToServer() {
                            const noteId = editNoteId.value;
                            const title = editNoteTitle.value.trim();
                            const content = editNoteContent.value.trim();
                            
                            // Don't save if both fields are empty or if no note ID
                            if (!noteId || (title === '' && content === '')) {
                                return;
                            }
                            
                            // Create form data
                            const formData = new FormData();
                            formData.append('autosave_note', '1');
                            formData.append('note_id', noteId);
                            formData.append('title', title);
                            formData.append('content', content);
                            
                            // Send AJAX request
                            fetch('index.php', {
                                method: 'POST',
                                body: formData
                            })
                            .then(response => response.json())
                            .then(data => {
                                if (data.success) {
                                    showToast('Changes saved', 1000);
                                } else {
                                    showToast('Failed to save changes', 3000);
                                }
                            })
                            .catch(error => {
                                console.error('Error:', error);
                                showToast('Error saving changes', 3000);
                            });
                        }
                        
                        // Modify the edit form submission to use AJAX
                        editNoteForm.addEventListener('submit', function(e) {
                            e.preventDefault(); // Prevent traditional form submission
                            
                            const noteId = editNoteId.value;
                            const title = editNoteTitle.value.trim();
                            const content = editNoteContent.value.trim();
                            
                            // Don't save if both fields are empty
                            if (title === '' && content === '') {
                                showToast('Please enter a title or content', 3000);
                                return;
                            }
                            
                            // Show loading indicator
                            showLoading();
                            
                            // Create form data
                            const formData = new FormData();
                            formData.append('update_note', '1');
                            formData.append('note_id', noteId);
                            formData.append('title', title);
                            formData.append('content', content);
                            
                            // Send AJAX request
                            fetch('index.php', {
                                method: 'POST',
                                body: formData
                            })
                            .then(response => {
                                // Remove the edit draft from localStorage
                                localStorage.removeItem(`sunote_edit_draft_${noteId}`);
                                
                                // Refresh the page to show the updated note
                                window.location.reload();
                            })
                            .catch(error => {
                                console.error('Error:', error);
                                hideLoading();
                                showToast('Error updating note. Please try again.', 5000);
                            });
                        });
                    }
                });
            </script>
        <?php endif; ?>
    </div>
</body>
</html>