<?php
session_start();

// Check authentication
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    header('Location: login.php');
    exit;
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width, initial-scale=1" />
<link rel="icon" type="image/png" href="https://app.dezki.biz.id/linkpic/assets/uploads/688763ef78d63.png" />
<title>Reminder App</title>
<script src="https://cdn.tailwindcss.com"></script>
<style>
  /* Custom scrollbar */
  ::-webkit-scrollbar {
    width: 6px;
    height: 6px;
  }
  ::-webkit-scrollbar-thumb {
    background-color: #cbd5e1;
    border-radius: 10px;
  }
  ::-webkit-scrollbar-track {
    background-color: transparent;
  }
  
  /* Animations */
  .fade-in {
    animation: fadeIn 0.3s ease-in;
  }
  @keyframes fadeIn {
    from { opacity: 0; transform: translateY(10px); }
    to { opacity: 1; transform: translateY(0); }
  }
  
  .slide-up {
    animation: slideUp 0.3s ease-out;
  }
  @keyframes slideUp {
    from { transform: translateY(100%); opacity: 0; }
    to { transform: translateY(0); opacity: 1; }
  }

  .slide-down {
    animation: slideDown 0.3s ease-out;
  }
  @keyframes slideDown {
    from { max-height: 0; opacity: 0; }
    to { max-height: 500px; opacity: 1; }
  }
  
  /* Custom checkbox */
  .custom-checkbox {
    appearance: none;
    width: 20px;
    height: 20px;
    border: 2px solid #d1d5db;
    border-radius: 50%;
    position: relative;
    cursor: pointer;
    transition: all 0.2s ease;
    flex-shrink: 0;
  }
  
  .custom-checkbox:checked {
    background-color: #6366f1;
    border-color: #6366f1;
  }
  
  .custom-checkbox:checked::after {
    content: '✓';
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    color: white;
    font-size: 12px;
    font-weight: bold;
  }

  .custom-checkbox.subtask {
    width: 16px;
    height: 16px;
  }

  .custom-checkbox.subtask:checked::after {
    font-size: 10px;
  }
  
  /* Priority indicators */
  .priority-high { border-left: 4px solid #ef4444; }
  .priority-medium { border-left: 4px solid #f59e0b; }
  .priority-low { border-left: 4px solid #10b981; }
  
  /* Samsung-style card */
  .samsung-card {
    background: linear-gradient(135deg, #f8fafc 0%, #ffffff 100%);
    border: 1px solid #e2e8f0;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08);
    transition: all 0.2s ease;
  }
  
  .samsung-card:hover {
    box-shadow: 0 4px 16px rgba(0, 0, 0, 0.12);
    transform: translateY(-1px);
  }
  
  /* Modal backdrop */
  .modal-backdrop {
    background-color: rgba(0, 0, 0, 0.5);
    backdrop-filter: blur(4px);
  }

  /* Subtask styles */
  .subtask-container {
    border-left: 2px solid #e5e7eb;
    margin-left: 10px;
    padding-left: 12px;
  }

  .subtask-item {
    background: #f9fafb;
    border: 1px solid #f3f4f6;
    border-radius: 8px;
    padding: 8px 12px;
    margin-bottom: 6px;
    transition: all 0.2s ease;
  }

  .subtask-item:hover {
    background: #f3f4f6;
  }

  .task-expanded .subtask-container {
    animation: slideDown 0.3s ease-out;
  }

  /* Mobile optimizations */
  @media (max-width: 768px) {
    .stats-mobile {
      padding: 12px 16px;
    }
    
    .stats-mobile h3 {
      font-size: 14px;
      margin-bottom: 4px;
    }
    
    .stats-mobile p {
      font-size: 20px;
      font-weight: bold;
    }
    
    .stats-mobile .icon {
      width: 32px;
      height: 32px;
    }
    
    .stats-mobile .icon svg {
      width: 16px;
      height: 16px;
    }
  }
</style>
</head>
<body class="bg-gradient-to-br from-slate-50 to-blue-50 text-gray-800 antialiased min-h-screen flex flex-col">

<!-- Top bar -->
<header class="flex items-center justify-between px-4 py-4 bg-white/80 backdrop-blur-sm shadow-sm sticky top-0 z-20 border-b border-gray-100">
  <button id="btn-menu" aria-label="Toggle menu" class="text-2xl md:hidden focus:outline-none hover:bg-gray-100 p-2 rounded-lg transition-colors" title="Toggle menu">☰</button>
  <div class="flex items-center space-x-3">
    <div class="w-8 h-8 bg-gradient-to-br from-blue-500 to-purple-600 rounded-lg flex items-center justify-center">
        <div class="flex flex-col gap-0.5">
            <div class="flex items-center gap-1">
                <div class="w-2 h-2 bg-white rounded-sm"></div>
                <div class="w-2 h-0.5 bg-white rounded"></div>
            </div>
            <div class="flex items-center gap-1">
                <div class="w-2 h-2 bg-white rounded-sm"></div>
                <div class="w-2 h-0.5 bg-white rounded"></div>
            </div>
        </div>
    </div>
    <h1 id="page-title" class="font-bold text-xl text-gray-900">Semua Pengingat</h1>
  </div>
  <div class="flex space-x-2 items-center">
    <button id="btn-search" aria-label="Search reminder" class="p-2 hover:bg-gray-100 rounded-lg transition-colors focus:outline-none" title="Cari pengingat">
      <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
      </svg>
    </button>
    <button id="btn-add-category" aria-label="Add category" class="p-2 hover:bg-gray-100 rounded-lg transition-colors focus:outline-none" title="Tambah kategori">
      <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
      </svg>
    </button>
    <a href="logout.php" class="p-2 hover:bg-gray-100 rounded-lg transition-colors focus:outline-none" title="Logout">
      <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"></path>
      </svg>
    </a>
  </div>
</header>

<div class="flex flex-1 overflow-hidden relative">

  <!-- Sidebar -->
  <nav id="sidebar" class="fixed inset-y-0 left-0 w-80 bg-white/95 backdrop-blur-sm shadow-xl p-6 overflow-y-auto transform -translate-x-full md:translate-x-0 md:static md:flex-shrink-0 z-30 border-r border-gray-100">
    <div class="flex items-center justify-between mb-6">
      <h2 class="font-bold text-lg text-gray-900">Kategori</h2>
      <button id="btn-close-sidebar" aria-label="Close menu" class="md:hidden p-2 hover:bg-gray-100 rounded-lg transition-colors focus:outline-none" title="Close menu">
        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
        </svg>
      </button>
    </div>
    
    <!-- All reminders -->
    <div id="all-category" class="mb-4 p-3 rounded-xl cursor-pointer hover:bg-blue-50 transition-colors bg-blue-100 border-l-4 border-blue-500" data-id="cat_all">
      <div class="flex items-center justify-between">
        <div class="flex items-center space-x-3">
          <div class="w-8 h-8 bg-blue-500 rounded-lg flex items-center justify-center">
            <span class="text-white text-sm">📋</span>
          </div>
          <span class="font-medium text-gray-900">Semua</span>
        </div>
        <span id="all-count" class="bg-blue-500 text-white text-xs px-2 py-1 rounded-full font-medium">0</span>
      </div>
    </div>
    
    <div class="mb-4">
      <h3 class="text-sm font-medium text-gray-500 mb-3 uppercase tracking-wide">Kategori Saya</h3>
      <ul id="category-list" class="space-y-2">
        <!-- Categories inserted by JS -->
      </ul>
    </div>
  </nav>

  <!-- Overlay -->
  <div id="overlay" class="fixed inset-0 bg-black bg-opacity-30 hidden z-20 md:hidden" tabindex="-1"></div>

  <!-- Main content -->
  <main class="flex-1 overflow-y-auto p-3 md:p-6 max-w-full min-w-0">
    
    <!-- Stats cards - Mobile optimized -->
    <div class="grid grid-cols-3 md:grid-cols-3 gap-2 md:gap-4 mb-4 md:mb-6">
      <div class="samsung-card rounded-xl stats-mobile p-3 md:p-4">
        <div class="flex items-center justify-between">
          <div>
            <p class="text-xs md:text-sm text-gray-500">Total</p>
            <p id="total-tasks" class="text-lg md:text-2xl font-bold text-gray-900">0</p>
          </div>
          <div class="icon w-8 h-8 md:w-12 md:h-12 bg-blue-100 rounded-xl flex items-center justify-center">
            <svg class="w-4 h-4 md:w-6 md:h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v10a2 2 0 002 2h8a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"></path>
            </svg>
          </div>
        </div>
      </div>
      
      <div class="samsung-card rounded-xl stats-mobile p-3 md:p-4">
        <div class="flex items-center justify-between">
          <div>
            <p class="text-xs md:text-sm text-gray-500">Selesai</p>
            <p id="completed-tasks" class="text-lg md:text-2xl font-bold text-green-600">0</p>
          </div>
          <div class="icon w-8 h-8 md:w-12 md:h-12 bg-green-100 rounded-xl flex items-center justify-center">
            <svg class="w-4 h-4 md:w-6 md:h-6 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
            </svg>
          </div>
        </div>
      </div>
      
      <div class="samsung-card rounded-xl stats-mobile p-3 md:p-4">
        <div class="flex items-center justify-between">
          <div>
            <p class="text-xs md:text-sm text-gray-500">Tertunda</p>
            <p id="pending-tasks" class="text-lg md:text-2xl font-bold text-orange-600">0</p>
          </div>
          <div class="icon w-8 h-8 md:w-12 md:h-12 bg-orange-100 rounded-xl flex items-center justify-center">
            <svg class="w-4 h-4 md:w-6 md:h-6 text-orange-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
            </svg>
          </div>
        </div>
      </div>
    </div>

    <!-- Filter and sort -->
    <div class="flex justify-between items-center mb-4 md:mb-6">
      <div class="flex space-x-2">
        <select id="filter-priority" class="px-2 md:px-3 py-2 bg-white border border-gray-200 rounded-lg text-xs md:text-sm focus:outline-none focus:ring-2 focus:ring-blue-500">
          <option value="">Semua Prioritas</option>
          <option value="high">Tinggi</option>
          <option value="medium">Sedang</option>
          <option value="low">Rendah</option>
        </select>
      </div>
      
      <div class="flex space-x-2">
        <button id="sort-date" class="px-2 md:px-3 py-2 bg-white border border-gray-200 rounded-lg text-xs md:text-sm hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors">
          <svg class="w-3 h-3 md:w-4 md:h-4 inline mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z"></path>
          </svg>
          <span class="hidden md:inline">Tanggal</span>
        </button>
        <button id="sort-priority" class="px-2 md:px-3 py-2 bg-white border border-gray-200 rounded-lg text-xs md:text-sm hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors">
          <svg class="w-3 h-3 md:w-4 md:h-4 inline mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 11l5-5m0 0l5 5m-5-5v12"></path>
          </svg>
          <span class="hidden md:inline">Prioritas</span>
        </button>
      </div>
    </div>

    <!-- Task List -->
    <section class="mb-6">
      <ul id="todo-list" class="space-y-3">
        <!-- Tasks inserted by JS -->
      </ul>
      
      <div id="empty-state" class="text-center py-12 hidden">
        <div class="w-24 h-24 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
          <svg class="w-12 h-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v10a2 2 0 002 2h8a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"></path>
          </svg>
        </div>
        <h3 class="text-lg font-medium text-gray-900 mb-2">Belum ada tugas</h3>
        <p class="text-gray-500">Tambahkan tugas pertama Anda untuk memulai</p>
      </div>
    </section>

    <!-- Add new reminder -->
    <div class="fixed bottom-6 right-6 z-10">
      <button id="btn-add-task" class="w-14 h-14 bg-blue-600 hover:bg-blue-700 text-white rounded-full shadow-lg hover:shadow-xl focus:outline-none focus:ring-4 focus:ring-blue-200 transition-all duration-200 flex items-center justify-center">
        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
        </svg>
      </button>
    </div>
  </main>
</div>

<!-- Add/Edit Task Modal -->
<div id="task-modal" class="fixed inset-0 z-50 hidden">
  <div class="modal-backdrop absolute inset-0"></div>
  <div class="relative flex items-center justify-center min-h-screen p-4">
    <div class="bg-white rounded-2xl shadow-2xl w-full max-w-lg slide-up max-h-[90vh] overflow-y-auto">
      <div class="px-6 py-4 border-b border-gray-100">
        <h3 id="task-modal-title" class="text-lg font-semibold text-gray-900">Tambah Tugas Baru</h3>
      </div>
      
      <form id="add-task-form" class="p-6 space-y-4">
        <input type="hidden" id="edit-task-id">
        
        <div>
          <label class="block text-sm font-medium text-gray-700 mb-2">Judul Tugas</label>
          <input type="text" id="task-title" placeholder="Masukkan judul tugas..." class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent">
        </div>
        
        <div>
          <label class="block text-sm font-medium text-gray-700 mb-2">Deskripsi (Opsional)</label>
          <textarea id="task-description" placeholder="Tambahkan deskripsi..." rows="3" class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent resize-none"></textarea>
        </div>
        
        <!-- Subtasks Section -->
        <div>
          <div class="flex items-center justify-between mb-2">
            <label class="block text-sm font-medium text-gray-700">Sub Tugas</label>
            <button type="button" id="add-subtask-btn" class="text-blue-600 hover:text-blue-700 text-sm font-medium">+ Tambah Sub Tugas</button>
          </div>
          <div id="subtasks-container" class="space-y-2">
            <!-- Subtasks will be added here -->
          </div>
        </div>
        
        <div class="grid grid-cols-2 gap-4">
          <div>
            <label class="block text-sm font-medium text-gray-700 mb-2">Kategori</label>
            <select id="task-category" class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500">
              <!-- Categories inserted by JS -->
            </select>
          </div>
          
          <div>
            <label class="block text-sm font-medium text-gray-700 mb-2">Prioritas</label>
            <select id="task-priority" class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500">
              <option value="low">Rendah</option>
              <option value="medium" selected>Sedang</option>
              <option value="high">Tinggi</option>
            </select>
          </div>
        </div>
        
        <div>
          <label class="block text-sm font-medium text-gray-700 mb-2">Tanggal Jatuh Tempo (Opsional)</label>
          <input type="datetime-local" id="task-due-date" class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500">
        </div>
        
        <div class="flex space-x-3 pt-4">
          <button type="button" id="cancel-task" class="flex-1 px-4 py-3 text-gray-600 bg-gray-100 hover:bg-gray-200 rounded-xl transition-colors">Batal</button>
          <button type="submit" id="submit-task-btn" class="flex-1 px-4 py-3 bg-blue-600 hover:bg-blue-700 text-white rounded-xl transition-colors">Tambah Tugas</button>
        </div>
      </form>
    </div>
  </div>
</div>

<!-- Add Category Modal -->
<div id="category-modal" class="fixed inset-0 z-50 hidden">
  <div class="modal-backdrop absolute inset-0"></div>
  <div class="relative flex items-center justify-center min-h-screen p-4">
    <div class="bg-white rounded-2xl shadow-2xl w-full max-w-sm slide-up">
      <div class="px-6 py-4 border-b border-gray-100">
        <h3 class="text-lg font-semibold text-gray-900">Tambah Kategori</h3>
      </div>
      
      <form id="add-category-form" class="p-6 space-y-4">
        <div>
          <label class="block text-sm font-medium text-gray-700 mb-2">Nama Kategori</label>
          <input type="text" id="category-name" placeholder="Masukkan nama kategori..." class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent">
        </div>
        
        <div>
          <label class="block text-sm font-medium text-gray-700 mb-2">Icon</label>
          <input type="text" id="category-icon" placeholder="📁" maxlength="2" class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent text-center text-2xl">
        </div>
        
        <div>
          <label class="block text-sm font-medium text-gray-700 mb-2">Warna</label>
          <input type="color" id="category-color" value="#3b82f6" class="w-full h-12 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500">
        </div>
        
        <div class="flex space-x-3 pt-4">
          <button type="button" id="cancel-category" class="flex-1 px-4 py-3 text-gray-600 bg-gray-100 hover:bg-gray-200 rounded-xl transition-colors">Batal</button>
          <button type="submit" class="flex-1 px-4 py-3 bg-blue-600 hover:bg-blue-700 text-white rounded-xl transition-colors">Tambah</button>
        </div>
      </form>
    </div>
  </div>
</div>

<!-- Search Modal -->
<div id="search-modal" class="fixed inset-0 z-50 hidden">
  <div class="modal-backdrop absolute inset-0"></div>
  <div class="relative flex items-start justify-center min-h-screen p-4 pt-20">
    <div class="bg-white rounded-2xl shadow-2xl w-full max-w-2xl slide-up">
      <div class="px-6 py-4 border-b border-gray-100">
        <div class="flex items-center space-x-3">
          <svg class="w-5 h-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
          </svg>
          <input type="text" id="search-input" placeholder="Cari tugas..." class="flex-1 text-lg focus:outline-none">
          <button id="close-search" class="p-2 hover:bg-gray-100 rounded-lg transition-colors">
            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
            </svg>
          </button>
        </div>
      </div>
      
      <div id="search-results" class="max-h-96 overflow-y-auto p-4">
        <p class="text-gray-500 text-center py-8">Mulai mengetik untuk mencari tugas...</p>
      </div>
    </div>
  </div>
</div>

<script>
// Global variables
let appData = { categories: [], tasks: [] };
let currentCategory = 'cat_all';
let currentSort = 'date';
let currentFilter = '';
let isEditing = false;
let editingTaskId = null;

// API functions
async function fetchData() {
    try {
        const response = await fetch('api.php?action=data');
        if (!response.ok) throw new Error('Failed to fetch data');
        appData = await response.json();
        renderApp();
    } catch (error) {
        console.error('Error fetching data:', error);
        alert('Gagal memuat data. Silakan refresh halaman.');
    }
}

async function saveTask(taskData) {
    try {
        const action = isEditing ? 'update_task' : 'add_task';
        const response = await fetch(`api.php?action=${action}`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(taskData)
        });
        
        if (!response.ok) throw new Error('Failed to save task');
        
        await fetchData();
        return true;
    } catch (error) {
        console.error('Error saving task:', error);
        alert('Gagal menyimpan tugas. Silakan coba lagi.');
        return false;
    }
}

async function toggleTask(taskId) {
    try {
        const response = await fetch('api.php?action=toggle_task', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ id: taskId })
        });
        
        if (!response.ok) throw new Error('Failed to toggle task');
        await fetchData();
    } catch (error) {
        console.error('Error toggling task:', error);
        alert('Gagal mengubah status tugas.');
    }
}

async function toggleSubtask(taskId, subtaskId) {
    try {
        const response = await fetch('api.php?action=toggle_subtask', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ task_id: taskId, subtask_id: subtaskId })
        });
        
        if (!response.ok) throw new Error('Failed to toggle subtask');
        await fetchData();
    } catch (error) {
        console.error('Error toggling subtask:', error);
        alert('Gagal mengubah status sub tugas.');
    }
}

async function deleteTask(taskId) {
    if (!confirm('Apakah Anda yakin ingin menghapus tugas ini?')) return;
    
    try {
        const response = await fetch(`api.php?action=delete_task&id=${taskId}`, {
            method: 'DELETE'
        });
        
        if (!response.ok) throw new Error('Failed to delete task');
        await fetchData();
    } catch (error) {
        console.error('Error deleting task:', error);
        alert('Gagal menghapus tugas.');
    }
}

async function saveCategory(categoryData) {
    try {
        const response = await fetch('api.php?action=add_category', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(categoryData)
        });
        
        if (!response.ok) throw new Error('Failed to save category');
        await fetchData();
        return true;
    } catch (error) {
        console.error('Error saving category:', error);
        alert('Gagal menyimpan kategori. Silakan coba lagi.');
        return false;
    }
}

// Utility functions
function generateId(prefix = 'item') {
    return prefix + '_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
}

function formatDate(dateString) {
    if (!dateString) return '';
    const date = new Date(dateString);
    return date.toLocaleDateString('id-ID', {
        day: 'numeric',
        month: 'short',
        year: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    });
}

function getPriorityText(priority) {
    const priorities = {
        'high': 'Tinggi',
        'medium': 'Sedang',
        'low': 'Rendah'
    };
    return priorities[priority] || 'Sedang';
}

function getCategoryById(categoryId) {
    return appData.categories.find(cat => cat.id === categoryId);
}

function getFilteredTasks() {
    let tasks = appData.tasks;
    
    // Filter by category
    if (currentCategory !== 'cat_all') {
        tasks = tasks.filter(task => task.category === currentCategory);
    }
    
    // Filter by priority
    if (currentFilter) {
        tasks = tasks.filter(task => task.priority === currentFilter);
    }
    
    // Sort tasks - UPDATED: Always put completed tasks at bottom
    tasks.sort((a, b) => {
        // First, separate completed and incomplete tasks
        if (a.completed !== b.completed) {
            return a.completed ? 1 : -1; // Completed tasks go to bottom
        }
        
        // Then sort within each group based on selected sort method
        if (currentSort === 'date') {
            return new Date(b.created_at) - new Date(a.created_at);
        } else if (currentSort === 'priority') {
            const priorityOrder = { 'high': 3, 'medium': 2, 'low': 1 };
            return priorityOrder[b.priority] - priorityOrder[a.priority];
        }
        return 0;
    });
    
    return tasks;
}

// Render functions
function renderCategories() {
    const categoryList = document.getElementById('category-list');
    const taskCategorySelect = document.getElementById('task-category');
    
    categoryList.innerHTML = '';
    taskCategorySelect.innerHTML = '';
    
    appData.categories.forEach(category => {
        // Sidebar category
        const categoryItem = document.createElement('li');
        categoryItem.className = `p-3 rounded-xl cursor-pointer hover:bg-gray-50 transition-colors ${currentCategory === category.id ? 'bg-gray-100 border-l-4' : ''}`;
        categoryItem.style.borderLeftColor = currentCategory === category.id ? category.color : 'transparent';
        categoryItem.dataset.id = category.id;
        
        const taskCount = appData.tasks.filter(task => task.category === category.id).length;
        
        categoryItem.innerHTML = `
            <div class="flex items-center justify-between">
                <div class="flex items-center space-x-3">
                    <div class="w-8 h-8 rounded-lg flex items-center justify-center" style="background-color: ${category.color}20;">
                        <span class="text-sm">${category.icon}</span>
                    </div>
                    <span class="font-medium text-gray-900">${category.name}</span>
                </div>
                <span class="text-white text-xs px-2 py-1 rounded-full font-medium" style="background-color: ${category.color};">${taskCount}</span>
            </div>
        `;
        
        categoryItem.addEventListener('click', () => {
            currentCategory = category.id;
            updatePageTitle();
            renderApp();
        });
        
        categoryList.appendChild(categoryItem);
        
        // Task form category option
        const option = document.createElement('option');
        option.value = category.id;
        option.textContent = category.name;
        taskCategorySelect.appendChild(option);
    });
    
    // Update all category count
    const allCount = document.getElementById('all-count');
    allCount.textContent = appData.tasks.length;
}

function renderTasks() {
    const todoList = document.getElementById('todo-list');
    const emptyState = document.getElementById('empty-state');
    const filteredTasks = getFilteredTasks();
    
    if (filteredTasks.length === 0) {
        todoList.innerHTML = '';
        emptyState.classList.remove('hidden');
        return;
    }
    
    emptyState.classList.add('hidden');
    todoList.innerHTML = '';
    
    filteredTasks.forEach(task => {
        const category = getCategoryById(task.category);
        const taskElement = document.createElement('li');
        taskElement.className = `samsung-card rounded-xl p-4 priority-${task.priority} fade-in ${task.completed ? 'opacity-60' : ''}`;
        taskElement.dataset.taskId = task.id;
        
        const completedSubtasks = task.subtasks ? task.subtasks.filter(st => st.completed).length : 0;
        const totalSubtasks = task.subtasks ? task.subtasks.length : 0;
        const hasSubtasks = totalSubtasks > 0;
        
        taskElement.innerHTML = `
            <div class="flex items-start space-x-3">
                <input type="checkbox" class="custom-checkbox mt-1" ${task.completed ? 'checked' : ''} 
                       onchange="toggleTask('${task.id}')">
                
                <div class="flex-1 min-w-0">
                    <div class="flex items-start justify-between">
                        <div class="flex-1">
                            <h3 class="font-medium text-gray-900 ${task.completed ? 'line-through' : ''}">${task.title}</h3>
                            ${task.description ? `<p class="text-sm text-gray-600 mt-1">${task.description}</p>` : ''}
                            
                            <div class="flex items-center space-x-4 mt-2 text-xs text-gray-500">
                                ${category ? `
                                    <span class="flex items-center space-x-1">
                                        <span>${category.icon}</span>
                                        <span>${category.name}</span>
                                    </span>
                                ` : ''}
                                <span class="px-2 py-1 rounded-full text-xs font-medium ${
                                    task.priority === 'high' ? 'bg-red-100 text-red-700' :
                                    task.priority === 'medium' ? 'bg-yellow-100 text-yellow-700' :
                                    'bg-green-100 text-green-700'
                                }">${getPriorityText(task.priority)}</span>
                                ${task.due_date ? `<span>📅 ${formatDate(task.due_date)}</span>` : ''}
                                ${hasSubtasks ? `<span>📋 ${completedSubtasks}/${totalSubtasks}</span>` : ''}
                            </div>
                        </div>
                        
                        <div class="flex items-center space-x-2 ml-4">
                            ${hasSubtasks ? `
                                <button class="expand-btn p-1 hover:bg-gray-100 rounded transition-colors" 
                                        onclick="toggleTaskExpansion('${task.id}')">
                                    <svg class="w-4 h-4 transform transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
                                    </svg>
                                </button>
                            ` : ''}
                            <button class="p-1 hover:bg-gray-100 rounded transition-colors" 
                                    onclick="editTask('${task.id}')">
                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path>
                                </svg>
                            </button>
                            <button class="p-1 hover:bg-gray-100 rounded transition-colors text-red-600" 
                                    onclick="deleteTask('${task.id}')">
                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                                </svg>
                            </button>
                        </div>
                    </div>
                    
                    ${hasSubtasks ? `
    <div class="subtask-container mt-4">
                            ${task.subtasks.map(subtask => `
                                <div class="subtask-item flex items-center space-x-2">
                                    <input type="checkbox" class="custom-checkbox subtask" ${subtask.completed ? 'checked' : ''} 
                                           onchange="toggleSubtask('${task.id}', '${subtask.id}')">
                                    <span class="text-sm ${subtask.completed ? 'line-through text-gray-500' : 'text-gray-700'}">${subtask.title}</span>
                                </div>
                            `).join('')}
                        </div>
                    ` : ''}
                </div>
            </div>
        `;
        
        todoList.appendChild(taskElement);
    });
}

function renderStats() {
    const totalTasks = appData.tasks.length;
    const completedTasks = appData.tasks.filter(task => task.completed).length;
    const pendingTasks = totalTasks - completedTasks;
    
    document.getElementById('total-tasks').textContent = totalTasks;
    document.getElementById('completed-tasks').textContent = completedTasks;
    document.getElementById('pending-tasks').textContent = pendingTasks;
}

function renderApp() {
    renderCategories();
    renderTasks();
    renderStats();
}

function updatePageTitle() {
    const pageTitle = document.getElementById('page-title');
    if (currentCategory === 'cat_all') {
        pageTitle.textContent = 'Semua Pengingat';
    } else {
        const category = getCategoryById(currentCategory);
        pageTitle.textContent = category ? category.name : 'Kategori';
    }
}

// Task management functions
function toggleTaskExpansion(taskId) {
    const taskElement = document.querySelector(`[data-task-id="${taskId}"]`);
    const subtaskContainer = taskElement.querySelector('.subtask-container');
    const expandBtn = taskElement.querySelector('.expand-btn svg');
    
    if (subtaskContainer.classList.contains('hidden')) {
        subtaskContainer.classList.remove('hidden');
        expandBtn.style.transform = 'rotate(180deg)';
        taskElement.classList.add('task-expanded');
    } else {
        subtaskContainer.classList.add('hidden');
        expandBtn.style.transform = 'rotate(0deg)';
        taskElement.classList.remove('task-expanded');
    }
}

function editTask(taskId) {
    const task = appData.tasks.find(t => t.id === taskId);
    if (!task) return;
    
    isEditing = true;
    editingTaskId = taskId;
    
    document.getElementById('edit-task-id').value = taskId;
    document.getElementById('task-title').value = task.title;
    document.getElementById('task-description').value = task.description || '';
    document.getElementById('task-category').value = task.category;
    document.getElementById('task-priority').value = task.priority;
    document.getElementById('task-due-date').value = task.due_date ? task.due_date.slice(0, 16) : '';
    
    // Load subtasks
    const subtasksContainer = document.getElementById('subtasks-container');
    subtasksContainer.innerHTML = '';
    if (task.subtasks) {
        task.subtasks.forEach(subtask => {
            addSubtaskField(subtask.title, subtask.id);
        });
    }
    
    document.getElementById('task-modal-title').textContent = 'Edit Tugas';
    document.getElementById('submit-task-btn').textContent = 'Update Tugas';
    document.getElementById('task-modal').classList.remove('hidden');
}

function addSubtaskField(title = '', id = null) {
    const container = document.getElementById('subtasks-container');
    const subtaskId = id || generateId('subtask');
    
    const subtaskDiv = document.createElement('div');
    subtaskDiv.className = 'flex items-center space-x-2';
    subtaskDiv.innerHTML = `
        <input type="text" 
               placeholder="Sub tugas..." 
               value="${title}" 
               class="flex-1 px-3 py-2 border border-gray-200 rounded-lg text-sm focus:outline-none focus:ring-2 focus:ring-blue-500"
               data-subtask-id="${subtaskId}"
               enterkeyhint="next"
               inputmode="text"
               autocomplete="off"
               onkeydown="handleSubtaskEnter(event)">
        <button type="button" onclick="removeSubtask(this)" 
                class="p-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors">
            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
            </svg>
        </button>
    `;
    
    container.appendChild(subtaskDiv);
    
    // Focus pada input yang baru dibuat jika tidak ada title (subtask baru)
    if (!title) {
        const newInput = subtaskDiv.querySelector('input');
        setTimeout(() => {
            newInput.focus();
        }, 10);
    }
}

function handleSubtaskEnter(event) {
    if (event.key === 'Enter') {
        event.preventDefault();
        
        const currentInput = event.target;
        const currentValue = currentInput.value.trim();
        
        // Jika input kosong, jangan buat subtask baru
        if (!currentValue) {
            currentInput.focus();
            return;
        }
        
        // Buat subtask baru
        addSubtaskField();
        
        // Focus akan otomatis pindah ke input baru karena addSubtaskField() sudah mengaturnya
    }
}

function removeSubtask(button) {
    const subtaskDiv = button.parentElement;
    const container = document.getElementById('subtasks-container');
    
    // Jika ini adalah satu-satunya subtask dan kosong, jangan hapus
    const allInputs = container.querySelectorAll('input[type="text"]');
    if (allInputs.length === 1 && !allInputs[0].value.trim()) {
        return;
    }
    
    subtaskDiv.remove();
    
    // Jika tidak ada subtask lagi, buat satu input kosong
    if (container.children.length === 0) {
        addSubtaskField();
    }
}

// Search functionality
function performSearch(query) {
    if (!query.trim()) {
        document.getElementById('search-results').innerHTML = '<p class="text-gray-500 text-center py-8">Mulai mengetik untuk mencari tugas...</p>';
        return;
    }
    
    const results = appData.tasks.filter(task => 
        task.title.toLowerCase().includes(query.toLowerCase()) ||
        (task.description && task.description.toLowerCase().includes(query.toLowerCase()))
    );
    
    const resultsContainer = document.getElementById('search-results');
    
    if (results.length === 0) {
        resultsContainer.innerHTML = '<p class="text-gray-500 text-center py-8">Tidak ada tugas yang ditemukan.</p>';
        return;
    }
    
    resultsContainer.innerHTML = results.map(task => {
        const category = getCategoryById(task.category);
        return `
            <div class="p-3 hover:bg-gray-50 rounded-lg cursor-pointer border-b border-gray-100 last:border-b-0"
                 onclick="selectSearchResult('${task.id}')">
                <h4 class="font-medium text-gray-900">${task.title}</h4>
                ${task.description ? `<p class="text-sm text-gray-600 mt-1">${task.description}</p>` : ''}
                <div class="flex items-center space-x-2 mt-2 text-xs text-gray-500">
                    ${category ? `<span>${category.icon} ${category.name}</span>` : ''}
                    <span class="px-2 py-1 rounded-full ${
                        task.priority === 'high' ? 'bg-red-100 text-red-700' :
                        task.priority === 'medium' ? 'bg-yellow-100 text-yellow-700' :
                        'bg-green-100 text-green-700'
                    }">${getPriorityText(task.priority)}</span>
                </div>
            </div>
        `;
    }).join('');
}

function selectSearchResult(taskId) {
    const task = appData.tasks.find(t => t.id === taskId);
    if (task) {
        currentCategory = task.category;
        updatePageTitle();
        renderApp();
        document.getElementById('search-modal').classList.add('hidden');
        
        // Scroll to task
        setTimeout(() => {
            const taskElement = document.querySelector(`[data-task-id="${taskId}"]`);
            if (taskElement) {
                taskElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
                taskElement.style.backgroundColor = '#dbeafe';
                setTimeout(() => {
                    taskElement.style.backgroundColor = '';
                }, 2000);
            }
        }, 100);
    }
}

// Event listeners
document.addEventListener('DOMContentLoaded', function() {
    // Load initial data
    fetchData();
    
    // Mobile menu toggle
    document.getElementById('btn-menu').addEventListener('click', function() {
        document.getElementById('sidebar').classList.remove('-translate-x-full');
        document.getElementById('overlay').classList.remove('hidden');
    });
    
    document.getElementById('btn-close-sidebar').addEventListener('click', function() {
        document.getElementById('sidebar').classList.add('-translate-x-full');
        document.getElementById('overlay').classList.add('hidden');
    });
    
    document.getElementById('overlay').addEventListener('click', function() {
        document.getElementById('sidebar').classList.add('-translate-x-full');
        document.getElementById('overlay').classList.add('hidden');
    });
    
    // All category click
    document.getElementById('all-category').addEventListener('click', function() {
        currentCategory = 'cat_all';
        updatePageTitle();
        renderApp();
    });
    
    // Add task button
    document.getElementById('btn-add-task').addEventListener('click', function() {
        isEditing = false;
        editingTaskId = null;
        document.getElementById('add-task-form').reset();
        document.getElementById('subtasks-container').innerHTML = '';
        document.getElementById('task-modal-title').textContent = 'Tambah Tugas Baru';
        document.getElementById('submit-task-btn').textContent = 'Tambah Tugas';
        document.getElementById('task-modal').classList.remove('hidden');
    });
    
    // Add category button
    document.getElementById('btn-add-category').addEventListener('click', function() {
        document.getElementById('add-category-form').reset();
        document.getElementById('category-modal').classList.remove('hidden');
    });
    
    // Search button
    document.getElementById('btn-search').addEventListener('click', function() {
        document.getElementById('search-modal').classList.remove('hidden');
        document.getElementById('search-input').focus();
    });
    
    // Close search
    document.getElementById('close-search').addEventListener('click', function() {
        document.getElementById('search-modal').classList.add('hidden');
    });
    
    // Search input
    document.getElementById('search-input').addEventListener('input', function() {
        performSearch(this.value);
    });
    
    // Add subtask button
    document.getElementById('add-subtask-btn').addEventListener('click', function() {
    addSubtaskField();
});
    
    // Task form submission
    document.getElementById('add-task-form').addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const title = document.getElementById('task-title').value.trim();
        if (!title) {
            alert('Judul tugas harus diisi.');
            return;
        }
        
        // Collect subtasks
        const subtaskInputs = document.querySelectorAll('#subtasks-container input[type="text"]');
        const subtasks = Array.from(subtaskInputs)
            .map(input => ({
                id: input.dataset.subtaskId,
                title: input.value.trim(),
                completed: false
            }))
            .filter(subtask => subtask.title);
        
        const taskData = {
            title: title,
            description: document.getElementById('task-description').value.trim(),
            category: document.getElementById('task-category').value,
            priority: document.getElementById('task-priority').value,
            due_date: document.getElementById('task-due-date').value || null,
            subtasks: subtasks
        };
        
        if (isEditing) {
            taskData.id = editingTaskId;
        }
        
        const success = await saveTask(taskData);
        if (success) {
            document.getElementById('task-modal').classList.add('hidden');
        }
    });
    
    // Category form submission
    document.getElementById('add-category-form').addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const name = document.getElementById('category-name').value.trim();
        if (!name) {
            alert('Nama kategori harus diisi.');
            return;
        }
        
        const categoryData = {
            name: name,
            icon: document.getElementById('category-icon').value.trim() || '📁',
            color: document.getElementById('category-color').value
        };
        
        const success = await saveCategory(categoryData);
        if (success) {
            document.getElementById('category-modal').classList.add('hidden');
        }
    });
    
    // Cancel buttons
    document.getElementById('cancel-task').addEventListener('click', function() {
        document.getElementById('task-modal').classList.add('hidden');
    });
    
    document.getElementById('cancel-category').addEventListener('click', function() {
        document.getElementById('category-modal').classList.add('hidden');
    });
    
    // Filter and sort
    document.getElementById('filter-priority').addEventListener('change', function() {
        currentFilter = this.value;
        renderTasks();
    });
    
    document.getElementById('sort-date').addEventListener('click', function() {
        currentSort = 'date';
        renderTasks();
    });
    
    document.getElementById('sort-priority').addEventListener('click', function() {
        currentSort = 'priority';
        renderTasks();
    });
    
    // Close modals when clicking backdrop
    document.querySelectorAll('.modal-backdrop').forEach(backdrop => {
        backdrop.addEventListener('click', function() {
            this.closest('.fixed').classList.add('hidden');
        });
    });
    
    // Keyboard shortcuts
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            document.querySelectorAll('.fixed.z-50').forEach(modal => {
                modal.classList.add('hidden');
            });
        }
        
        if (e.ctrlKey && e.key === 'k') {
            e.preventDefault();
            document.getElementById('search-modal').classList.remove('hidden');
            document.getElementById('search-input').focus();
        }
    });
});

// Global functions for onclick handlers
window.toggleTask = toggleTask;
window.toggleSubtask = toggleSubtask;
window.deleteTask = deleteTask;
window.editTask = editTask;
window.toggleTaskExpansion = toggleTaskExpansion;
window.selectSearchResult = selectSearchResult;
</script>

</body>
</html>

