<?php
// Initialize session if not already started
session_start();

// Define JSON file paths
define('MEDIA_PARTNERS_FILE', 'data/media_partners.json');
define('ORGANIZERS_FILE', 'data/organizers.json');
define('EVENTS_FILE', 'data/events.json');
define('PROMOTIONS_FILE', 'data/promotions.json');
define('PINS_FILE', 'data/pins.json');

// Authentication settings
define('ACCESS_PIN', '051205');
define('COOKIE_NAME', 'event_monitor_auth');
define('COOKIE_DURATION', 30 * 24 * 60 * 60); // 30 days

// Check if user is logged in
function isLoggedIn() {
    // Check session
    if (isset($_SESSION['authenticated']) && $_SESSION['authenticated'] === true) {
        return true;
    }
    
    // Check remember me cookie
    if (isset($_COOKIE[COOKIE_NAME])) {
        // Validate the cookie value (you might want to use a more secure approach in production)
        if ($_COOKIE[COOKIE_NAME] === hash('sha256', ACCESS_PIN)) {
            $_SESSION['authenticated'] = true;
            return true;
        }
    }
    
    return false;
}

// Handle login
if (isset($_POST['action']) && $_POST['action'] === 'login') {
    if ($_POST['pin'] === ACCESS_PIN) {
        $_SESSION['authenticated'] = true;
        
        // Set remember me cookie if requested
        if (isset($_POST['remember']) && $_POST['remember'] === 'on') {
            setcookie(COOKIE_NAME, hash('sha256', ACCESS_PIN), time() + COOKIE_DURATION, '/');
        }
        
        // Redirect to the dashboard to avoid form resubmission
        header('Location: index.php');
        exit;
    } else {
        $loginError = 'PIN yang Anda masukkan salah.';
    }
}

// Handle logout
if (isset($_GET['logout'])) {
    // Clear session
    $_SESSION = array();
    session_destroy();
    
    // Clear cookie
    setcookie(COOKIE_NAME, '', time() - 3600, '/');
    
    // Redirect to login page
    header('Location: index.php');
    exit;
}

// Create data directory if it doesn't exist
if (!file_exists('data')) {
    mkdir('data', 0777, true);
}

// Initialize JSON files if they don't exist
function initializeJsonFile($file, $defaultData = []) {
    if (!file_exists($file)) {
        file_put_contents($file, json_encode($defaultData, JSON_PRETTY_PRINT));
    }
}

initializeJsonFile(MEDIA_PARTNERS_FILE);
initializeJsonFile(ORGANIZERS_FILE);
initializeJsonFile(EVENTS_FILE);
initializeJsonFile(PROMOTIONS_FILE);
initializeJsonFile(PINS_FILE);

// Helper functions
function loadData($file) {
    return json_decode(file_get_contents($file), true) ?: [];
}

function saveData($file, $data) {
    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT));
}

function generateId() {
    return uniqid();
}

// Get current page
$page = isset($_GET['page']) ? $_GET['page'] : 'dashboard';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isLoggedIn()) {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'add_media_partner':
                $mediaPartners = loadData(MEDIA_PARTNERS_FILE);
                $newPartner = [
                    'id' => generateId(),
                    'name' => $_POST['name'],
                    'instagram' => $_POST['instagram'],
                    'whatsapp' => $_POST['whatsapp']
                ];
                $mediaPartners[] = $newPartner;
                saveData(MEDIA_PARTNERS_FILE, $mediaPartners);
                header('Location: index.php?page=media_partners');
                exit;
                
            case 'add_organizer':
                $organizers = loadData(ORGANIZERS_FILE);
                $newOrganizer = [
                    'id' => generateId(),
                    'name' => $_POST['name'],
                    'contact' => $_POST['contact']
                ];
                $organizers[] = $newOrganizer;
                saveData(ORGANIZERS_FILE, $organizers);
                header('Location: index.php?page=organizers');
                exit;
                
            case 'add_event':
                $events = loadData(EVENTS_FILE);
                $newEvent = [
                    'id' => generateId(),
                    'name' => $_POST['name'],
                    'organizer_id' => $_POST['organizer_id'],
                    'description' => $_POST['description'],
                    'date' => $_POST['date']
                ];
                $events[] = $newEvent;
                saveData(EVENTS_FILE, $events);
                header('Location: index.php?page=events');
                exit;
                
            case 'add_promotion':
                $promotions = loadData(PROMOTIONS_FILE);
                $postDates = $_POST['post_dates'];
                
                foreach ($postDates as $postDate) {
                    if (!empty($postDate)) {
                        $newPromotion = [
                            'id' => generateId(),
                            'event_id' => $_POST['event_id'],
                            'media_partner_id' => $_POST['media_partner_id'],
                            'post_date' => $postDate,
                            'cost' => $_POST['cost'],
                            'completed' => false
                        ];
                        $promotions[] = $newPromotion;
                    }
                }
                
                saveData(PROMOTIONS_FILE, $promotions);
                header('Location: index.php');
                exit;
                
            case 'add_pin':
                $pins = loadData(PINS_FILE);
                $newPin = [
                    'id' => generateId(),
                    'event_id' => $_POST['event_id'],
                    'media_partner_id' => $_POST['media_partner_id'],
                    'start_date' => $_POST['start_date'],
                    'end_date' => $_POST['end_date'],
                    'completed' => false
                ];
                $pins[] = $newPin;
                saveData(PINS_FILE, $pins);
                header('Location: index.php?page=pins');
                exit;
                
            case 'mark_completed':
                $promotions = loadData(PROMOTIONS_FILE);
                foreach ($promotions as &$promotion) {
                    if ($promotion['id'] === $_POST['promotion_id']) {
                        $promotion['completed'] = true;
                        break;
                    }
                }
                saveData(PROMOTIONS_FILE, $promotions);
                header('Location: index.php');
                exit;
                
            case 'mark_pin_completed':
                $pins = loadData(PINS_FILE);
                foreach ($pins as &$pin) {
                    if ($pin['id'] === $_POST['pin_id']) {
                        $pin['completed'] = true;
                        break;
                    }
                }
                saveData(PINS_FILE, $pins);
                header('Location: index.php?page=pins');
                exit;
        }
    }
}

// Helper function to get entity by ID
function getEntityById($file, $id) {
    $data = loadData($file);
    foreach ($data as $entity) {
        if ($entity['id'] === $id) {
            return $entity;
        }
    }
    return null;
}

// Get current and upcoming promotions
function getCurrentPromotions() {
    $promotions = loadData(PROMOTIONS_FILE);
    $result = [];
    
    foreach ($promotions as $promotion) {
        $postDate = strtotime($promotion['post_date']);
        $now = time();
        
        // Show promotions that are within the next 30 days or past but not completed
        if ($postDate <= $now + (30 * 24 * 60 * 60)) {
            $result[] = $promotion;
        }
    }
    
    // Sort by post date
    usort($result, function($a, $b) {
        return strtotime($a['post_date']) - strtotime($b['post_date']);
    });
    
    return $result;
}

// Get current and upcoming pins
function getCurrentPins() {
    $pins = loadData(PINS_FILE);
    $result = [];
    
    foreach ($pins as $pin) {
        $endDate = strtotime($pin['end_date']);
        $now = time();
        
        // Show pins that are not yet ended or not completed
        if ($endDate >= $now || !$pin['completed']) {
            $result[] = $pin;
        }
    }
    
    // Sort by end date
    usort($result, function($a, $b) {
        return strtotime($a['end_date']) - strtotime($b['end_date']);
    });
    
    return $result;
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="robots" content="noindex, nofollow">
    <link rel="icon" href="https://app.dezki.biz.id/linkpic/assets/uploads/67dc7f76c30e0.png" type="image/png">
    <title>PostSync - Sistem Monitoring Media Partner</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        :root {
            --primary: #e60000;
            --secondary: #000000;
            --light: #ffffff;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #f8f9fa;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
        }
        
        .navbar {
            background-color: var(--primary);
        }
        
        .navbar-brand {
            color: var(--light) !important;
            font-weight: bold;
        }
        
        .nav-link {
            color: var(--light) !important;
        }
        
        .nav-link:hover {
            color: rgba(255, 255, 255, 0.8) !important;
        }
        
        .btn-primary {
            background-color: var(--primary);
            border-color: var(--primary);
        }
        
        .btn-primary:hover {
            background-color: #b80000;
            border-color: #b80000;
        }
        
        .btn-secondary {
            background-color: var(--secondary);
            border-color: var(--secondary);
        }
        
        .btn-secondary:hover {
            background-color: #333333;
            border-color: #333333;
        }
        
        .card {
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            margin-bottom: 20px;
        }
        
        .card-header {
            background-color: var(--primary);
            color: var(--light);
            font-weight: bold;
        }
        
        .promotion-card {
            border-left: 5px solid var(--primary);
        }
        
        .completed {
            background-color: #e8f5e9;
            border-left: 5px solid #4caf50;
        }
        
        .overdue {
            background-color: #ffebee;
            border-left: 5px solid #f44336;
        }
        
        .upcoming {
            background-color: #e3f2fd;
            border-left: 5px solid #2196f3;
        }
        
        .today {
            background-color: #fff8e1;
            border-left: 5px solid #ffc107;
        }
        
        .login-form {
            max-width: 400px;
            margin: 100px auto;
        }
        
        .main-content {
            flex: 1;
        }
        
        footer {
            margin-top: auto;
            background-color: var(--secondary);
            color: var(--light);
            text-align: center;
            padding: 1rem 0;
            width: 100%;
        }
        
        /* Responsive adjustments */
        @media (max-width: 768px) {
            .container {
                padding: 10px;
            }
            
            .card {
                margin-bottom: 15px;
            }
            
            .btn-sm {
                padding: 0.25rem 0.5rem;
                font-size: 0.75rem;
            }
            
            .login-form {
                margin: 50px auto;
                padding: 0 15px;
            }
        }
    </style>
</head>
<body>
    <?php if (!isLoggedIn()): ?>
        <!-- Login Form -->
        <div class="container">
            <div class="login-form">
                <div class="card">
                    <div class="card-header">
                        <h4 class="mb-0">Masuk ke Sistem</h4>
                    </div>
                    <div class="card-body">
                        <?php if (isset($loginError)): ?>
                            <div class="alert alert-danger"><?= $loginError ?></div>
                        <?php endif; ?>
                        
                        <form method="post">
                            <input type="hidden" name="action" value="login">
                            <div class="mb-3">
                                <label for="pin" class="form-label">PIN Akses</label>
                                <input type="password" class="form-control" id="pin" name="pin" required>
                            </div>
                            <div class="mb-3 form-check">
                                <input type="checkbox" class="form-check-input" id="remember" name="remember">
                                <label class="form-check-label" for="remember">Ingat Saya</label>
                            </div>
                            <button type="submit" class="btn btn-primary w-100">Masuk</button>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    <?php else: ?>
        <!-- Navigation -->
        <nav class="navbar navbar-expand-lg navbar-dark">
            <div class="container">
                <a class="navbar-brand" href="index.php">PostSync - Monitoring Media Partner</a>
                <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                    <span class="navbar-toggler-icon"></span>
                </button>
                <div class="collapse navbar-collapse" id="navbarNav">
                    <ul class="navbar-nav ms-auto">
                        <li class="nav-item">
                            <a class="nav-link <?= $page === 'dashboard' ? 'active' : '' ?>" href="index.php">Dashboard</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link <?= $page === 'media_partners' ? 'active' : '' ?>" href="index.php?page=media_partners">Media Partner</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link <?= $page === 'organizers' ? 'active' : '' ?>" href="index.php?page=organizers">Penyelenggara</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link <?= $page === 'events' ? 'active' : '' ?>" href="index.php?page=events">Event</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link <?= $page === 'add_promotion' ? 'active' : '' ?>" href="index.php?page=add_promotion">Tambah Promosi</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link <?= $page === 'pins' ? 'active' : '' ?>" href="index.php?page=pins">Pin Event</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="index.php?logout=1">Keluar</a>
                        </li>
                    </ul>
                </div>
            </div>
        </nav>

        <!-- Main Content -->
        <div class="container my-4 main-content">
            <?php if ($page === 'dashboard'): ?>
                <div class="row">
                    <div class="col-12">
                        <div class="card">
                            <div class="card-header d-flex justify-content-between align-items-center">
                                <h5 class="mb-0">Jadwal Posting Media Partner</h5>
                                <a href="index.php?page=add_promotion" class="btn btn-sm btn-light">+ Tambah Promosi</a>
                            </div>
                            <div class="card-body">
                                <?php
                                $promotions = getCurrentPromotions();
                                
                                if (empty($promotions)) {
                                    echo '<div class="alert alert-info">Belum ada jadwal promosi.</div>';
                                } else {
                                    foreach ($promotions as $promotion) {
                                        $event = getEntityById(EVENTS_FILE, $promotion['event_id']);
                                        $mediaPartner = getEntityById(MEDIA_PARTNERS_FILE, $promotion['media_partner_id']);
                                        $organizer = getEntityById(ORGANIZERS_FILE, $event['organizer_id']);
                                        
                                        $postDate = strtotime($promotion['post_date']);
                                        $now = time();
                                        $dayDiff = floor(($postDate - $now) / (60 * 60 * 24));
                                        
                                        $cardClass = 'promotion-card';
                                        $statusText = '';
                                        
                                        if ($promotion['completed']) {
                                            $cardClass = 'completed';
                                            $statusText = '<span class="badge bg-success">Sudah Diposting</span>';
                                        } elseif ($dayDiff < 0) {
                                            $cardClass = 'overdue';
                                            $statusText = '<span class="badge bg-danger">Terlambat</span>';
                                        } elseif ($dayDiff === 0) {
                                            $cardClass = 'today';
                                            $statusText = '<span class="badge bg-warning">Hari Ini</span>';
                                        } else {
                                            $cardClass = 'upcoming';
                                            $statusText = '<span class="badge bg-primary">Dalam ' . $dayDiff . ' hari</span>';
                                        }
                                        ?>
                                        <div class="card mb-3 <?= $cardClass ?>">
                                            <div class="card-body">
                                                <div class="row">
                                                    <div class="col-md-8">
                                                        <h5><?= htmlspecialchars($event['name']) ?> <?= $statusText ?></h5>
                                                        <p class="mb-1"><strong>Tanggal Post:</strong> <?= date('d/m/Y', $postDate) ?></p>
                                                        <p class="mb-1"><strong>Media Partner:</strong> <?= htmlspecialchars($mediaPartner['name']) ?></p>
                                                        <p class="mb-1"><strong>Penyelenggara:</strong> <?= htmlspecialchars($organizer['name']) ?></p>
                                                        <p class="mb-1"><strong>Biaya:</strong> Rp <?= number_format($promotion['cost'], 0, ',', '.') ?></p>
                                                    </div>
                                                    <div class="col-md-4 text-md-end mt-3 mt-md-0">
                                                        <a href="https://instagram.com/<?= htmlspecialchars(ltrim($mediaPartner['instagram'], '@')) ?>" target="_blank" class="btn btn-sm btn-outline-secondary mb-2">
                                                            <i class="bi bi-instagram"></i> Lihat Instagram
                                                        </a>
                                                        <a href="https://wa.me/<?= htmlspecialchars(ltrim($mediaPartner['whatsapp'], '+')) ?>" target="_blank" class="btn btn-sm btn-outline-success mb-2">
                                                            <i class="bi bi-whatsapp"></i> Hubungi WhatsApp
                                                        </a>
                                                        <?php if (!$promotion['completed']): ?>
                                                            <form method="post" class="d-inline">
                                                                <input type="hidden" name="action" value="mark_completed">
                                                                <input type="hidden" name="promotion_id" value="<?= $promotion['id'] ?>">
                                                                <button type="submit" class="btn btn-sm btn-primary">
                                                                    <i class="bi bi-check-circle"></i> Tandai Sudah
                                                                </button>
                                                            </form>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <?php
                                    }
                                }
                                ?>
                            </div>
                        </div>
                    </div>
                </div>
                
            <?php elseif ($page === 'media_partners'): ?>
                <div class="row">
                    <div class="col-md-8">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">Daftar Media Partner</h5>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-striped">
                                        <thead>
                                            <tr>
                                                <th>Nama</th>
                                                <th>Instagram</th>
                                                <th>WhatsApp</th>
                                                <th>Aksi</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php
                                            $mediaPartners = loadData(MEDIA_PARTNERS_FILE);
                                            if (empty($mediaPartners)) {
                                                echo '<tr><td colspan="4" class="text-center">Belum ada media partner.</td></tr>';
                                            } else {
                                                foreach ($mediaPartners as $partner) {
                                                    ?>
                                                    <tr>
                                                        <td><?= htmlspecialchars($partner['name']) ?></td>
                                                        <td><a href="https://instagram.com/<?= htmlspecialchars(ltrim($partner['instagram'], '@')) ?>" target="_blank"><?= htmlspecialchars($partner['instagram']) ?></a></td>
                                                        <td><a href="https://wa.me/<?= htmlspecialchars(ltrim($partner['whatsapp'], '+')) ?>" target="_blank"><?= htmlspecialchars($partner['whatsapp']) ?></a></td>
                                                        <td>
                                                            <a href="index.php?page=add_promotion&media_partner_id=<?= $partner['id'] ?>" class="btn btn-sm btn-outline-primary">Jadwalkan</a>
                                                        </td>
                                                    </tr>
                                                    <?php
                                                }
                                            }
                                            ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">Tambah Media Partner</h5>
                            </div>
                            <div class="card-body">
                                <form method="post">
                                    <input type="hidden" name="action" value="add_media_partner">
                                    <div class="mb-3">
                                        <label for="name" class="form-label">Nama</label>
                                        <input type="text" class="form-control" id="name" name="name" required>
                                    </div>
                                    <div class="mb-3">
                                        <label for="instagram" class="form-label">Instagram</label>
                                        <input type="text" class="form-control" id="instagram" name="instagram" placeholder="@username" required>
                                    </div>
                                    <div class="mb-3">
                                        <label for="whatsapp" class="form-label">WhatsApp</label>
                                        <input type="text" class="form-control" id="whatsapp" name="whatsapp" placeholder="628xxxxxxxxxx" required>
                                    </div>
                                    <button type="submit" class="btn btn-primary">Simpan</button>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
                
            <?php elseif ($page === 'organizers'): ?>
                <div class="row">
                    <div class="col-md-8">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">Daftar Penyelenggara</h5>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-striped">
                                        <thead>
                                            <tr>
                                                <th>Nama</th>
                                                <th>Kontak</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php
                                            $organizers = loadData(ORGANIZERS_FILE);
                                            if (empty($organizers)) {
                                                echo '<tr><td colspan="2" class="text-center">Belum ada penyelenggara.</td></tr>';
                                            } else {
                                                foreach ($organizers as $organizer) {
                                                    ?>
                                                    <tr>
                                                        <td><?= htmlspecialchars($organizer['name']) ?></td>
                                                        <td><?= htmlspecialchars($organizer['contact']) ?></td>
                                                    </tr>
                                                    <?php
                                                }
                                            }
                                            ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">Tambah Penyelenggara</h5>
                            </div>
                            <div class="card-body">
                                <form method="post">
                                    <input type="hidden" name="action" value="add_organizer">
                                    <div class="mb-3">
                                        <label for="name" class="form-label">Nama</label>
                                        <input type="text" class="form-control" id="name" name="name" required>
                                    </div>
                                    <div class="mb-3">
                                        <label for="contact" class="form-label">Kontak</label>
                                        <input type="text" class="form-control" id="contact" name="contact" required>
                                    </div>
                                    <button type="submit" class="btn btn-primary">Simpan</button>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
                
            <?php elseif ($page === 'events'): ?>
                <div class="row">
                    <div class="col-md-8">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">Daftar Event</h5>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-striped">
                                        <thead>
                                            <tr>
                                                <th>Nama Event</th>
                                                <th>Penyelenggara</th>
                                                <th>Tanggal</th>
                                                <th>Aksi</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php
                                            $events = loadData(EVENTS_FILE);
                                            $organizers = loadData(ORGANIZERS_FILE);
                                            
                                            if (empty($events)) {
                                                echo '<tr><td colspan="4" class="text-center">Belum ada event.</td></tr>';
                                            } else {
                                                foreach ($events as $event) {
                                                    $organizerName = '';
                                                    foreach ($organizers as $organizer) {
                                                        if ($organizer['id'] === $event['organizer_id']) {
                                                            $organizerName = $organizer['name'];
                                                            break;
                                                        }
                                                    }
                                                    ?>
                                                    <tr>
                                                        <td><?= htmlspecialchars($event['name']) ?></td>
                                                        <td><?= htmlspecialchars($organizerName) ?></td>
                                                        <td><?= htmlspecialchars($event['date']) ?></td>
                                                        <td>
                                                            <a href="index.php?page=add_promotion&event_id=<?= $event['id'] ?>" class="btn btn-sm btn-outline-primary">Promosikan</a>
                                                        </td>
                                                    </tr>
                                                    <?php
                                                }
                                            }
                                            ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">Tambah Event</h5>
                            </div>
                            <div class="card-body">
                                <form method="post">
                                    <input type="hidden" name="action" value="add_event">
                                    <div class="mb-3">
                                        <label for="name" class="form-label">Nama Event</label>
                                        <input type="text" class="form-control" id="name" name="name" required>
                                    </div>
                                    <div class="mb-3">
                                        <label for="organizer_id" class="form-label">Penyelenggara</label>
                                        <select class="form-select" id="organizer_id" name="organizer_id" required>
                                            <option value="">Pilih Penyelenggara</option>
                                            <?php
                                            $organizers = loadData(ORGANIZERS_FILE);
                                            foreach ($organizers as $organizer) {
                                                echo '<option value="' . $organizer['id'] . '">' . htmlspecialchars($organizer['name']) . '</option>';
                                            }
                                            ?>
                                        </select>
                                    </div>
                                    <div class="mb-3">
                                        <label for="description" class="form-label">Deskripsi</label>
                                        <textarea class="form-control" id="description" name="description" rows="3"></textarea>
                                    </div>
                                    <div class="mb-3">
                                        <label for="date" class="form-label">Tanggal Event</label>
                                        <input type="date" class="form-control" id="date" name="date" required>
                                    </div>
                                    <button type="submit" class="btn btn-primary">Simpan</button>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
                
            <?php elseif ($page === 'add_promotion'): ?>
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Tambah Jadwal Promosi</h5>
                    </div>
                    <div class="card-body">
                        <form method="post">
                            <input type="hidden" name="action" value="add_promotion">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="event_id" class="form-label">Event</label>
                                    <select class="form-select" id="event_id" name="event_id" required>
                                        <option value="">Pilih Event</option>
                                        <?php
                                        $events = loadData(EVENTS_FILE);
                                        $preselectedEventId = isset($_GET['event_id']) ? $_GET['event_id'] : '';
                                        
                                        foreach ($events as $event) {
                                            $selected = ($event['id'] === $preselectedEventId) ? 'selected' : '';
                                            echo '<option value="' . $event['id'] . '" ' . $selected . '>' . htmlspecialchars($event['name']) . '</option>';
                                        }
                                        ?>
                                    </select>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="media_partner_id" class="form-label">Media Partner</label>
                                    <select class="form-select" id="media_partner_id" name="media_partner_id" required>
                                        <option value="">Pilih Media Partner</option>
                                        <?php
                                        $mediaPartners = loadData(MEDIA_PARTNERS_FILE);
                                        $preselectedPartnerId = isset($_GET['media_partner_id']) ? $_GET['media_partner_id'] : '';
                                        
                                        foreach ($mediaPartners as $partner) {
                                            $selected = ($partner['id'] === $preselectedPartnerId) ? 'selected' : '';
                                            echo '<option value="' . $partner['id'] . '" ' . $selected . '>' . htmlspecialchars($partner['name']) . '</option>';
                                        }
                                        ?>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Tanggal Posting</label>
                                <div id="post-dates-container">
                                    <div class="mb-2 row">
                                        <div class="col-md-6">
                                            <input type="date" class="form-control" name="post_dates[]" required>
                                        </div>
                                        <div class="col-md-6">
                                            <button type="button" class="btn btn-sm btn-outline-secondary add-date">+ Tambah Tanggal</button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="cost" class="form-label">Biaya Promosi</label>
                                <div class="input-group">
                                    <span class="input-group-text">Rp</span>
                                    <input type="number" class="form-control" id="cost" name="cost" required>
                                </div>
                            </div>
                            
                            <button type="submit" class="btn btn-primary">Simpan Jadwal</button>
                        </form>
                    </div>
                </div>
                
            <?php elseif ($page === 'pins'): ?>
                <div class="row mb-4">
                    <div class="col-md-8">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">Daftar Pin Event</h5>
                            </div>
                            <div class="card-body">
                                <?php
                                $pins = getCurrentPins();
                                
                                if (empty($pins)) {
                                    echo '<div class="alert alert-info">Belum ada pin event.</div>';
                                } else {
                                    foreach ($pins as $pin) {
                                        $event = getEntityById(EVENTS_FILE, $pin['event_id']);
                                        $mediaPartner = getEntityById(MEDIA_PARTNERS_FILE, $pin['media_partner_id']);
                                        
                                        $startDate = strtotime($pin['start_date']);
                                        $endDate = strtotime($pin['end_date']);
                                        $now = time();
                                        
                                        $cardClass = 'promotion-card';
                                        $statusText = '';
                                        
                                        if ($pin['completed']) {
                                            $cardClass = 'completed';
                                            $statusText = '<span class="badge bg-success">Selesai</span>';
                                        } elseif ($now < $startDate) {
                                            $cardClass = 'upcoming';
                                            $statusText = '<span class="badge bg-primary">Akan Datang</span>';
                                        } elseif ($now > $endDate) {
                                            $cardClass = 'overdue';
                                            $statusText = '<span class="badge bg-danger">Sudah Berakhir</span>';
                                        } else {
                                            $cardClass = 'today';
                                            $statusText = '<span class="badge bg-warning">Aktif</span>';
                                        }
                                        ?>
                                        <div class="card mb-3 <?= $cardClass ?>">
                                            <div class="card-body">
                                                <div class="row">
                                                    <div class="col-md-8">
                                                        <h5><?= htmlspecialchars($event['name']) ?> <?= $statusText ?></h5>
                                                        <p class="mb-1"><strong>Periode:</strong> <?= date('d/m/Y', $startDate) ?> - <?= date('d/m/Y', $endDate) ?></p>
                                                        <p class="mb-1"><strong>Media Partner:</strong> <?= htmlspecialchars($mediaPartner['name']) ?></p>
                                                    </div>
                                                    <div class="col-md-4 text-md-end mt-3 mt-md-0">
                                                        <a href="https://instagram.com/<?= htmlspecialchars(ltrim($mediaPartner['instagram'], '@')) ?>" target="_blank" class="btn btn-sm btn-outline-secondary mb-2">
                                                            <i class="bi bi-instagram"></i> Lihat Instagram
                                                        </a>
                                                        <?php if (!$pin['completed']): ?>
                                                            <form method="post" class="d-inline">
                                                                <input type="hidden" name="action" value="mark_pin_completed">
                                                                <input type="hidden" name="pin_id" value="<?= $pin['id'] ?>">
                                                                <button type="submit" class="btn btn-sm btn-primary">
                                                                    <i class="bi bi-check-circle"></i> Tandai Selesai
                                                                </button>
                                                            </form>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <?php
                                    }
                                }
                                ?>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">Tambah Pin Event</h5>
                            </div>
                            <div class="card-body">
                                <form method="post">
                                    <input type="hidden" name="action" value="add_pin">
                                    <div class="mb-3">
                                        <label for="pin_event_id" class="form-label">Event</label>
                                        <select class="form-select" id="pin_event_id" name="event_id" required>
                                            <option value="">Pilih Event</option>
                                            <?php
                                            $events = loadData(EVENTS_FILE);
                                            foreach ($events as $event) {
                                                echo '<option value="' . $event['id'] . '">' . htmlspecialchars($event['name']) . '</option>';
                                            }
                                            ?>
                                        </select>
                                    </div>
                                    <div class="mb-3">
                                        <label for="pin_media_partner_id" class="form-label">Media Partner</label>
                                        <select class="form-select" id="pin_media_partner_id" name="media_partner_id" required>
                                            <option value="">Pilih Media Partner</option>
                                            <?php
                                            $mediaPartners = loadData(MEDIA_PARTNERS_FILE);
                                            foreach ($mediaPartners as $partner) {
                                                echo '<option value="' . $partner['id'] . '">' . htmlspecialchars($partner['name']) . '</option>';
                                            }
                                            ?>
                                        </select>
                                    </div>
                                    <div class="mb-3">
                                        <label for="start_date" class="form-label">Tanggal Mulai</label>
                                        <input type="date" class="form-control" id="start_date" name="start_date" required>
                                    </div>
                                    <div class="mb-3">
                                        <label for="end_date" class="form-label">Tanggal Berakhir</label>
                                        <input type="date" class="form-control" id="end_date" name="end_date" required>
                                    </div>
                                    <button type="submit" class="btn btn-primary">Simpan</button>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    <?php endif; ?>

    <!-- Footer -->
    <footer>
        <p class="mb-0">Sistem Monitoring Media Partner &copy; <?= date('Y') ?></p>
    </footer>

    <!-- JavaScript -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Handle adding multiple post dates
            const addDateButton = document.querySelector('.add-date');
            const postDatesContainer = document.getElementById('post-dates-container');
            
            if (addDateButton && postDatesContainer) {
                addDateButton.addEventListener('click', function() {
                    const dateRow = document.createElement('div');
                    dateRow.className = 'mb-2 row';
                    dateRow.innerHTML = `
                        <div class="col-md-6">
                            <input type="date" class="form-control" name="post_dates[]">
                        </div>
                        <div class="col-md-6">
                            <button type="button" class="btn btn-sm btn-outline-danger remove-date">Hapus</button>
                        </div>
                    `;
                    postDatesContainer.appendChild(dateRow);
                    
                    // Add event listener to the remove button
                    const removeButton = dateRow.querySelector('.remove-date');
                    removeButton.addEventListener('click', function() {
                        postDatesContainer.removeChild(dateRow);
                    });
                });
            }
        });
    </script>
</body>
</html>
