/**
 * Aplikasi Pengelola Utang Piutang - app.js
 * JavaScript untuk fungsionalitas interaktif aplikasi
 */

document.addEventListener('DOMContentLoaded', function() {
    // Deteksi tema preferensi sistem
    initThemeDetection();
    
    // Form validations
    initFormValidations();
    
    // Format currency inputs
    initCurrencyInputs();
    
    // Handle sidebar mobile toggle
    initSidebarToggle();
    
    // Enable responsive tables
    initResponsiveTables();
});

/**
 * Deteksi tema berdasarkan preferensi sistem
 */
function initThemeDetection() {
    // Gunakan theme yang sudah diatur di HTML via PHP (data-theme)
    const currentTheme = document.documentElement.getAttribute('data-theme');
    
    // Jika tema diatur ke auto, dengarkan perubahan preferensi sistem
    if (currentTheme === 'auto') {
        // Cek apakah browser mendukung prefers-color-scheme
        if (window.matchMedia) {
            // Dengarkan perubahan preferensi tema
            const prefersDarkScheme = window.matchMedia('(prefers-color-scheme: dark)');
            
            // Set tema awal
            document.documentElement.setAttribute('data-theme', 
                prefersDarkScheme.matches ? 'dark' : 'light');
            
            // Update tema saat preferensi berubah
            prefersDarkScheme.addEventListener('change', (e) => {
                document.documentElement.setAttribute('data-theme', 
                    e.matches ? 'dark' : 'light');
            });
        }
    }
}

/**
 * Inisialisasi validasi form
 */
function initFormValidations() {
    const forms = document.querySelectorAll('form');
    
    forms.forEach(form => {
        form.addEventListener('submit', function(event) {
            // Cek apakah form memiliki required fields yang belum diisi
            const requiredFields = form.querySelectorAll('[required]');
            let isValid = true;
            
            requiredFields.forEach(field => {
                if (!field.value.trim()) {
                    isValid = false;
                    // Tambahkan class error
                    field.classList.add('is-invalid');
                    
                    // Tampilkan pesan error jika belum ada
                    if (!field.nextElementSibling || !field.nextElementSibling.classList.contains('error-message')) {
                        const errorMessage = document.createElement('div');
                        errorMessage.className = 'error-message text-danger';
                        errorMessage.textContent = 'Bidang ini wajib diisi';
                        field.parentNode.insertBefore(errorMessage, field.nextElementSibling);
                    }
                } else {
                    // Hapus class error dan pesan error
                    field.classList.remove('is-invalid');
                    if (field.nextElementSibling && field.nextElementSibling.classList.contains('error-message')) {
                        field.parentNode.removeChild(field.nextElementSibling);
                    }
                }
            });
            
            // Hentikan submit jika form tidak valid
            if (!isValid) {
                event.preventDefault();
            }
        });
    });
    
    // Validasi real-time saat input berubah
    document.addEventListener('input', function(event) {
        const input = event.target;
        
        if (input.hasAttribute('required')) {
            if (!input.value.trim()) {
                input.classList.add('is-invalid');
            } else {
                input.classList.remove('is-invalid');
                if (input.nextElementSibling && input.nextElementSibling.classList.contains('error-message')) {
                    input.parentNode.removeChild(input.nextElementSibling);
                }
            }
        }
    });
}

/**
 * Inisialisasi format input currency
 */
function initCurrencyInputs() {
    // Format number dengan separasi ribuan
    document.addEventListener('input', function(event) {
        if (event.target.classList.contains('currency-input')) {
            const value = event.target.value.replace(/\D/g, '');
            
            if (value === '') {
                event.target.value = '';
                return;
            }
            
            // Format dengan separator ribuan
            event.target.value = formatNumber(value);
        }
    });
    
    // Format awal ketika halaman dimuat
    document.querySelectorAll('.currency-input').forEach(input => {
        const value = input.value.replace(/\D/g, '');
        if (value !== '') {
            input.value = formatNumber(value);
        }
    });
}

/**
 * Format angka dengan separator ribuan
 * @param {string} number - Angka yang akan diformat
 * @returns {string} Angka yang telah diformat
 */
function formatNumber(number) {
    return number.toString().replace(/\B(?=(\d{3})+(?!\d))/g, '.');
}

/**
 * Inisialisasi toggle sidebar untuk tampilan mobile
 */
function initSidebarToggle() {
    const toggleButton = document.querySelector('.sidebar-toggle');
    const sidebar = document.querySelector('.sidebar');
    
    if (toggleButton && sidebar) {
        toggleButton.addEventListener('click', function() {
            sidebar.classList.toggle('active');
        });
        
        // Tutup sidebar saat klik di luar sidebar
        document.addEventListener('click', function(event) {
            if (!sidebar.contains(event.target) && event.target !== toggleButton) {
                sidebar.classList.remove('active');
            }
        });
    }
}

/**
 * Inisialisasi tabel responsif
 */
function initResponsiveTables() {
    const tables = document.querySelectorAll('.table-responsive table');
    
    tables.forEach(table => {
        // Buat div untuk setiap sel pada mobile view
        const headerTexts = [];
        const headerCells = table.querySelectorAll('thead th');
        
        // Ambil teks header
        headerCells.forEach(headerCell => {
            headerTexts.push(headerCell.textContent.trim());
        });
        
        // Tambahkan data label untuk tiap sel
        const rows = table.querySelectorAll('tbody tr');
        rows.forEach(row => {
            const cells = row.querySelectorAll('td');
            cells.forEach((cell, index) => {
                if (index < headerTexts.length) {
                    cell.setAttribute('data-label', headerTexts[index]);
                }
            });
        });
    });
}

/**
 * Menampilkan konfirmasi sebelum menghapus data
 * @param {string} message - Pesan konfirmasi
 * @returns {boolean} True jika user mengonfirmasi, false jika membatalkan
 */
function confirmDelete(message) {
    return confirm(message || 'Apakah Anda yakin ingin menghapus data ini?');
}

/**
 * Handle untuk form transaksi khususnya entitas selection
 * @param {string} prefix - Prefix form (from/to)
 * @param {string} value - Nilai tipe entitas
 */
function handleEntityTypeChange(prefix, value) {
    // Sembunyikan semua pilihan entitas untuk prefix ini
    document.querySelectorAll(`.entity-select[id^="${prefix}_"]`).forEach(el => {
        el.style.display = 'none';
    });
    
    // Tampilkan yang sesuai dengan pilihan
    if (value === 'business') {
        document.getElementById(`${prefix}_business_group`).style.display = '';
    } else if (value === 'external') {
        document.getElementById(`${prefix}_external_group`).style.display = '';
    }
}

/**
 * Fungsi untuk menampilkan notifikasi toast
 * @param {string} message - Pesan notifikasi
 * @param {string} type - Tipe notifikasi (success, error, warning, info)
 */
function showNotification(message, type = 'info') {
    // Cek apakah container notifikasi sudah ada
    let notifContainer = document.querySelector('.notification-container');
    
    if (!notifContainer) {
        // Buat container notifikasi jika belum ada
        notifContainer = document.createElement('div');
        notifContainer.className = 'notification-container';
        document.body.appendChild(notifContainer);
    }
    
    // Buat elemen notifikasi
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <span class="notification-message">${message}</span>
        </div>
        <button class="notification-close">&times;</button>
    `;
    
    // Tambahkan ke container
    notifContainer.appendChild(notification);
    
    // Atur timeout untuk menghapus notifikasi
    setTimeout(() => {
        notification.classList.add('show');
        setTimeout(() => {
            notification.classList.remove('show');
            setTimeout(() => {
                notification.remove();
            }, 300);
        }, 5000);
    }, 10);
    
    // Tambahkan event handler untuk tombol close
    notification.querySelector('.notification-close').addEventListener('click', () => {
        notification.classList.remove('show');
        setTimeout(() => {
            notification.remove();
        }, 300);
    });
}

/**
 * Toggle mode tampilan transaksi antara tabel dan daftar
 */
function toggleTransactionView() {
    const transactionList = document.querySelector('.transaction-list');
    if (transactionList) {
        transactionList.classList.toggle('grid-view');
    }
}

/**
 * Export tabel ke file CSV
 * @param {HTMLElement} table - Elemen tabel yang akan diexport
 * @param {string} filename - Nama file CSV
 */
function exportTableToCSV(table, filename = 'data.csv') {
    const rows = table.querySelectorAll('tr');
    let csv = [];
    
    for (let i = 0; i < rows.length; i++) {
        const row = [], cols = rows[i].querySelectorAll('td, th');
        
        for (let j = 0; j < cols.length; j++) {
            // Bersihkan teks dari HTML
            let text = cols[j].innerText.replace(/(\r\n|\n|\r)/gm, ' ').replace(/,/g, ';');
            row.push('"' + text + '"');
        }
        
        csv.push(row.join(','));
    }
    
    // Download CSV file
    downloadCSV(csv.join('\n'), filename);
}

/**
 * Download file CSV
 * @param {string} csv - Isi file CSV
 * @param {string} filename - Nama file
 */
function downloadCSV(csv, filename) {
    const csvFile = new Blob([csv], {type: 'text/csv'});
    const downloadLink = document.createElement('a');
    
    // Set properti link
    downloadLink.download = filename;
    downloadLink.href = window.URL.createObjectURL(csvFile);
    downloadLink.style.display = 'none';
    
    // Tambahkan ke DOM dan trigger download
    document.body.appendChild(downloadLink);
    downloadLink.click();
    document.body.removeChild(downloadLink);
}