<?php
/**
 * Fungsi-fungsi Utama Sistem Pengelola Event
 */

require_once __DIR__ . '/config.php';

class EventManager {
    
    /**
     * Muat data event dari file
     */
    public static function loadEvents() {
        try {
            if (!file_exists(DATA_FILE)) {
                return [];
            }
            
            $data = file_get_contents(DATA_FILE);
            $decoded = json_decode($data, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                Config::log('JSON decode error: ' . json_last_error_msg(), 'ERROR');
                return [];
            }
            
            return $decoded['events'] ?? [];
        } catch (Exception $e) {
            Config::log('Error loading events: ' . $e->getMessage(), 'ERROR');
            return [];
        }
    }
    
    /**
     * Simpan data event ke file
     */
    public static function saveEvents($events) {
        try {
            $data = [
                'events' => $events,
                'updated_at' => date(DATETIME_FORMAT),
                'version' => APP_VERSION
            ];
            
            $result = file_put_contents(DATA_FILE, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
            
            if ($result === false) {
                throw new Exception('Failed to write to data file');
            }
            
            Config::log('Events saved successfully');
            return true;
        } catch (Exception $e) {
            Config::log('Error saving events: ' . $e->getMessage(), 'ERROR');
            return false;
        }
    }
    
    /**
     * Cari slot optimal untuk event baru
     */
    public static function findOptimalSlots($existingEvents) {
        $today = new DateTime();
        $startRegistration = clone $today;
        $startRegistration->modify('+' . DEFAULT_PREPARATION_DAYS . ' days');
        
        $slotOptions = [];
        
        // Coba cari slot optimal untuk 90 hari ke depan
        for ($i = 0; $i < 90; $i += 7) {
            $testStart = clone $startRegistration;
            $testStart->modify("+$i days");
            
            // Cari durasi pendaftaran optimal
            for ($duration = MIN_REGISTRATION_DAYS; $duration <= MAX_REGISTRATION_DAYS; $duration++) {
                $testEnd = clone $testStart;
                $testEnd->modify('+' . ($duration - 1) . ' days');
                
                // Harus berakhir di Jumat
                if ($testEnd->format('w') != 5) continue;
                
                // Pelaksanaan harus di Sabtu
                $testExecution = clone $testEnd;
                $testExecution->modify('+1 day');
                
                // Cek konflik
                $conflicts = self::checkConflicts([
                    'pelaksanaan' => $testExecution->format(DATE_FORMAT)
                ], $existingEvents);
                
                if (empty($conflicts)) {
                    // Cek distribusi per bulan
                    $executionMonth = $testExecution->format('Y-m');
                    $monthEvents = array_filter($existingEvents, function($event) use ($executionMonth) {
                        return substr($event['pelaksanaan'], 0, 7) == $executionMonth;
                    });
                    
                    if (count($monthEvents) < MAX_EVENTS_PER_MONTH) {
                        $slotOptions[] = [
                            'mulai_pendaftaran' => $testStart->format(DATE_FORMAT),
                            'durasi_pendaftaran' => $duration,
                            'selesai_pendaftaran' => $testEnd->format(DATE_FORMAT),
                            'pelaksanaan' => $testExecution->format(DATE_FORMAT),
                            'bulan' => $testExecution->format('F Y'),
                            'event_bulan_ini' => count($monthEvents),
                            'skor' => (MAX_EVENTS_PER_MONTH - count($monthEvents)) * 10 + (45 - $i)
                        ];
                        
                        if (count($slotOptions) >= 3) break 2;
                    }
                }
            }
        }
        
        // Urutkan berdasarkan skor tertinggi
        usort($slotOptions, function($a, $b) {
            return $b['skor'] - $a['skor'];
        });
        
        return array_slice($slotOptions, 0, 3);
    }
    
    /**
     * Hitung timeline lengkap event
     */
    public static function calculateTimeline($inputData, $selectedSlot = null) {
        try {
            if ($selectedSlot) {
                $startRegistration = new DateTime($selectedSlot['mulai_pendaftaran']);
                $registrationDuration = $selectedSlot['durasi_pendaftaran'];
            } else {
                $startRegistration = new DateTime($inputData['tanggalMulai']);
                $registrationDuration = (int)$inputData['durasiPendaftaran'];
            }
            
            $hasSubstitute = ($inputData['susulan'] ?? 'tidak') === 'ya';
            $substituteDay = $inputData['hariSusulan'] ?? 'minggu';
            $hasFinal = ($inputData['babakFinal'] ?? 'tidak') === 'ya';
            
            // Tanggal akhir pendaftaran
            $endRegistration = clone $startRegistration;
            $endRegistration->modify('+' . ($registrationDuration - 1) . ' hari');
            
            // Tanggal pelaksanaan - HARUS Sabtu
            $execution = clone $endRegistration;
            $execution->modify('+1 hari');
            
            // Pastikan jatuh di Sabtu
            while ($execution->format('w') != 6) {
                $execution->modify('+1 hari');
            }
            
            // Tanggal susulan
            $substituteDate = null;
            if ($hasSubstitute) {
                $substituteDate = clone $execution;
                if ($substituteDay === 'minggu') {
                    $substituteDate->modify('+1 hari');
                } else if ($substituteDay === 'senin') {
                    $substituteDate->modify('+2 hari');
                }
            }
            
            // Tanggal pengumuman
            $announcement = $substituteDate ? clone $substituteDate : clone $execution;
            $announcement->modify('+' . TIMELINE_CONFIG['announcement_delay_days'] . ' hari');
            
            $timeline = [
                'mulai_pendaftaran' => $startRegistration->format(DATE_FORMAT),
                'selesai_pendaftaran' => $endRegistration->format(DATE_FORMAT),
                'pelaksanaan' => $execution->format(DATE_FORMAT),
                'tanggal_susulan' => $substituteDate ? $substituteDate->format(DATE_FORMAT) : null,
                'pengumuman' => $announcement->format(DATE_FORMAT),
                'durasi_pendaftaran' => $registrationDuration,
                'susulan' => $hasSubstitute ? 'ya' : 'tidak',
                'hari_susulan' => $substituteDay,
                'babak_final' => $hasFinal ? 'ya' : 'tidak'
            ];
            
            // Hitung fase tambahan
            if ($hasFinal) {
                $timeline = array_merge($timeline, self::calculateFinalPhases($announcement));
            } else {
                $timeline = array_merge($timeline, self::calculateRegularPhases($announcement));
            }
            
            return $timeline;
        } catch (Exception $e) {
            Config::log('Error calculating timeline: ' . $e->getMessage(), 'ERROR');
            return null;
        }
    }
    
    /**
     * Hitung fase untuk event dengan babak final
     */
    private static function calculateFinalPhases($announcement) {
        $finalRegistrationStart = clone $announcement;
        $finalRegistrationStart->modify('+1 hari');
        
        $finalRegistrationEnd = clone $finalRegistrationStart;
        $finalRegistrationEnd->modify('+' . TIMELINE_CONFIG['registration_final_days'] . ' hari');
        
        $finalDate = clone $finalRegistrationEnd;
        $finalDate->modify('+1 hari');
        while ($finalDate->format('w') != 6) {
            $finalDate->modify('+1 hari');
        }
        
        $finalAnnouncement = clone $finalDate;
        $finalAnnouncement->modify('+' . TIMELINE_CONFIG['announcement_delay_days'] . ' hari');
        
        $rewardAccessStart = clone $finalAnnouncement;
        $rewardAccessStart->modify('+1 hari');
        
        $rewardAccessEnd = clone $rewardAccessStart;
        $rewardAccessEnd->modify('+' . TIMELINE_CONFIG['final_access_days'] . ' hari');
        
        $cleanup = clone $rewardAccessEnd;
        $cleanup->modify('+' . TIMELINE_CONFIG['cleanup_delay_days'] . ' hari');
        
        return [
            'daftar_ulang_mulai' => $finalRegistrationStart->format(DATE_FORMAT),
            'daftar_ulang_selesai' => $finalRegistrationEnd->format(DATE_FORMAT),
            'tanggal_final' => $finalDate->format(DATE_FORMAT),
            'pengumuman_final' => $finalAnnouncement->format(DATE_FORMAT),
            'akses_penghargaan_mulai' => $rewardAccessStart->format(DATE_FORMAT),
            'akses_penghargaan_selesai' => $rewardAccessEnd->format(DATE_FORMAT),
            'sapu_bersih' => $cleanup->format(DATE_FORMAT)
        ];
    }
    
    /**
     * Hitung fase untuk event reguler
     */
    private static function calculateRegularPhases($announcement) {
        $claim1Start = clone $announcement;
        $claim1Start->modify('+1 hari');
        
        $claim1End = clone $claim1Start;
        $claim1End->modify('+' . TIMELINE_CONFIG['claim_period_1_days'] . ' hari');
        
        $feedbackForm = clone $claim1End;
        $feedbackForm->modify('+' . TIMELINE_CONFIG['feedback_form_days'] . ' hari');
        
        $break1Start = clone $feedbackForm;
        $break1Start->modify('+1 hari');
        
        $break1End = clone $break1Start;
        $break1End->modify('+' . TIMELINE_CONFIG['break_period_days'] . ' hari');
        
        $claim2Start = clone $break1End;
        $claim2Start->modify('+1 hari');
        
        $claim2End = clone $claim2Start;
        $claim2End->modify('+' . TIMELINE_CONFIG['claim_period_2_days'] . ' hari');
        
        $break2Start = clone $claim2End;
        $break2Start->modify('+1 hari');
        
        $break2End = clone $break2Start;
        $break2End->modify('+' . TIMELINE_CONFIG['break_period_days'] . ' hari');
        
        $cleanup = clone $break2End;
        $cleanup->modify('+1 hari');
        
        return [
            'klaim_gel1_mulai' => $claim1Start->format(DATE_FORMAT),
            'klaim_gel1_selesai' => $claim1End->format(DATE_FORMAT),
            'form_umpan_balik' => $feedbackForm->format(DATE_FORMAT),
            'jeda1_mulai' => $break1Start->format(DATE_FORMAT),
            'jeda1_selesai' => $break1End->format(DATE_FORMAT),
            'klaim_gel2_mulai' => $claim2Start->format(DATE_FORMAT),
            'klaim_gel2_selesai' => $claim2End->format(DATE_FORMAT),
            'jeda2_mulai' => $break2Start->format(DATE_FORMAT),
            'jeda2_selesai' => $break2End->format(DATE_FORMAT),
            'sapu_bersih' => $cleanup->format(DATE_FORMAT)
        ];
    }
    
    /**
     * Cek konflik jadwal
     */
    public static function checkConflicts($newEvent, $existingEvents, $excludeId = null) {
        $conflicts = [];
        $newExecutionTimestamp = strtotime($newEvent['pelaksanaan']);
        
        foreach ($existingEvents as $event) {
            if ($excludeId && $event['id'] == $excludeId) continue;
            
            $existingExecutionTimestamp = strtotime($event['pelaksanaan']);
            $daysDifference = abs(($newExecutionTimestamp - $existingExecutionTimestamp) / (60 * 60 * 24));
            
            // Cek jarak minimal antar event
            if ($daysDifference < MIN_DAYS_BETWEEN_EVENTS) {
                $conflicts[] = [
                    'type' => 'distance',
                    'event' => $event,
                    'days' => $daysDifference,
                    'message' => sprintf(
                        "Jarak hanya %.0f hari dengan %s (minimal %d hari)",
                        $daysDifference,
                        $event['nama_lomba'],
                        MIN_DAYS_BETWEEN_EVENTS
                    )
                ];
            }
            
            // Cek jika di minggu yang sama
            $newWeek = date('Y-W', $newExecutionTimestamp);
            $existingWeek = date('Y-W', $existingExecutionTimestamp);
            if ($newWeek == $existingWeek) {
                $conflicts[] = [
                    'type' => 'same_week',
                    'event' => $event,
                    'message' => sprintf(
                        "Pelaksanaan di minggu yang sama dengan %s",
                        $event['nama_lomba']
                    )
                ];
            }
        }
        
        return $conflicts;
    }
    
    /**
     * Hitung beban kerja
     */
    public static function calculateWorkload($events, $targetMonth = null) {
        $workload = [];
        $targetMonth = $targetMonth ?: date('Y-m');
        
        foreach ($events as $event) {
            $phases = [
                'pra_pendaftaran' => [
                    date('Y-m-d', strtotime($event['mulai_pendaftaran'] . ' -7 days')),
                    $event['mulai_pendaftaran'],
                    WORKLOAD_CONFIG['high_intensity']
                ],
                'pendaftaran' => [
                    $event['mulai_pendaftaran'],
                    $event['selesai_pendaftaran'],
                    WORKLOAD_CONFIG['low_intensity']
                ],
                'pelaksanaan' => [
                    $event['pelaksanaan'],
                    $event['pelaksanaan'],
                    WORKLOAD_CONFIG['high_intensity']
                ],
                'packing' => [
                    date('Y-m-d', strtotime($event['pengumuman'] . ' +1 day')),
                    date('Y-m-d', strtotime($event['pengumuman'] . ' +7 days')),
                    WORKLOAD_CONFIG['high_intensity']
                ]
            ];
            
            foreach ($phases as $phaseName => $data) {
                list($start, $end, $intensity) = $data;
                if (substr($start, 0, 7) <= $targetMonth && substr($end, 0, 7) >= $targetMonth) {
                    $week = date('Y-W', strtotime($start));
                    if (!isset($workload[$week])) $workload[$week] = 0;
                    $workload[$week] += $intensity;
                }
            }
        }
        
        return $workload;
    }
}

/**
 * Kelas untuk utilitas tanggal dan formatting
 */
class DateHelper {
    
    /**
     * Format tanggal ke bahasa Indonesia
     */
    public static function formatIndonesian($date) {
        return (new DateTime($date))->format('l, d F Y');
    }
    
    /**
     * Format rentang tanggal
     */
    public static function formatDateRange($startDate, $endDate) {
        $start = new DateTime($startDate);
        $end = new DateTime($endDate);
        
        if ($start->format('Y-m') === $end->format('Y-m')) {
            return $start->format('l, d') . ' - ' . $end->format('l, d F Y');
        } else {
            return $start->format('l, d F Y') . ' - ' . $end->format('l, d F Y');
        }
    }
    
    /**
     * Cek apakah tanggal valid
     */
    public static function isValidDate($date, $format = DATE_FORMAT) {
        $d = DateTime::createFromFormat($format, $date);
        return $d && $d->format($format) === $date;
    }
}

/**
 * Kelas untuk autentikasi dan session
 */
class Auth {
    
    /**
     * Cek apakah user sudah login
     */
    public static function isLoggedIn() {
        return isset($_SESSION['logged_in']) && 
               $_SESSION['logged_in'] === true &&
               self::isSessionValid();
    }
    
    /**
     * Validasi session timeout
     */
    private static function isSessionValid() {
        if (!isset($_SESSION['last_activity'])) {
            return false;
        }
        
        $inactive = time() - $_SESSION['last_activity'];
        if ($inactive > SESSION_TIMEOUT) {
            self::logout();
            return false;
        }
        
        $_SESSION['last_activity'] = time();
        return true;
    }
    
    /**
     * Proses login
     */
    public static function login($pin) {
        if ($pin === LOGIN_PIN) {
            $_SESSION['logged_in'] = true;
            $_SESSION['last_activity'] = time();
            $_SESSION['login_time'] = time();
            Config::log('User logged in');
            return true;
        }
        
        Config::log('Failed login attempt', 'WARNING');
        return false;
    }
    
    /**
     * Logout
     */
    public static function logout() {
        Config::log('User logged out');
        session_destroy();
    }
    
    /**
     * Require login - redirect jika belum login
     */
    public static function requireLogin() {
        if (!self::isLoggedIn()) {
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
        }
    }
}

/**
 * Kelas untuk validasi input
 */
class Validator {
    
    /**
     * Validasi data event
     */
    public static function validateEventData($data) {
        $errors = [];
        
        // Required fields
        $required = ['namaLomba', 'singkatanLomba', 'penyelenggara'];
        foreach ($required as $field) {
            if (empty($data[$field])) {
                $errors[] = "Field '$field' harus diisi";
            }
        }
        
        // Validasi tanggal jika ada
        if (!empty($data['tanggalMulai']) && !DateHelper::isValidDate($data['tanggalMulai'])) {
            $errors[] = "Format tanggal mulai tidak valid";
        }
        
        // Validasi durasi pendaftaran
        if (!empty($data['durasiPendaftaran'])) {
            $duration = (int)$data['durasiPendaftaran'];
            if ($duration < MIN_REGISTRATION_DAYS || $duration > MAX_REGISTRATION_DAYS) {
                $errors[] = sprintf(
                    "Durasi pendaftaran harus antara %d-%d hari",
                    MIN_REGISTRATION_DAYS,
                    MAX_REGISTRATION_DAYS
                );
            }
        }
        
        return $errors;
    }
    
    /**
     * Sanitize input data
     */
    public static function sanitizeInput($data) {
        if (is_array($data)) {
            return array_map([self::class, 'sanitizeInput'], $data);
        }
        return htmlspecialchars(trim($data), ENT_QUOTES, 'UTF-8');
    }
}