<?php
/**
 * Konfigurasi Sistem Pengelola Event Olimpiade
 * 
 * File ini berisi semua konfigurasi yang dibutuhkan sistem
 */

// Konfigurasi Database/Storage
define('DATA_DIR', __DIR__ . '/data');
define('DATA_FILE', DATA_DIR . '/events.json');
define('LOG_FILE', DATA_DIR . '/system.log');

// Konfigurasi Keamanan
define('LOGIN_PIN', '051205'); // Ganti dengan PIN yang lebih aman
define('SESSION_TIMEOUT', 3600); // 1 jam dalam detik
define('CSRF_TOKEN_NAME', 'csrf_token');

// Konfigurasi Event
define('MIN_REGISTRATION_DAYS', 25);
define('MAX_REGISTRATION_DAYS', 45);
define('MAX_EVENTS_PER_MONTH', 3);
define('MIN_DAYS_BETWEEN_EVENTS', 12);
define('DEFAULT_PREPARATION_DAYS', 5);

// Konfigurasi Tanggal
define('DEFAULT_TIMEZONE', 'Asia/Jakarta');
define('DATE_FORMAT', 'Y-m-d');
define('DATETIME_FORMAT', 'Y-m-d H:i:s');

// Konfigurasi Aplikasi
define('APP_NAME', 'Pengelola Event Olimpiade');
define('APP_VERSION', '1.0.0');
define('DEBUG_MODE', false); // Set false untuk production

// Konfigurasi Locale
define('LOCALE_ID', ['id_ID.UTF-8', 'Indonesian_Indonesia.1252', 'id_ID', 'IND']);

// Pesan Error dan Sukses
define('MESSAGES', [
    'login_success' => 'Login berhasil!',
    'login_failed' => 'PIN salah!',
    'event_saved' => 'Event berhasil disimpan!',
    'event_deleted' => 'Event berhasil dihapus!',
    'event_updated' => 'Event berhasil diperbarui!',
    'validation_required' => 'Mohon lengkapi semua field yang diperlukan!',
    'no_optimal_slot' => 'Tidak ditemukan slot optimal untuk jadwal ini',
    'conflict_detected' => 'Terdeteksi konflik jadwal dengan event lain',
    'session_expired' => 'Sesi telah berakhir, silakan login kembali',
    'permission_denied' => 'Akses ditolak',
    'data_not_found' => 'Data tidak ditemukan',
    'invalid_date' => 'Format tanggal tidak valid',
    'file_error' => 'Terjadi kesalahan pada file sistem'
]);

// Konfigurasi Beban Kerja
define('WORKLOAD_CONFIG', [
    'high_intensity' => 3,
    'medium_intensity' => 2,
    'low_intensity' => 1,
    'max_weekly_load' => 6,
    'phases' => [
        'pre_registration' => ['days_before' => 7, 'intensity' => 'high'],
        'registration' => ['intensity' => 'low'],
        'execution' => ['intensity' => 'high'],
        'packing' => ['days_after' => 7, 'intensity' => 'high']
    ]
]);

// Konfigurasi Timeline Default
define('TIMELINE_CONFIG', [
    'announcement_delay_days' => 3,
    'registration_final_days' => 10,
    'feedback_form_days' => 1,
    'break_period_days' => 5,
    'claim_period_1_days' => 10,
    'claim_period_2_days' => 14,
    'final_access_days' => 10,
    'cleanup_delay_days' => 5
]);

// Fungsi Helper untuk Konfigurasi
class Config {
    
    /**
     * Inisialisasi sistem
     */
    public static function init() {
        // Set timezone
        date_default_timezone_set(DEFAULT_TIMEZONE);
        
        // Set locale untuk format tanggal Indonesia
        setlocale(LC_TIME, ...LOCALE_ID);
        
        // Buat direktori data jika belum ada
        if (!is_dir(DATA_DIR)) {
            mkdir(DATA_DIR, 0755, true);
        }
        
        // Inisialisasi file data jika belum ada
        if (!file_exists(DATA_FILE)) {
            self::createDataFile();
        }
        
        // Start session dengan konfigurasi aman
        self::startSecureSession();
    }
    
    /**
     * Buat file data awal
     */
    private static function createDataFile() {
        $initialData = [
            'events' => [],
            'created_at' => date(DATETIME_FORMAT),
            'version' => APP_VERSION
        ];
        
        file_put_contents(DATA_FILE, json_encode($initialData, JSON_PRETTY_PRINT));
        self::log('Data file created: ' . DATA_FILE);
    }
    
    /**
     * Start session dengan konfigurasi keamanan
     */
    private static function startSecureSession() {
        if (session_status() === PHP_SESSION_NONE) {
            ini_set('session.cookie_httponly', 1);
            ini_set('session.cookie_secure', isset($_SERVER['HTTPS']));
            ini_set('session.use_only_cookies', 1);
            session_start();
        }
    }
    
    /**
     * Generate CSRF Token
     */
    public static function generateCSRFToken() {
        if (!isset($_SESSION[CSRF_TOKEN_NAME])) {
            $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
        }
        return $_SESSION[CSRF_TOKEN_NAME];
    }
    
    /**
     * Validasi CSRF Token
     */
    public static function validateCSRFToken($token) {
        return isset($_SESSION[CSRF_TOKEN_NAME]) && 
               hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
    }
    
    /**
     * Logging system
     */
    public static function log($message, $level = 'INFO') {
        if (DEBUG_MODE) {
            $timestamp = date(DATETIME_FORMAT);
            $logMessage = "[$timestamp] [$level] $message" . PHP_EOL;
            file_put_contents(LOG_FILE, $logMessage, FILE_APPEND | LOCK_EX);
        }
    }
    
    /**
     * Get message by key
     */
    public static function getMessage($key, $default = '') {
        return MESSAGES[$key] ?? $default;
    }
    
    /**
     * Validasi environment
     */
    public static function validateEnvironment() {
        $errors = [];
        
        // Cek versi PHP
        if (version_compare(PHP_VERSION, '7.4.0', '<')) {
            $errors[] = 'PHP version 7.4.0 atau lebih tinggi diperlukan';
        }
        
        // Cek ekstensi yang diperlukan
        $required_extensions = ['json', 'session', 'date'];
        foreach ($required_extensions as $ext) {
            if (!extension_loaded($ext)) {
                $errors[] = "Ekstensi PHP '$ext' tidak tersedia";
            }
        }
        
        // Cek permission direktori
        if (!is_writable(dirname(DATA_DIR))) {
            $errors[] = 'Direktori tidak memiliki permission write: ' . dirname(DATA_DIR);
        }
        
        return $errors;
    }
    
    /**
     * Get application info
     */
    public static function getAppInfo() {
        return [
            'name' => APP_NAME,
            'version' => APP_VERSION,
            'php_version' => PHP_VERSION,
            'timezone' => DEFAULT_TIMEZONE,
            'debug_mode' => DEBUG_MODE
        ];
    }
}

// Inisialisasi otomatis
Config::init();