<?php
// Aplikasi Pengelola Kardus Penyimpanan Pribadi
// Versi: 1.0
// Fitur: Tanpa Database, Menggunakan JSON, QR Code, URL Unik

// Konfigurasi
define('DATA_FILE', 'boxes_data.json');
define('BASE_URL', 'http://localhost/'); // Ganti dengan URL domain Anda

// Fungsi untuk memuat data dari JSON
function loadData() {
    if (!file_exists(DATA_FILE)) {
        return ['boxes' => [], 'counters' => ['A' => 0, 'B' => 0, 'C' => 0, 'D' => 0, 'E' => 0]];
    }
    return json_decode(file_get_contents(DATA_FILE), true);
}

// Fungsi untuk menyimpan data ke JSON
function saveData($data) {
    file_put_contents(DATA_FILE, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

// Fungsi untuk generate kode kardus otomatis
function generateBoxCode($size, $counters) {
    $counters[$size]++;
    return $size . $counters[$size];
}

// Fungsi untuk generate QR Code (menggunakan Google Charts API)
function generateQRCode($boxCode) {
    $url = BASE_URL . '?box=' . strtolower($boxCode);
    $qrUrl = "https://chart.googleapis.com/chart?chs=200x200&cht=qr&chl=" . urlencode($url);
    return $qrUrl;
}

// Fungsi untuk mendeteksi kategori otomatis (AI sederhana)
function detectCategory($itemName) {
    $electronics = ['hp', 'laptop', 'kabel', 'charger', 'mouse', 'keyboard', 'headset', 'speaker', 'tv', 'radio', 'komputer', 'tablet', 'powerbank', 'elektronik'];
    $stationery = ['pulpen', 'pensil', 'kertas', 'buku', 'penggaris', 'gunting', 'lem', 'stapler', 'spidol', 'penghapus', 'map', 'amplop'];
    $clothing = ['baju', 'celana', 'rok', 'sepatu', 'sandal', 'topi', 'jaket', 'kaos', 'kemeja', 'dress', 'pakaian'];
    $kitchen = ['piring', 'gelas', 'sendok', 'garpu', 'pisau', 'panci', 'wajan', 'kompor', 'blender', 'rice cooker', 'kulkas'];
    
    $itemLower = strtolower($itemName);
    
    foreach ($electronics as $keyword) {
        if (strpos($itemLower, $keyword) !== false) return 'Elektronik';
    }
    foreach ($stationery as $keyword) {
        if (strpos($itemLower, $keyword) !== false) return 'Alat Tulis';
    }
    foreach ($clothing as $keyword) {
        if (strpos($itemLower, $keyword) !== false) return 'Pakaian';
    }
    foreach ($kitchen as $keyword) {
        if (strpos($itemLower, $keyword) !== false) return 'Dapur';
    }
    
    return 'Campuran';
}

// Fungsi untuk mencari barang duplikat
function findDuplicateItems($itemName, $currentBoxCode, $data) {
    $duplicates = [];
    foreach ($data['boxes'] as $box) {
        if ($box['code'] !== $currentBoxCode) {
            foreach ($box['items'] as $item) {
                if (stripos($item['name'], $itemName) !== false || stripos($itemName, $item['name']) !== false) {
                    $duplicates[] = ['box' => $box['code'], 'alias' => $box['alias'], 'item' => $item['name']];
                }
            }
        }
    }
    return $duplicates;
}

// Memuat data
$data = loadData();

// Routing sederhana
$action = $_GET['action'] ?? '';
$boxCode = $_GET['box'] ?? '';

// Proses form submission
if ($_POST) {
    if (isset($_POST['add_box'])) {
        // Tambah kardus baru
        $size = $_POST['size'];
        $alias = $_POST['alias'];
        $weight = $_POST['weight'];
        $category = $_POST['category'];
        
        $code = generateBoxCode($size, $data['counters']);
        
        $newBox = [
            'code' => $code,
            'alias' => $alias,
            'size' => $size,
            'weight' => $weight,
            'category' => $category,
            'items' => [],
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        $data['boxes'][] = $newBox;
        saveData($data);
        header("Location: ?box=" . strtolower($code));
        exit;
    }
    
    if (isset($_POST['add_items'])) {
        // Tambah barang ke kardus
        $targetBox = $_POST['box_code'];
        $itemsText = $_POST['items_text'];
        
        // Cari index kardus
        $boxIndex = -1;
        foreach ($data['boxes'] as $index => $box) {
            if (strtolower($box['code']) === strtolower($targetBox)) {
                $boxIndex = $index;
                break;
            }
        }
        
        if ($boxIndex !== -1) {
            // Parse items (mendukung input massal)
            $lines = explode("\n", $itemsText);
            foreach ($lines as $line) {
                $line = trim($line);
                if (!empty($line)) {
                    // Parse format: "nama|jumlah|harga|kondisi" atau sekadar "nama"
                    $parts = explode('|', $line);
                    $itemName = trim($parts[0]);
                    $quantity = isset($parts[1]) ? trim($parts[1]) : '';
                    $price = isset($parts[2]) ? trim($parts[2]) : '';
                    $condition = isset($parts[3]) ? trim($parts[3]) : '';
                    
                    if (!empty($itemName)) {
                        $newItem = [
                            'name' => $itemName,
                            'quantity' => $quantity,
                            'price' => $price,
                            'condition' => $condition,
                            'suggested_category' => detectCategory($itemName),
                            'added_at' => date('Y-m-d H:i:s')
                        ];
                        
                        $data['boxes'][$boxIndex]['items'][] = $newItem;
                    }
                }
            }
            
            saveData($data);
        }
        
        header("Location: ?box=" . strtolower($targetBox));
        exit;
    }
    
    if (isset($_POST['delete_item'])) {
        // Hapus barang
        $targetBox = $_POST['box_code'];
        $itemIndex = $_POST['item_index'];
        
        foreach ($data['boxes'] as $boxIdx => $box) {
            if (strtolower($box['code']) === strtolower($targetBox)) {
                unset($data['boxes'][$boxIdx]['items'][$itemIndex]);
                $data['boxes'][$boxIdx]['items'] = array_values($data['boxes'][$boxIdx]['items']);
                break;
            }
        }
        
        saveData($data);
        header("Location: ?box=" . strtolower($targetBox));
        exit;
    }
}

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pengelola Kardus Penyimpanan</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            color: #333;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .header {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 15px;
            padding: 25px;
            margin-bottom: 20px;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
            text-align: center;
        }
        
        .header h1 {
            color: #4a5568;
            font-size: 2.5em;
            margin-bottom: 10px;
        }
        
        .header p {
            color: #718096;
            font-size: 1.1em;
        }
        
        .card {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 15px;
            padding: 25px;
            margin-bottom: 20px;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
            backdrop-filter: blur(10px);
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #4a5568;
        }
        
        .form-group input,
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 12px;
            border: 2px solid #e2e8f0;
            border-radius: 8px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        
        .form-group input:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: #667eea;
        }
        
        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
            text-decoration: none;
            display: inline-block;
            margin: 5px;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }
        
        .btn-secondary {
            background: #e2e8f0;
            color: #4a5568;
        }
        
        .btn-secondary:hover {
            background: #cbd5e0;
        }
        
        .btn-danger {
            background: #f56565;
            color: white;
        }
        
        .btn-danger:hover {
            background: #e53e3e;
        }
        
        .box-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }
        
        .box-card {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 15px;
            padding: 20px;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
            transition: transform 0.3s;
        }
        
        .box-card:hover {
            transform: translateY(-5px);
        }
        
        .box-code {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 8px 16px;
            border-radius: 20px;
            font-weight: bold;
            font-size: 1.2em;
            display: inline-block;
            margin-bottom: 10px;
        }
        
        .size-indicator {
            display: inline-block;
            width: 10px;
            height: 10px;
            border-radius: 50%;
            margin-right: 5px;
        }
        
        .size-A { background: #f56565; }
        .size-B { background: #ed8936; }
        .size-C { background: #ecc94b; }
        .size-D { background: #48bb78; }
        .size-E { background: #4299e1; }
        
        .item-list {
            margin-top: 15px;
        }
        
        .item {
            background: #f7fafc;
            padding: 10px;
            margin-bottom: 8px;
            border-radius: 8px;
            border-left: 4px solid #667eea;
        }
        
        .item-name {
            font-weight: 600;
            color: #2d3748;
        }
        
        .item-details {
            font-size: 0.9em;
            color: #718096;
            margin-top: 5px;
        }
        
        .qr-code {
            text-align: center;
            margin: 20px 0;
        }
        
        .qr-code img {
            border-radius: 8px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
        }
        
        .alert {
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .alert-warning {
            background: #fed7aa;
            color: #c2410c;
            border: 1px solid #fdba74;
        }
        
        .duplicate-warning {
            background: #fef3c7;
            border: 1px solid #f59e0b;
            color: #92400e;
            padding: 10px;
            border-radius: 8px;
            margin-top: 10px;
        }
        
        .stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }
        
        .stat-card {
            background: rgba(255, 255, 255, 0.2);
            padding: 15px;
            border-radius: 10px;
            text-align: center;
            color: white;
        }
        
        .stat-number {
            font-size: 2em;
            font-weight: bold;
        }
        
        .stat-label {
            font-size: 0.9em;
            opacity: 0.9;
        }
        
        .search-box {
            margin-bottom: 20px;
        }
        
        .search-box input {
            width: 100%;
            padding: 15px;
            border: 2px solid rgba(255, 255, 255, 0.3);
            border-radius: 10px;
            font-size: 16px;
            background: rgba(255, 255, 255, 0.1);
            color: white;
            backdrop-filter: blur(10px);
        }
        
        .search-box input::placeholder {
            color: rgba(255, 255, 255, 0.7);
        }
        
        .back-btn {
            margin-bottom: 20px;
        }
        
        @media (max-width: 768px) {
            .container {
                padding: 10px;
            }
            
            .header h1 {
                font-size: 2em;
            }
            
            .box-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <?php if (empty($boxCode)): ?>
            <!-- Halaman Utama -->
            <div class="header">
                <h1>📦 Pengelola Kardus Penyimpanan</h1>
                <p>Kelola dan lacak semua barang dalam kardus penyimpanan Anda</p>
            </div>
            
            <!-- Statistik -->
            <div class="stats">
                <div class="stat-card">
                    <div class="stat-number"><?= count($data['boxes']) ?></div>
                    <div class="stat-label">Total Kardus</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number">
                        <?php 
                        $totalItems = 0;
                        foreach ($data['boxes'] as $box) {
                            $totalItems += count($box['items']);
                        }
                        echo $totalItems;
                        ?>
                    </div>
                    <div class="stat-label">Total Barang</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number">
                        <?php 
                        $categories = [];
                        foreach ($data['boxes'] as $box) {
                            $categories[] = $box['category'];
                        }
                        echo count(array_unique($categories));
                        ?>
                    </div>
                    <div class="stat-label">Kategori Unik</div>
                </div>
            </div>
            
            <!-- Form Tambah Kardus -->
            <div class="card">
                <h2>➕ Tambah Kardus Baru</h2>
                <form method="post">
                    <div class="form-group">
                        <label>Nama Alias Kardus</label>
                        <input type="text" name="alias" required placeholder="Contoh: Kardus Elektronik Lama">
                    </div>
                    
                    <div class="form-group">
                        <label>Ukuran Kardus</label>
                        <select name="size" required>
                            <option value="">Pilih Ukuran</option>
                            <option value="A">A - Sangat Besar</option>
                            <option value="B">B - Besar</option>
                            <option value="C">C - Sedang</option>
                            <option value="D">D - Kecil</option>
                            <option value="E">E - Sangat Kecil</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>Berat Kardus (kg)</label>
                        <input type="number" name="weight" step="0.1" placeholder="Contoh: 2.5">
                    </div>
                    
                    <div class="form-group">
                        <label>Kategori Utama</label>
                        <select name="category" required>
                            <option value="">Pilih Kategori</option>
                            <option value="Elektronik">Elektronik</option>
                            <option value="Alat Tulis">Alat Tulis</option>
                            <option value="Pakaian">Pakaian</option>
                            <option value="Dapur">Dapur</option>
                            <option value="Buku">Buku</option>
                            <option value="Mainan">Mainan</option>
                            <option value="Campuran">Campuran</option>
                        </select>
                    </div>
                    
                    <button type="submit" name="add_box" class="btn btn-primary">Buat Kardus</button>
                </form>
            </div>
            
            <!-- Pencarian -->
            <div class="search-box">
                <input type="text" id="searchInput" placeholder="🔍 Cari kardus atau barang...">
            </div>
            
            <!-- Daftar Kardus -->
            <div class="box-grid" id="boxGrid">
                <?php foreach ($data['boxes'] as $box): ?>
                    <div class="box-card" data-search="<?= strtolower($box['alias'] . ' ' . $box['code'] . ' ' . $box['category'] . ' ' . implode(' ', array_column($box['items'], 'name'))) ?>">
                        <div class="box-code">
                            <span class="size-indicator size-<?= $box['size'] ?>"></span>
                            <?= $box['code'] ?>
                        </div>
                        <h3><?= htmlspecialchars($box['alias']) ?></h3>
                        <p><strong>Kategori:</strong> <?= $box['category'] ?></p>
                        <p><strong>Berat:</strong> <?= $box['weight'] ?> kg</p>
                        <p><strong>Jumlah Barang:</strong> <?= count($box['items']) ?> item</p>
                        <p><strong>Dibuat:</strong> <?= date('d/m/Y H:i', strtotime($box['created_at'])) ?></p>
                        
                        <div style="margin-top: 15px;">
                            <a href="?box=<?= strtolower($box['code']) ?>" class="btn btn-primary">Lihat Detail</a>
                            <button onclick="copyToClipboard('<?= BASE_URL ?>?box=<?= strtolower($box['code']) ?>')" class="btn btn-secondary">Copy Link</button>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
            
            <?php if (empty($data['boxes'])): ?>
                <div class="card" style="text-align: center; color: #718096;">
                    <h3>Belum ada kardus</h3>
                    <p>Silakan tambahkan kardus pertama Anda menggunakan form di atas</p>
                </div>
            <?php endif; ?>
            
        <?php else: ?>
            <!-- Halaman Detail Kardus -->
            <?php
            $currentBox = null;
            foreach ($data['boxes'] as $box) {
                if (strtolower($box['code']) === strtolower($boxCode)) {
                    $currentBox = $box;
                    break;
                }
            }
            
            if (!$currentBox): ?>
                <div class="card">
                    <h2>❌ Kardus Tidak Ditemukan</h2>
                    <p>Kardus dengan kode "<?= htmlspecialchars($boxCode) ?>" tidak ditemukan.</p>
                    <a href="?" class="btn btn-primary">Kembali ke Beranda</a>
                </div>
            <?php else: ?>
                <div class="back-btn">
                    <a href="?" class="btn btn-secondary">← Kembali ke Beranda</a>
                </div>
                
                <div class="card">
                    <div style="display: flex; justify-content: space-between; align-items: flex-start; flex-wrap: wrap;">
                        <div style="flex: 1; min-width: 300px;">
                            <div class="box-code">
                                <span class="size-indicator size-<?= $currentBox['size'] ?>"></span>
                                <?= $currentBox['code'] ?>
                            </div>
                            <h2><?= htmlspecialchars($currentBox['alias']) ?></h2>
                            <p><strong>Kategori:</strong> <?= $currentBox['category'] ?></p>
                            <p><strong>Berat:</strong> <?= $currentBox['weight'] ?> kg</p>
                            <p><strong>Jumlah Barang:</strong> <?= count($currentBox['items']) ?> item</p>
                            <p><strong>Dibuat:</strong> <?= date('d/m/Y H:i', strtotime($currentBox['created_at'])) ?></p>
                        </div>
                        
                        <div class="qr-code">
                            <img src="<?= generateQRCode($currentBox['code']) ?>" alt="QR Code">
                            <br>
                            <a href="<?= generateQRCode($currentBox['code']) ?>" download="qr-<?= strtolower($currentBox['code']) ?>.png" class="btn btn-secondary">Download QR</a>
                        </div>
                    </div>
                    
                    <div style="margin-top: 20px;">
                        <strong>URL Kardus:</strong> 
                        <input type="text" value="<?= BASE_URL ?>?box=<?= strtolower($currentBox['code']) ?>" readonly style="width: 70%; display: inline-block;">
                        <button onclick="copyToClipboard('<?= BASE_URL ?>?box=<?= strtolower($currentBox['code']) ?>')" class="btn btn-secondary">Copy</button>
                    </div>
                </div>
                
                <!-- Form Tambah Barang -->
                <div class="card">
                    <h3>➕ Tambah Barang</h3>
                    <form method="post">
                        <input type="hidden" name="box_code" value="<?= $currentBox['code'] ?>">
                        
                        <div class="form-group">
                            <label>Daftar Barang</label>
                            <textarea name="items_text" rows="8" placeholder="Masukkan barang satu per baris. Format:
Nama Barang
atau
Nama Barang|Jumlah|Harga|Kondisi

Contoh:
Laptop ASUS|1 unit|Rp 5.000.000|Baik
Mouse Gaming
Keyboard Mechanical|1 unit|Rp 800.000|Seperti Baru"></textarea>
                        </div>
                        
                        <button type="submit" name="add_items" class="btn btn-primary">Tambah Barang</button>
                    </form>
                </div>
                
                <!-- Daftar Barang -->
                <div class="card">
                    <h3>📋 Daftar Barang (<?= count($currentBox['items']) ?> item)</h3>
                    
                    <?php if (empty($currentBox['items'])): ?>
                        <p style="text-align: center; color: #718096; padding: 20px;">
                            Belum ada barang dalam kardus ini
                        </p>
                    <?php else: ?>
                        <div class="item-list">
                            <?php foreach ($currentBox['items'] as $index => $item): ?>
                                <div class="item">
                                    <div class="item-name"><?= htmlspecialchars($item['name']) ?></div>
                                    <div class="item-details">
                                        <?php if (!empty($item['quantity'])): ?>
                                            <span>📦 <?= htmlspecialchars($item['quantity']) ?></span>
                                        <?php endif; ?>
                                        
                                        <?php if (!empty($item['price'])): ?>
                                            <span>💰 <?= htmlspecialchars($item['price']) ?></span>
                                        <?php endif; ?>
                                        
                                        <?php if (!empty($item['condition'])): ?>
                                            <span>🔧 <?= htmlspecialchars($item['condition']) ?></span>
                                        <?php endif; ?>
                                        
                                        <span>🏷️ <?= $item['suggested_category'] ?></span>
                                        <span>📅 <?= date('d/m/Y H:i', strtotime($item['added_at'])) ?></span>
                                    </div>
                                    
                                    <!-- Cek duplikat -->
                                    <?php $duplicates = findDuplicateItems($item['name'], $currentBox['code'], $data); ?>
                                    <?php if (!empty($duplicates)): ?>
                                        <div class="duplicate-warning">
                                            ⚠️ Barang serupa ditemukan di:
                                            <?php foreach ($duplicates as $dup): ?>
                                                <a href="?box=<?= strtolower($dup['box']) ?>" style="color: #92400e; text-decoration: underline;">
                                                    <?= $dup['box'] ?> (<?= $dup['alias'] ?>)
                                                </a>
                                            <?php endforeach; ?>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <div style="margin-top: 10px;">
                                        <form method="post" style="display: inline;">
                                            <input type="hidden" name="box_code" value="<?= $currentBox['code'] ?>">
                                            <input type="hidden" name="item_index" value="<?= $index ?>">
                                            <button type="submit" name="delete_item" class="btn btn-danger" onclick="return confirm('Yakin ingin menghapus barang ini?')">Hapus</button>
                                        </form>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
    
    <script>
        // Fungsi copy to clipboard
        function copyToClipboard(text) {
            navigator.clipboard.writeText(text).then(function() {
                alert('Link berhasil disalin!');
            });
        }
        
        // Fungsi pencarian
        document.getElementById('searchInput')?.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const boxes = document.querySelectorAll('.box-card');
            
            boxes.forEach(box => {
                const searchData = box.dataset.search;
                if (searchData.includes(searchTerm)) {
                    box.style.display = 'block';
                } else {
                    box.style.display = 'none';
                }
            });
        });
        
        // Auto-focus pada form
        document.addEventListener('DOMContentLoaded', function() {
            const firstInput = document.querySelector('input[type="text"]:not([readonly])');
            if (firstInput) {
                firstInput.focus();
            }
        });
        
        // Smooth animations
        document.querySelectorAll('.box-card').forEach(card => {
            card.style.opacity = '0';
            card.style.transform = 'translateY(20px)';
            
            setTimeout(() => {
                card.style.transition = 'all 0.5s ease';
                card.style.opacity = '1';
                card.style.transform = 'translateY(0)';
            }, Math.random() * 300);
        });
        
        // Form validation
        const forms = document.querySelectorAll('form');
        forms.forEach(form => {
            form.addEventListener('submit', function(e) {
                const requiredFields = form.querySelectorAll('[required]');
                let isValid = true;
                
                requiredFields.forEach(field => {
                    if (!field.value.trim()) {
                        field.style.borderColor = '#f56565';
                        isValid = false;
                    } else {
                        field.style.borderColor = '#e2e8f0';
                    }
                });
                
                if (!isValid) {
                    e.preventDefault();
                    alert('Mohon isi semua field yang wajib!');
                }
            });
        });
        
        // Keyboard shortcuts
        document.addEventListener('keydown', function(e) {
            // Ctrl+K untuk fokus search
            if (e.ctrlKey && e.key === 'k') {
                e.preventDefault();
                const searchInput = document.getElementById('searchInput');
                if (searchInput) {
                    searchInput.focus();
                }
            }
            
            // Escape untuk clear search
            if (e.key === 'Escape') {
                const searchInput = document.getElementById('searchInput');
                if (searchInput && searchInput.value) {
                    searchInput.value = '';
                    searchInput.dispatchEvent(new Event('input'));
                }
            }
        });
        
        // Export function
        function exportData() {
            const data = <?= json_encode($data) ?>;
            const blob = new Blob([JSON.stringify(data, null, 2)], {type: 'application/json'});
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'boxes_backup_' + new Date().getTime() + '.json';
            a.click();
            URL.revokeObjectURL(url);
        }
        
        // Print QR Code
        function printQR(boxCode) {
            const qrUrl = "https://chart.googleapis.com/chart?chs=300x300&cht=qr&chl=" + encodeURIComponent("<?= BASE_URL ?>?box=" + boxCode.toLowerCase());
            const printWindow = window.open('', '_blank');
            printWindow.document.write(`
                <html>
                <head>
                    <title>QR Code - ${boxCode}</title>
                    <style>
                        body { text-align: center; font-family: Arial, sans-serif; padding: 20px; }
                        .qr-container { margin: 20px; }
                        .box-info { margin: 10px 0; font-size: 18px; font-weight: bold; }
                        img { border: 2px solid #333; }
                    </style>
                </head>
                <body>
                    <div class="qr-container">
                        <div class="box-info">Kardus: ${boxCode}</div>
                        <img src="${qrUrl}" alt="QR Code">
                        <div class="box-info"><?= BASE_URL ?>?box=${boxCode.toLowerCase()}</div>
                    </div>
                    <script>
                        window.onload = function() {
                            window.print();
                            window.close();
                        };
                    </script>
                </body>
                </html>
            `);
        }
    </script>
</body>
</html>