<?php
// Aplikasi Fokus Harian - Brand & Mata Kuliah Management + Upah Per Brand
// File: index.php

session_start();
date_default_timezone_set('Asia/Jakarta');

// Konfigurasi data
$brands = [
    'cita_cerdas' => ['name' => '📗 Cita Cerdas', 'color' => '#28a745'],
    'arena_kompetisi' => ['name' => '📘 Arena Kompetisi', 'color' => '#007bff'],
    'dedikasi_prestasi' => ['name' => '📙 Dedikasi Prestasi', 'color' => '#fd7e14'],
    'lumina_prestasi' => ['name' => '📒 Lumina Prestasi', 'color' => '#6f42c1']
];

$subjects = [
    'pancasila' => ['name' => '🎓 Pancasila', 'code' => 'MKWN411001'],
    'hubungan_industrial' => ['name' => '📁 Hubungan Industrial', 'code' => 'EMBS20401'],
    'perilaku_organisasi' => ['name' => '📁 Prilaku Organisasi', 'code' => 'EKMA415803'],
    'akuntansi_biaya' => ['name' => '📁 Akuntansi Biaya', 'code' => 'EKMA431503'],
    'bahasa_inggris' => ['name' => '📁 Bahasa Inggris', 'code' => 'MKWI420102'],
    'ekonomi_makro' => ['name' => '📁 Pengantar Ekonomi Makro', 'code' => 'ESPA411003'],
    'organisasi' => ['name' => '📁 Organisasi', 'code' => 'EKMA415702']
];

$hourlyRate = 6000; // Upah per jam: Rp 6,000
$dataFile = 'focus_data.json';
$payrollFile = 'payroll_data.json';

// Fungsi untuk membaca data
function loadData($file) {
    if (file_exists($file)) {
        $content = file_get_contents($file);
        return json_decode($content, true) ?: [];
    }
    return [];
}

// Fungsi untuk menyimpan data
function saveData($file, $data) {
    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT));
}

// Fungsi untuk menghitung upah berdasarkan jam kerja
function calculateWage($workHours, $hourlyRate) {
    return floor($workHours * $hourlyRate);
}

// Fungsi untuk format rupiah
function formatRupiah($amount) {
    return 'Rp ' . number_format($amount, 0, ',', '.');
}

// Fungsi untuk menghitung prioritas brand berdasarkan histori
function calculateBrandPriority($data, $brands) {
    $today = date('Y-m-d');
    $brandScores = [];
    
    foreach ($brands as $brandKey => $brand) {
        $recentAttendance = 0;
        $totalDays = 0;
        $missedDays = 0;
        
        for ($i = 0; $i < 7; $i++) {
            $checkDate = date('Y-m-d', strtotime("-$i days"));
            $totalDays++;
            
            if (isset($data[$checkDate]['attendance'][$brandKey])) {
                $recentAttendance++;
            } else {
                $missedDays++;
            }
        }
        
        $attendanceRate = $totalDays > 0 ? $recentAttendance / $totalDays : 0;
        $priorityScore = (1 - $attendanceRate) * 100 + $missedDays * 10;
        
        $brandScores[$brandKey] = [
            'score' => $priorityScore,
            'attendance_rate' => $attendanceRate,
            'missed_days' => $missedDays
        ];
    }
    
    arsort($brandScores);
    return $brandScores;
}

// Fungsi untuk mendapatkan rekomendasi mata kuliah
function getSubjectRecommendation($subjects) {
    $dayOfWeek = date('N');
    $subjectKeys = array_keys($subjects);
    $recommendedKey = $subjectKeys[($dayOfWeek - 1) % count($subjectKeys)];
    return [$recommendedKey => $subjects[$recommendedKey]];
}

// Proses absensi dengan input jam kerja
if ($_POST['action'] ?? '' === 'attend') {
    $data = loadData($dataFile);
    $payrollData = loadData($payrollFile);
    
    $today = date('Y-m-d');
    $brandKey = $_POST['brand'] ?? '';
    $workHours = floatval($_POST['work_hours'] ?? 0);
    
    if ($workHours <= -10) {
        $_SESSION['error'] = "❌ Jam kerja harus lebih dari 0!";
    } elseif ($workHours > 12) {
        $_SESSION['error'] = "❌ Jam kerja tidak boleh lebih dari 12 jam per hari!";
    } else {
        // Simpan data absensi
        if (!isset($data[$today])) {
            $data[$today] = ['attendance' => []];
        }
        
        $wageEarned = calculateWage($workHours, $hourlyRate);
        
        $data[$today]['attendance'][$brandKey] = [
            'timestamp' => time(),
            'work_hours' => $workHours,
            'wage_earned' => $wageEarned
        ];
        
        saveData($dataFile, $data);
        
        // Update saldo per brand
        if (!isset($payrollData['balances'])) {
            $payrollData['balances'] = [];
        }
        if (!isset($payrollData['balances'][$brandKey])) {
            $payrollData['balances'][$brandKey] = 0;
        }
        
        $payrollData['balances'][$brandKey] += $wageEarned;
        
        // Simpan riwayat earning
        if (!isset($payrollData['earnings'])) {
            $payrollData['earnings'] = [];
        }
        $payrollData['earnings'][] = [
            'date' => $today,
            'brand' => $brandKey,
            'work_hours' => $workHours,
            'wage_earned' => $wageEarned,
            'timestamp' => time()
        ];
        
        saveData($payrollFile, $payrollData);
        
        $_SESSION['message'] = "✅ Absen berhasil untuk " . $brands[$brandKey]['name'] . 
                               " | Jam kerja: " . $workHours . " jam | " .
                               "Upah: " . formatRupiah($wageEarned);
    }
    
    header('Location: ' . $_SERVER['PHP_SELF'] . '?tab=attendance');
    exit;
}

// Proses penarikan upah per brand
if ($_POST['action'] ?? '' === 'withdraw') {
    $payrollData = loadData($payrollFile);
    $brandKey = $_POST['brand'] ?? '';
    $withdrawAmount = intval($_POST['withdraw_amount'] ?? 0);
    
    $currentBalance = $payrollData['balances'][$brandKey] ?? 0;
    
    if ($withdrawAmount <= 0) {
        $_SESSION['error'] = "❌ Jumlah penarikan tidak valid!";
    } elseif ($withdrawAmount > $currentBalance) {
        $_SESSION['error'] = "❌ Saldo " . $brands[$brandKey]['name'] . " tidak mencukupi! Saldo tersedia: " . formatRupiah($currentBalance);
    } else {
        $payrollData['balances'][$brandKey] = $currentBalance - $withdrawAmount;
        
        // Simpan riwayat penarikan
        if (!isset($payrollData['withdrawals'])) {
            $payrollData['withdrawals'] = [];
        }
        $payrollData['withdrawals'][] = [
            'date' => date('Y-m-d'),
            'brand' => $brandKey,
            'amount' => $withdrawAmount,
            'timestamp' => time()
        ];
        
        saveData($payrollFile, $payrollData);
        $_SESSION['message'] = "✅ Penarikan berhasil dari " . $brands[$brandKey]['name'] . 
                               " sebesar " . formatRupiah($withdrawAmount) . 
                               " | Sisa saldo: " . formatRupiah($payrollData['balances'][$brandKey]);
    }
    
    header('Location: ' . $_SERVER['PHP_SELF'] . '?tab=payroll');
    exit;
}

// Load data untuk tampilan
$data = loadData($dataFile);
$payrollData = loadData($payrollFile);
$today = date('Y-m-d');
$brandPriorities = calculateBrandPriority($data, $brands);
$subjectRecommendation = getSubjectRecommendation($subjects);
$recommendedBrands = array_slice($brandPriorities, 0, 2, true);

// Hitung total upah hari ini per brand
$todayEarnings = [];
$todayTotalEarnings = 0;
foreach ($brands as $brandKey => $brand) {
    $todayEarnings[$brandKey] = 0;
    if (isset($data[$today]['attendance'][$brandKey])) {
        $todayEarnings[$brandKey] = $data[$today]['attendance'][$brandKey]['wage_earned'];
        $todayTotalEarnings += $todayEarnings[$brandKey];
    }
}

// Tab aktif
$activeTab = $_GET['tab'] ?? 'dashboard';

// Fungsi untuk mendapatkan data visualisasi
function getVisualizationData($data, $brands, $days = 7) {
    $chartData = [];
    $dates = [];
    
    for ($i = $days - 1; $i >= 0; $i--) {
        $date = date('Y-m-d', strtotime("-$i days"));
        $dates[] = date('d/m', strtotime($date));
        
        foreach ($brands as $brandKey => $brand) {
            if (!isset($chartData[$brandKey])) {
                $chartData[$brandKey] = ['name' => $brand['name'], 'data' => [], 'color' => $brand['color']];
            }
            
            $hasAttendance = isset($data[$date]['attendance'][$brandKey]);
            $chartData[$brandKey]['data'][] = $hasAttendance ? 1 : 0;
        }
    }
    
    return ['dates' => $dates, 'brands' => $chartData];
}

$chartData = getVisualizationData($data, $brands, 7);
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>📚 Fokus Harian - Brand & Kuliah + Upah Per Brand</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            color: #333;
        }
        
        .container {
            max-width: 1500px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .header {
            text-align: center;
            background: rgba(255, 255, 255, 0.95);
            padding: 20px;
            border-radius: 15px;
            margin-bottom: 20px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
        }
        
        .header h1 {
            color: #4a5568;
            margin-bottom: 10px;
            font-size: 2.5em;
        }
        
        .date-info {
            color: #718096;
            font-size: 1.1em;
        }
        
        /* Navigation Tabs */
        .nav-tabs {
            display: flex;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 15px;
            padding: 8px;
            margin-bottom: 20px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            gap: 8px;
            flex-wrap: wrap;
        }
        
        .nav-tab {
            flex: 1;
            min-width: 150px;
            padding: 15px 20px;
            text-align: center;
            border-radius: 10px;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            color: #4a5568;
            font-weight: 600;
            font-size: 1.1em;
        }
        
        .nav-tab:hover {
            background: rgba(103, 126, 234, 0.1);
            transform: translateY(-2px);
        }
        
        .nav-tab.active {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            box-shadow: 0 4px 15px rgba(103, 126, 234, 0.4);
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
        
        .grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }
        
        .brands-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(320px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }
        
        .card {
            background: rgba(255, 255, 255, 0.95);
            padding: 25px;
            border-radius: 15px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            backdrop-filter: blur(10px);
        }
        
        .card h2 {
            color: #2d3748;
            margin-bottom: 15px;
            font-size: 1.5em;
            border-bottom: 3px solid #e2e8f0;
            padding-bottom: 8px;
        }
        
        .brand-card {
            border: 3px solid;
            position: relative;
            overflow: hidden;
        }
        
        .brand-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 5px;
            background: linear-gradient(90deg, var(--brand-color), rgba(255,255,255,0.3));
        }
        
        .brand-name {
            font-weight: 600;
            font-size: 1.3em;
            margin-bottom: 15px;
            text-align: center;
        }
        
        .balance-display {
            background: linear-gradient(135deg, rgba(255,255,255,0.2), rgba(255,255,255,0.1));
            padding: 20px;
            border-radius: 12px;
            text-align: center;
            margin-bottom: 20px;
        }
        
        .balance-amount {
            font-size: 2.2em;
            font-weight: bold;
            margin-bottom: 5px;
            color: #2d3748;
        }
        
        .today-earnings {
            font-size: 1em;
            color: #4a5568;
        }
        
        .work-status {
            background: #c6f6d5;
            padding: 15px;
            border-radius: 10px;
            text-align: center;
            margin-bottom: 20px;
            border: 2px solid #48bb78;
        }
        
        .work-status.not-attended {
            background: #fef5e7;
            border-color: #ed8936;
            color: #c05621;
        }
        
        .work-status h3 {
            margin-bottom: 8px;
            font-size: 1.1em;
        }
        
        .attend-form {
            display: grid;
            gap: 15px;
        }
        
        .form-group {
            display: flex;
            flex-direction: column;
            gap: 8px;
        }
        
        .form-label {
            font-weight: 600;
            color: #4a5568;
            font-size: 1em;
        }
        
        .hours-input, .withdraw-input {
            padding: 12px 15px;
            border: 2px solid #e2e8f0;
            border-radius: 10px;
            font-size: 1.1em;
            transition: all 0.3s ease;
        }
        
        .hours-input:focus, .withdraw-input:focus {
            outline: none;
            border-color: #4299e1;
            box-shadow: 0 0 0 3px rgba(66, 153, 225, 0.1);
        }
        
        .wage-preview {
            font-size: 0.9em;
            color: #4a5568;
            padding: 8px 12px;
            background: rgba(66, 153, 225, 0.1);
            border-radius: 8px;
            text-align: center;
            margin-top: 5px;
        }
        
        .attend-btn {
            background: linear-gradient(135deg, #4299e1, #3182ce);
            color: white;
            border: none;
            padding: 15px 25px;
            border-radius: 10px;
            cursor: pointer;
            font-weight: 600;
            font-size: 1.1em;
            transition: all 0.3s ease;
        }
        
        .attend-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(66, 153, 225, 0.4);
        }
        
        .attend-btn:disabled {
            background: #a0aec0;
            cursor: not-allowed;
            transform: none;
        }
        
        .withdraw-btn {
            background: linear-gradient(135deg, #f093fb, #f5576c);
            color: white;
            border: none;
            padding: 15px 25px;
            border-radius: 10px;
            cursor: pointer;
            font-weight: 600;
            font-size: 1.1em;
            transition: all 0.3s ease;
        }
        
        .withdraw-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(245, 87, 108, 0.4);
        }
        
        .summary-card {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            grid-column: 1 / -1;
        }
        
        .summary-card h2 {
            color: white;
            border-bottom-color: rgba(255, 255, 255, 0.3);
        }
        
        .summary-stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }
        
        .summary-stat {
            text-align: center;
            background: rgba(255, 255, 255, 0.1);
            padding: 20px;
            border-radius: 12px;
        }
        
        .summary-stat-number {
            font-size: 2.5em;
            font-weight: bold;
            margin-bottom: 8px;
        }
        
        .summary-stat-label {
            font-size: 1.1em;
            opacity: 0.9;
        }
        
        .chart-container {
            grid-column: 1 / -1;
        }
        
        .chart {
            width: 100%;
            height: 200px;
            display: flex;
            align-items: end;
            justify-content: space-evenly;
            padding: 25px 20px;
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.1), rgba(255, 255, 255, 0.05));
            border-radius: 15px;
            margin-bottom: 20px;
            box-shadow: inset 0 2px 10px rgba(0, 0, 0, 0.1);
        }
        
        .chart-bar {
            display: flex;
            flex-direction: column;
            align-items: center;
            flex: 1;
            max-width: 80px;
        }
        
        .bar-group {
            display: flex;
            align-items: end;
            justify-content: center;
            height: 120px;
            margin-bottom: 15px;
            gap: 2px;
        }
        
        .bar {
            width: 12px;
            min-height: 3px;
            border-radius: 6px 6px 0 0;
            transition: all 0.3s ease;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.2);
        }
        
        .bar:hover {
            opacity: 0.8;
            transform: scaleY(1.1) scaleX(1.1);
        }
        
        .chart-date {
            font-size: 0.85em;
            color: #4a5568;
            font-weight: 600;
            text-align: center;
            background: rgba(255, 255, 255, 0.3);
            padding: 4px 8px;
            border-radius: 12px;
            min-width: 45px;
        }
        
        .legend {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
            gap: 15px;
            margin-bottom: 25px;
            padding: 0 10px;
        }
        
        .legend-item {
            display: flex;
            align-items: center;
            gap: 10px;
            font-size: 0.95em;
            font-weight: 500;
            color: #4a5568;
            background: rgba(255, 255, 255, 0.3);
            padding: 8px 12px;
            border-radius: 20px;
        }
        
        .legend-color {
            width: 18px;
            height: 18px;
            border-radius: 50%;
            box-shadow: 0 2px 6px rgba(0, 0, 0, 0.2);
        }
        
        .message {
            background: #c6f6d5;
            color: #22543d;
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 20px;
            text-align: center;
            font-weight: 600;
        }
        
        .error {
            background: #fed7d7;
            color: #c53030;
        }
        
        .stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(140px, 1fr));
            gap: 20px;
            padding: 0 10px;
        }
        
        .stat-item {
            text-align: center;
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.2), rgba(255, 255, 255, 0.1));
            padding: 20px 15px;
            border-radius: 15px;
            border: 1px solid rgba(255, 255, 255, 0.2);
            transition: all 0.3s ease;
        }
        
        .stat-number {
            font-size: 2em;
            font-weight: bold;
            background: linear-gradient(135deg, #4299e1, #3182ce);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            margin-bottom: 5px;
        }
        
        .stat-label {
            font-size: 0.9em;
            color: #4a5568;
            font-weight: 600;
        }
        
        .no-balance {
            text-align: center;
            color: #718096;
            padding: 20px;
            font-style: italic;
            background: rgba(0,0,0,0.02);
            border-radius: 10px;
        }
        
        @media (max-width: 768px) {
            .nav-tabs {
                flex-direction: column;
            }
            
            .nav-tab {
                min-width: auto;
            }
            
            .brands-grid {
                grid-template-columns: 1fr;
            }
            
            .summary-stats {
                grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>📚 Fokus Harian + 💰 Upah Per Brand</h1>
            <div class="date-info">
                <?php echo strftime('%A, %d %B %Y', strtotime($today)); ?>
            </div>
        </div>
        
        <!-- Navigation Tabs -->
        <div class="nav-tabs">
            <a href="?tab=dashboard" class="nav-tab <?php echo $activeTab === 'dashboard' ? 'active' : ''; ?>">
                🏠 Dashboard
            </a>
            <a href="?tab=attendance" class="nav-tab <?php echo $activeTab === 'attendance' ? 'active' : ''; ?>">
                📝 Absensi
            </a>
            <a href="?tab=payroll" class="nav-tab <?php echo $activeTab === 'payroll' ? 'active' : ''; ?>">
                💰 Kelola Upah
            </a>
            <a href="?tab=analytics" class="nav-tab <?php echo $activeTab === 'analytics' ? 'active' : ''; ?>">
                📊 Analitik
            </a>
        </div>
        
        <?php if (isset($_SESSION['message'])): ?>
            <div class="message">
                <?php echo $_SESSION['message']; unset($_SESSION['message']); ?>
            </div>
        <?php endif; ?>
        
        <?php if (isset($_SESSION['error'])): ?>
            <div class="message error">
                <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
            </div>
        <?php endif; ?>
        
        <!-- Dashboard Tab -->
        <div class="tab-content <?php echo $activeTab === 'dashboard' ? 'active' : ''; ?>">
            <div class="grid">
                <!-- Summary Card -->
                <div class="card summary-card">
                    <h2>📊 Ringkasan Hari Ini</h2>
                    <div class="summary-stats">
                        <div class="summary-stat">
                            <div class="summary-stat-number"><?php echo formatRupiah(array_sum($payrollData['balances'] ?? [])); ?></div>
                            <div class="summary-stat-label">Total Saldo</div>
                        </div>
                        <div class="summary-stat">
                            <div class="summary-stat-number"><?php echo formatRupiah($todayTotalEarnings); ?></div>
                            <div class="summary-stat-label">Upah Hari Ini</div>
                        </div>
                        <div class="summary-stat">
                            <div class="summary-stat-number"><?php echo count($data[$today]['attendance'] ?? []); ?></div>
                            <div class="summary-stat-label">Brand Dikerjakan</div>
                        </div>
                        <div class="summary-stat">
                            <div class="summary-stat-number">
                                <?php 
                                $todayHours = 0;
                                if (isset($data[$today]['attendance'])) {
                                    foreach ($data[$today]['attendance'] as $attendance) {
                                        $todayHours += $attendance['work_hours'] ?? 0;
                                    }
                                }
                                echo number_format($todayHours, 1);
                                ?>
                            </div>
                            <div class="summary-stat-label">Jam Kerja Hari Ini</div>
                        </div>
                    </div>
                </div>
                
                <!-- Quick Status per Brand -->
                <div class="card">
                    <h2>🎯 Status Brand Hari Ini</h2>
                    <?php foreach ($brands as $brandKey => $brand): ?>
                        <?php $isAttended = isset($data[$today]['attendance'][$brandKey]); ?>
                        <div style="display: flex; justify-content: space-between; align-items: center; 
                                    padding: 12px 15px; margin-bottom: 10px; border-radius: 8px;
                                    background: <?php echo $isAttended ? '#c6f6d5' : '#fef5e7'; ?>;
                                    border-left: 4px solid <?php echo $brand['color']; ?>;">
                            <div>
                                <div style="font-weight: 600; color: #2d3748;"><?php echo $brand['name']; ?></div>
                                <div style="font-size: 0.9em; color: #718096;">
                                    Saldo: <?php echo formatRupiah($payrollData['balances'][$brandKey] ?? 0); ?>
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <?php if ($isAttended): ?>
                                    <div style="color: #22543d; font-weight: 600;">
                                        ✅ <?php echo $data[$today]['attendance'][$brandKey]['work_hours']; ?> jam
                                    </div>
                                    <div style="font-size: 0.9em; color: #22543d;">
                                        <?php echo formatRupiah($data[$today]['attendance'][$brandKey]['wage_earned']); ?>
                                    </div>
                                <?php else: ?>
                                    <div style="color: #c05621; font-weight: 600;">⏳ Belum Absen</div>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
                
                <!-- Mata Kuliah Recommendation -->
                <div class="card" style="background: linear-gradient(135deg, #48bb78, #38a169); color: white;">
                    <h2 style="color: white; border-bottom-color: rgba(255, 255, 255, 0.3);">🎓 Mata Kuliah Hari Ini</h2>
                    <?php foreach ($subjectRecommendation as $subjectKey => $subject): ?>
                        <div style="background: rgba(255, 255, 255, 0.1); padding: 20px; border-radius: 12px; text-align: center;">
                            <div style="font-size: 1.3em; font-weight: 600; margin-bottom: 8px;">
                                <?php echo $subject['name']; ?>
                            </div>
                            <div style="font-size: 1em; opacity: 0.9;">
                                Kode: <?php echo $subject['code']; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        
        <!-- Attendance Tab -->
        <div class="tab-content <?php echo $activeTab === 'attendance' ? 'active' : ''; ?>">
            <div class="brands-grid">
                <?php foreach ($brands as $brandKey => $brand): ?>
                    <div class="card brand-card" style="--brand-color: <?php echo $brand['color']; ?>; border-color: <?php echo $brand['color']; ?>;">
                        <div class="brand-name" style="color: <?php echo $brand['color']; ?>;"><?php echo $brand['name']; ?></div>
                        
                        <!-- Status Absensi -->
                        <?php $isAttended = isset($data[$today]['attendance'][$brandKey]); ?>
                        <div class="work-status <?php echo $isAttended ? '' : 'not-attended'; ?>">
                            <?php if ($isAttended): ?>
                                <h3 style="color: #22543d;">✅ Sudah Absen Hari Ini</h3>
                                <div style="color: #22543d;">
                                    Jam kerja: <strong><?php echo $data[$today]['attendance'][$brandKey]['work_hours']; ?> jam</strong><br>
                                    Upah diperoleh: <strong><?php echo formatRupiah($data[$today]['attendance'][$brandKey]['wage_earned']); ?></strong><br>
                                    Waktu absen: <?php echo date('H:i', $data[$today]['attendance'][$brandKey]['timestamp']); ?>
                                </div>
                            <?php else: ?>
                                <h3>⏳ Belum Absen Hari Ini</h3>
                                <div>Silakan isi jam kerja untuk absen</div>
                            <?php endif; ?>
                        </div>
                        
                        <!-- Form Absensi -->
                        <?php if (!$isAttended): ?>
                            <form method="POST" class="attend-form">
                                <input type="hidden" name="action" value="attend">
                                <input type="hidden" name="brand" value="<?php echo $brandKey; ?>">
                                
                                <div class="form-group">
                                    <label class="form-label">🕐 Jam Kerja Efektif</label>
                                    <input type="number" 
                                           name="work_hours" 
                                           class="hours-input" 
                                           placeholder="Contoh: 2.5 (untuk 2 jam 30 menit)" 
                                           step="0.5" 
                                           min="0.5" 
                                           max="12" 
                                           required>
                                    <div class="wage-preview" id="preview-<?php echo $brandKey; ?>" style="display: none;"></div>
                                </div>
                                
                                <button type="submit" class="attend-btn">📝 Submit Absensi</button>
                            </form>
                        <?php endif; ?>
                        
                        <!-- Weekly Stats -->
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-top: 20px;">
                            <div style="text-align: center; background: rgba(0,0,0,0.03); padding: 15px; border-radius: 10px;">
                                <div style="font-size: 1.8em; font-weight: bold; color: <?php echo $brand['color']; ?>;">
                                    <?php 
                                    $weeklyHours = 0;
                                    for ($i = 0; $i < 7; $i++) {
                                        $checkDate = date('Y-m-d', strtotime("-$i days"));
                                        if (isset($data[$checkDate]['attendance'][$brandKey])) {
                                            $weeklyHours += $data[$checkDate]['attendance'][$brandKey]['work_hours'];
                                        }
                                    }
                                    echo number_format($weeklyHours, 1);
                                    ?>
                                </div>
                                <div style="font-size: 0.9em; color: #718096; font-weight: 600;">Jam Minggu Ini</div>
                            </div>
                            <div style="text-align: center; background: rgba(0,0,0,0.03); padding: 15px; border-radius: 10px;">
                                <div style="font-size: 1.8em; font-weight: bold; color: <?php echo $brand['color']; ?>;">
                                    <?php 
                                    $weeklyDays = 0;
                                    for ($i = 0; $i < 7; $i++) {
                                        $checkDate = date('Y-m-d', strtotime("-$i days"));
                                        if (isset($data[$checkDate]['attendance'][$brandKey])) {
                                            $weeklyDays++;
                                        }
                                    }
                                    echo $weeklyDays;
                                    ?>
                                </div>
                                <div style="font-size: 0.9em; color: #718096; font-weight: 600;">Hari Minggu Ini</div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        
        <!-- Payroll Tab -->
        <div class="tab-content <?php echo $activeTab === 'payroll' ? 'active' : ''; ?>">
            <div class="brands-grid">
                <?php foreach ($brands as $brandKey => $brand): ?>
                    <div class="card brand-card" style="--brand-color: <?php echo $brand['color']; ?>; border-color: <?php echo $brand['color']; ?>;">
                        <div class="brand-name" style="color: <?php echo $brand['color']; ?>;"><?php echo $brand['name']; ?></div>
                        
                        <!-- Saldo Display -->
                        <div class="balance-display">
                            <div class="balance-amount"><?php echo formatRupiah($payrollData['balances'][$brandKey] ?? 0); ?></div>
                            <div class="today-earnings">Upah hari ini: <?php echo formatRupiah($todayEarnings[$brandKey]); ?></div>
                        </div>
                        
                        <!-- Form Penarikan -->
                        <?php if (($payrollData['balances'][$brandKey] ?? 0) > 0): ?>
                            <form method="POST" class="attend-form">
                                <input type="hidden" name="action" value="withdraw">
                                <input type="hidden" name="brand" value="<?php echo $brandKey; ?>">
                                
                                <div class="form-group">
                                    <label class="form-label">💸 Jumlah Penarikan</label>
                                    <input type="number" 
                                           name="withdraw_amount" 
                                           class="withdraw-input" 
                                           placeholder="Masukkan jumlah (min. Rp 1.000)" 
                                           min="1000" 
                                           step="1000"
                                           max="<?php echo $payrollData['balances'][$brandKey] ?? 0; ?>" 
                                           required>
                                    <div style="font-size: 0.85em; color: #718096; margin-top: 5px;">
                                        Maksimal: <?php echo formatRupiah($payrollData['balances'][$brandKey] ?? 0); ?>
                                    </div>
                                </div>
                                
                                <button type="submit" class="withdraw-btn">💸 Tarik Upah</button>
                            </form>
                        <?php else: ?>
                            <div class="no-balance">
                                <div style="font-size: 1.2em; margin-bottom: 8px;">💰</div>
                                <div>Belum ada saldo untuk ditarik</div>
                                <div style="font-size: 0.9em; margin-top: 5px; opacity: 0.8;">
                                    Lakukan absensi untuk mendapatkan upah
                                </div>
                            </div>
                        <?php endif; ?>
                        
                        <!-- Recent Transactions for this brand -->
                        <div style="margin-top: 20px;">
                            <h3 style="font-size: 1.1em; color: #4a5568; margin-bottom: 10px; border-bottom: 2px solid #e2e8f0; padding-bottom: 5px;">
                                📋 Transaksi Terakhir
                            </h3>
                            
                            <?php
                            $brandTransactions = [];
                            
                            // Earnings untuk brand ini
                            if (!empty($payrollData['earnings'])) {
                                foreach ($payrollData['earnings'] as $earning) {
                                    if ($earning['brand'] === $brandKey) {
                                        $brandTransactions[] = [
                                            'type' => 'earning',
                                            'timestamp' => $earning['timestamp'],
                                            'amount' => $earning['wage_earned'],
                                            'work_hours' => $earning['work_hours']
                                        ];
                                    }
                                }
                            }
                            
                            // Withdrawals untuk brand ini
                            if (!empty($payrollData['withdrawals'])) {
                                foreach ($payrollData['withdrawals'] as $withdrawal) {
                                    if ($withdrawal['brand'] === $brandKey) {
                                        $brandTransactions[] = [
                                            'type' => 'withdrawal',
                                            'timestamp' => $withdrawal['timestamp'],
                                            'amount' => $withdrawal['amount']
                                        ];
                                    }
                                }
                            }
                            
                            // Urutkan dan ambil 5 terakhir
                            usort($brandTransactions, function($a, $b) {
                                return $b['timestamp'] - $a['timestamp'];
                            });
                            $brandTransactions = array_slice($brandTransactions, 0, 5);
                            ?>
                            
                            <?php if (empty($brandTransactions)): ?>
                                <div style="text-align: center; color: #718096; font-style: italic; padding: 15px;">
                                    Belum ada transaksi
                                </div>
                            <?php else: ?>
                                <div style="display: grid; gap: 8px;">
                                    <?php foreach ($brandTransactions as $transaction): ?>
                                        <div style="display: flex; justify-content: space-between; align-items: center;
                                                    padding: 10px 12px; background: rgba(0,0,0,0.02); border-radius: 6px;
                                                    border-left: 3px solid <?php echo $transaction['type'] === 'earning' ? $brand['color'] : '#f56565'; ?>;">
                                            <div style="font-size: 0.9em;">
                                                <?php if ($transaction['type'] === 'earning'): ?>
                                                    <span style="color: #22543d;">💰 <?php echo $transaction['work_hours']; ?> jam</span>
                                                <?php else: ?>
                                                    <span style="color: #c53030;">💸 Penarikan</span>
                                                <?php endif; ?>
                                                <div style="font-size: 0.85em; color: #718096;">
                                                    <?php echo date('d/m H:i', $transaction['timestamp']); ?>
                                                </div>
                                            </div>
                                            <div style="font-weight: 600; color: <?php echo $transaction['type'] === 'earning' ? '#22543d' : '#c53030'; ?>;">
                                                <?php echo $transaction['type'] === 'earning' ? '+' : '-'; ?><?php echo formatRupiah($transaction['amount']); ?>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        
        <!-- Analytics Tab -->
        <div class="tab-content <?php echo $activeTab === 'analytics' ? 'active' : ''; ?>">
            <!-- Kartu Visualisasi -->
            <div class="card chart-container">
                <h2>📈 Visualisasi Kehadiran 7 Hari Terakhir</h2>
                <div class="chart">
                    <?php foreach ($chartData['dates'] as $index => $date): ?>
                        <div class="chart-bar">
                            <div class="bar-group">
                                <?php foreach ($chartData['brands'] as $brandKey => $brandChart): ?>
                                    <div class="bar" 
                                         style="height: <?php echo $brandChart['data'][$index] * 80 + 8; ?>px; 
                                                background-color: <?php echo $brandChart['color']; ?>;"
                                         title="<?php echo $brandChart['name'] . ' - ' . $date; ?>">
                                    </div>
                                <?php endforeach; ?>
                            </div>
                            <div class="chart-date"><?php echo $date; ?></div>
                        </div>
                    <?php endforeach; ?>
                </div>
                
                <div class="legend">
                    <?php foreach ($chartData['brands'] as $brandKey => $brandChart): ?>
                        <div class="legend-item">
                            <div class="legend-color" style="background-color: <?php echo $brandChart['color']; ?>"></div>
                            <span><?php echo $brandChart['name']; ?></span>
                        </div>
                    <?php endforeach; ?>
                </div>
                
                <div class="stats">
                    <?php 
                    $todayAttendance = count($data[$today]['attendance'] ?? []);
                    $weeklyAttendance = 0;
                    $weeklyTotalEarnings = 0;
                    $weeklyTotalHours = 0;
                    
                    for ($i = 0; $i < 7; $i++) {
                        $checkDate = date('Y-m-d', strtotime("-$i days"));
                        if (isset($data[$checkDate]['attendance'])) {
                            $weeklyAttendance += count($data[$checkDate]['attendance']);
                            foreach ($data[$checkDate]['attendance'] as $attendance) {
                                $weeklyTotalEarnings += $attendance['wage_earned'] ?? 0;
                                $weeklyTotalHours += $attendance['work_hours'] ?? 0;
                            }
                        }
                    }
                    ?>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo $todayAttendance; ?></div>
                        <div class="stat-label">Absen Hari Ini</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo $weeklyAttendance; ?></div>
                        <div class="stat-label">Total 7 Hari</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo number_format($weeklyTotalHours, 1); ?></div>
                        <div class="stat-label">Jam Kerja Minggu</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo formatRupiah($weeklyTotalEarnings); ?></div>
                        <div class="stat-label">Upah 7 Hari</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo number_format($weeklyTotalHours / 7, 1); ?></div>
                        <div class="stat-label">Rata-rata Jam/Hari</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo formatRupiah($weeklyTotalEarnings / 7); ?></div>
                        <div class="stat-label">Rata-rata Upah/Hari</div>
                    </div>
                </div>
            </div>
            
            <!-- Detailed Analytics per Brand -->
            <div class="brands-grid" style="margin-top: 20px;">
                <?php foreach ($brands as $brandKey => $brand): ?>
                    <div class="card" style="border-left: 5px solid <?php echo $brand['color']; ?>;">
                        <h2 style="color: <?php echo $brand['color']; ?>;"><?php echo $brand['name']; ?> - Analytics</h2>
                        
                        <?php
                        // Hitung statistik brand
                        $brandWeeklyHours = 0;
                        $brandWeeklyEarnings = 0;
                        $brandWeeklyDays = 0;
                        
                        for ($i = 0; $i < 7; $i++) {
                            $checkDate = date('Y-m-d', strtotime("-$i days"));
                            if (isset($data[$checkDate]['attendance'][$brandKey])) {
                                $brandWeeklyHours += $data[$checkDate]['attendance'][$brandKey]['work_hours'];
                                $brandWeeklyEarnings += $data[$checkDate]['attendance'][$brandKey]['wage_earned'];
                                $brandWeeklyDays++;
                            }
                        }
                        
                        $avgHoursPerDay = $brandWeeklyDays > 0 ? $brandWeeklyHours / $brandWeeklyDays : 0;
                        $consistency = ($brandWeeklyDays / 7) * 100;
                        ?>
                        
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                            <div style="text-align: center; background: rgba(0,0,0,0.03); padding: 15px; border-radius: 8px;">
                                <div style="font-size: 1.5em; font-weight: bold; color: <?php echo $brand['color']; ?>;">
                                    <?php echo formatRupiah($brandWeeklyEarnings); ?>
                                </div>
                                <div style="font-size: 0.9em; color: #718096;">Upah 7 Hari</div>
                            </div>
                            <div style="text-align: center; background: rgba(0,0,0,0.03); padding: 15px; border-radius: 8px;">
                                <div style="font-size: 1.5em; font-weight: bold; color: <?php echo $brand['color']; ?>;">
                                    <?php echo number_format($brandWeeklyHours, 1); ?>
                                </div>
                                <div style="font-size: 0.9em; color: #718096;">Total Jam</div>
                            </div>
                            <div style="text-align: center; background: rgba(0,0,0,0.03); padding: 15px; border-radius: 8px;">
                                <div style="font-size: 1.5em; font-weight: bold; color: <?php echo $brand['color']; ?>;">
                                    <?php echo number_format($avgHoursPerDay, 1); ?>
                                </div>
                                <div style="font-size: 0.9em; color: #718096;">Rata-rata Jam/Hari</div>
                            </div>
                            <div style="text-align: center; background: rgba(0,0,0,0.03); padding: 15px; border-radius: 8px;">
                                <div style="font-size: 1.5em; font-weight: bold; color: <?php echo $brand['color']; ?>;">
                                    <?php echo number_format($consistency, 1); ?>%
                                </div>
                                <div style="font-size: 0.9em; color: #718096;">Konsistensi</div>
                            </div>
                        </div>
                        
                        <!-- Quick Actions -->
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 10px; margin-top: 20px;">
                            <a href="?tab=attendance" style="text-decoration: none; color: white; background: <?php echo $brand['color']; ?>; padding: 12px; border-radius: 8px; text-align: center; font-weight: 600; transition: all 0.3s ease;">
                                📝 Absen
                            </a>
                            <div style="color: #718096; background: rgba(0,0,0,0.05); padding: 12px; border-radius: 8px; text-align: center; font-weight: 600;">
                                📊 Saldo: <?php echo formatRupiah($payrollData['balances'][$brandKey] ?? 0); ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
            
            <!-- Withdrawal History -->
            <?php if (!empty($payrollData['withdrawals'])): ?>
            <div class="card" style="margin-top: 20px;">
                <h2>💸 Riwayat Penarikan (10 Terakhir)</h2>
                
                <div style="display: grid; gap: 10px;">
                    <?php 
                    $recentWithdrawals = array_slice($payrollData['withdrawals'], -10);
                    $recentWithdrawals = array_reverse($recentWithdrawals);
                    ?>
                    <?php foreach ($recentWithdrawals as $withdrawal): ?>
                        <div style="display: flex; justify-content: space-between; align-items: center; 
                                    padding: 15px; background: #fef5e7; border-radius: 10px; 
                                    border-left: 5px solid <?php echo $brands[$withdrawal['brand']]['color']; ?>;">
                            <div>
                                <div style="font-weight: 600; color: #c05621; display: flex; align-items: center; gap: 8px;">
                                    <span style="color: <?php echo $brands[$withdrawal['brand']]['color']; ?>;">💸</span>
                                    <?php echo $brands[$withdrawal['brand']]['name']; ?>
                                </div>
                                <div style="font-size: 0.9em; color: #718096; margin-top: 4px;">
                                    <?php echo date('d/m/Y H:i', $withdrawal['timestamp']); ?>
                                </div>
                            </div>
                            <div style="font-weight: bold; font-size: 1.2em; color: #c53030;">
                                -<?php echo formatRupiah($withdrawal['amount']); ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>
        </div>
        
        <!-- All Transactions History -->
        <?php if (!empty($payrollData['earnings']) || !empty($payrollData['withdrawals'])): ?>
        <div class="card" style="margin-top: 20px; <?php echo $activeTab !== 'analytics' ? 'display: none;' : ''; ?>">
            <h2>📋 Semua Riwayat Transaksi (15 Terakhir)</h2>
            
            <?php
            $allTransactions = [];
            
            // Gabungkan earnings
            if (!empty($payrollData['earnings'])) {
                foreach (array_slice($payrollData['earnings'], -10) as $earning) {
                    $allTransactions[] = [
                        'type' => 'earning',
                        'date' => $earning['date'],
                        'timestamp' => $earning['timestamp'],
                        'brand' => $earning['brand'],
                        'amount' => $earning['wage_earned'],
                        'work_hours' => $earning['work_hours']
                    ];
                }
            }
            
            // Gabungkan withdrawals
            if (!empty($payrollData['withdrawals'])) {
                foreach (array_slice($payrollData['withdrawals'], -10) as $withdrawal) {
                    $allTransactions[] = [
                        'type' => 'withdrawal',
                        'date' => $withdrawal['date'],
                        'timestamp' => $withdrawal['timestamp'],
                        'brand' => $withdrawal['brand'],
                        'amount' => $withdrawal['amount']
                    ];
                }
            }
            
            // Urutkan berdasarkan timestamp
            usort($allTransactions, function($a, $b) {
                return $b['timestamp'] - $a['timestamp'];
            });
            
            $allTransactions = array_slice($allTransactions, 0, 15);
            ?>
            
            <div style="display: grid; gap: 12px;">
                <?php foreach ($allTransactions as $transaction): ?>
                    <div style="display: flex; justify-content: space-between; align-items: center; 
                                padding: 18px; background: #f8f9fa; border-radius: 12px; 
                                border-left: 5px solid <?php echo $transaction['type'] === 'earning' ? $brands[$transaction['brand']]['color'] : '#f56565'; ?>;">
                        <div>
                            <?php if ($transaction['type'] === 'earning'): ?>
                                <div style="font-weight: 600; color: #22543d; display: flex; align-items: center; gap: 8px;">
                                    <span style="color: <?php echo $brands[$transaction['brand']]['color']; ?>;">💰</span>
                                    <?php echo $brands[$transaction['brand']]['name']; ?>
                                </div>
                                <div style="font-size: 0.9em; color: #718096; margin-top: 4px;">
                                    <?php echo date('d/m/Y H:i', $transaction['timestamp']); ?> | 
                                    <?php echo $transaction['work_hours']; ?> jam kerja
                                </div>
                            <?php else: ?>
                                <div style="font-weight: 600; color: #c53030; display: flex; align-items: center; gap: 8px;">
                                    <span>💸</span>
                                    Penarikan dari <?php echo $brands[$transaction['brand']]['name']; ?>
                                </div>
                                <div style="font-size: 0.9em; color: #718096; margin-top: 4px;">
                                    <?php echo date('d/m/Y H:i', $transaction['timestamp']); ?>
                                </div>
                            <?php endif; ?>
                        </div>
                        <div style="font-weight: bold; font-size: 1.2em; color: <?php echo $transaction['type'] === 'earning' ? '#22543d' : '#c53030'; ?>;">
                            <?php echo $transaction['type'] === 'earning' ? '+' : '-'; ?><?php echo formatRupiah($transaction['amount']); ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <script>
        // Tab Navigation
        document.querySelectorAll('.nav-tab').forEach(tab => {
            tab.addEventListener('click', function(e) {
                e.preventDefault();
                const tabName = new URL(this.href).searchParams.get('tab') || 'dashboard';
                
                // Update URL without reload
                window.history.pushState({}, '', `?tab=${tabName}`);
                
                // Update active tab
                document.querySelectorAll('.nav-tab').forEach(t => t.classList.remove('active'));
                this.classList.add('active');
                
                // Show/hide content
                document.querySelectorAll('.tab-content').forEach(content => {
                    content.classList.remove('active');
                });
                document.querySelector(`.tab-content:nth-child(${Array.from(document.querySelectorAll('.tab-content')).findIndex(el => el.getAttribute('data-tab') === tabName || (tabName === 'dashboard' && !el.getAttribute('data-tab'))) + 4})`);
                
                // Simple approach - show the right content
                const contents = document.querySelectorAll('.tab-content');
                if (tabName === 'dashboard') contents[0].classList.add('active');
                else if (tabName === 'attendance') contents[1].classList.add('active');
                else if (tabName === 'payroll') contents[2].classList.add('active');
                else if (tabName === 'analytics') contents[3].classList.add('active');
            });
        });
        
        // Preview upah saat input jam kerja
        document.querySelectorAll('input[name="work_hours"]').forEach(input => {
            input.addEventListener('input', function() {
                const hours = parseFloat(this.value || 0);
                const wage = Math.floor(hours * 6000);
                const brandKey = this.closest('form').querySelector('input[name="brand"]').value;
                const previewEl = document.getElementById(`preview-${brandKey}`);
                
                if (hours > 0 && previewEl) {
                    previewEl.style.display = 'block';
                    previewEl.innerHTML = `💰 Upah yang akan diperoleh: <strong>${wage.toLocaleString('id-ID', { style: 'currency', currency: 'IDR', minimumFractionDigits: 0 })}</strong>`;
                } else if (previewEl) {
                    previewEl.style.display = 'none';
                }
            });
        });
        
        // Validation untuk jam kerja
        document.querySelectorAll('input[name="work_hours"]').forEach(input => {
            input.addEventListener('input', function() {
                let value = parseFloat(this.value);
                if (value > 12) {
                    this.setCustomValidity('Jam kerja tidak boleh lebih dari 12 jam');
                } else if (value < 0.5 && value > 0) {
                    this.setCustomValidity('Minimum jam kerja adalah 0.5 jam');
                } else {
                    this.setCustomValidity('');
                }
            });
        });
        
        // Format input dengan thousands separator
        document.querySelectorAll('input[name="withdraw_amount"]').forEach(input => {
            input.addEventListener('input', function() {
                let value = this.value.replace(/\D/g, '');
                if (value) {
                    value = parseInt(value);
                    this.value = value;
                }
            });
        });
        
        // Konfirmasi sebelum penarikan besar
        document.querySelectorAll('form').forEach(form => {
            if (form.querySelector('input[name="action"][value="withdraw"]')) {
                form.addEventListener('submit', function(e) {
                    const amount = parseInt(this.querySelector('input[name="withdraw_amount"]').value || 0);
                    const brandName = this.closest('.brand-card').querySelector('.brand-name').textContent;
                    
                    if (amount >= 50000) {
                        if (!confirm(`Anda yakin ingin menarik ${amount.toLocaleString('id-ID')} rupiah dari ${brandName}?`)) {
                            e.preventDefault();
                        }
                    }
                });
            }
        });
        
        // Smooth hover effects
        document.querySelectorAll('.attend-btn, .withdraw-btn').forEach(btn => {
            btn.addEventListener('mouseenter', function() {
                if (!this.disabled) {
                    this.style.transform = 'scale(1.05)';
                }
            });
            
            btn.addEventListener('mouseleave', function() {
                this.style.transform = 'scale(1)';
            });
        });
        
        // Auto refresh setiap 5 menit
        setTimeout(function() {
            location.reload();
        }, 300000);
        
        // Handle back/forward browser navigation
        window.addEventListener('popstate', function() {
            location.reload();
        });
    </script>
</body>
</html>