<?php
// Include configuration file if not already included
if (!function_exists('getEntities')) {
    require_once 'config.php';
}

// Function to get all products
function getProducts() {
    global $conn;
    
    $sql = "SELECT * FROM products ORDER BY name";
    $result = $conn->query($sql);
    
    $products = array();
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $products[] = $row;
        }
    }
    
    return $products;
}

// Function to get a single product
function getProduct($id) {
    global $conn;
    
    $sql = "SELECT * FROM products WHERE id = " . (int)$id;
    $result = $conn->query($sql);
    
    if ($result->num_rows > 0) {
        return $result->fetch_assoc();
    }
    
    return null;
}

// Function to add a new product
function addProduct($name, $description, $unit) {
    global $conn;
    
    $name = sanitize($name);
    $description = sanitize($description);
    $unit = sanitize($unit);
    
    $sql = "INSERT INTO products (name, description, unit) VALUES ('$name', '$description', '$unit')";
    
    if ($conn->query($sql) === TRUE) {
        logActivity('Product', "Created product: $name");
        return $conn->insert_id;
    }
    
    return false;
}

// Function to update a product
function updateProduct($id, $name, $description, $unit) {
    global $conn;
    
    $id = (int)$id;
    $name = sanitize($name);
    $description = sanitize($description);
    $unit = sanitize($unit);
    
    $sql = "UPDATE products SET name = '$name', description = '$description', unit = '$unit' WHERE id = $id";
    
    if ($conn->query($sql) === TRUE) {
        logActivity('Product', "Updated product #$id: $name");
        return true;
    }
    
    return false;
}

// Function to delete a product
function deleteProduct($id) {
    global $conn;
    
    $id = (int)$id;
    
    // Check if product has inventory
    $sql = "SELECT COUNT(*) as count FROM inventory WHERE product_id = $id AND quantity > 0";
    $result = $conn->query($sql);
    $row = $result->fetch_assoc();
    
    if ($row['count'] > 0) {
        return 'Cannot delete product that has inventory';
    }
    
    // Check if product is used in sales
    $sql = "SELECT COUNT(*) as count FROM sale_items WHERE product_id = $id";
    $result = $conn->query($sql);
    $row = $result->fetch_assoc();
    
    if ($row['count'] > 0) {
        return 'Cannot delete product that has been used in sales';
    }
    
    $sql = "DELETE FROM products WHERE id = $id";
    
    if ($conn->query($sql) === TRUE) {
        // Also delete from inventory where quantity is 0
        $conn->query("DELETE FROM inventory WHERE product_id = $id AND quantity = 0");
        
        logActivity('Product', "Deleted product #$id");
        return true;
    }
    
    return false;
}

// Function to get inventory for an entity
function getInventory($entityId = null) {
    global $conn;
    
    $sql = "SELECT i.*, p.name as product_name, p.unit, e.name as entity_name 
            FROM inventory i
            JOIN products p ON i.product_id = p.id
            JOIN entities e ON i.entity_id = e.id
            WHERE 1=1 ";
    
    if ($entityId !== null) {
        $sql .= "AND i.entity_id = " . (int)$entityId . " ";
    }
    
    $sql .= "ORDER BY e.name, p.name";
    
    $result = $conn->query($sql);
    
    $inventory = array();
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $inventory[] = $row;
        }
    }
    
    return $inventory;
}

// Function to get inventory item
function getInventoryItem($entityId, $productId) {
    global $conn;
    
    $entityId = (int)$entityId;
    $productId = (int)$productId;
    
    $sql = "SELECT i.*, p.name as product_name, p.unit 
            FROM inventory i
            JOIN products p ON i.product_id = p.id
            WHERE i.entity_id = $entityId AND i.product_id = $productId";
    
    $result = $conn->query($sql);
    
    if ($result->num_rows > 0) {
        return $result->fetch_assoc();
    }
    
    return null;
}

// Function to update inventory pricing
function updateInventoryPricing($entityId, $productId, $purchasePrice, $sellingPrice) {
    global $conn;
    
    $entityId = (int)$entityId;
    $productId = (int)$productId;
    $purchasePrice = (float)$purchasePrice;
    $sellingPrice = (float)$sellingPrice;
    
    // Check if inventory record exists
    $item = getInventoryItem($entityId, $productId);
    
    if ($item === null) {
        // Create new inventory record with zero quantity
        $sql = "INSERT INTO inventory (entity_id, product_id, quantity, purchase_price, selling_price) 
                VALUES ($entityId, $productId, 0, $purchasePrice, $sellingPrice)";
    } else {
        // Update existing record
        $sql = "UPDATE inventory SET purchase_price = $purchasePrice, selling_price = $sellingPrice 
                WHERE entity_id = $entityId AND product_id = $productId";
    }
    
    if ($conn->query($sql) === TRUE) {
        logActivity('Inventory', "Updated pricing for product #$productId in entity #$entityId");
        return true;
    }
    
    return false;
}

// Function to adjust inventory
function adjustInventory($entityId, $productId, $quantity, $purchasePrice, $notes) {
    global $conn;
    
    $entityId = (int)$entityId;
    $productId = (int)$productId;
    $quantity = (float)$quantity;
    $purchasePrice = (float)$purchasePrice;
    $notes = sanitize($notes);
    
    // Begin transaction
    $conn->begin_transaction();
    
    try {
        // Check if inventory record exists
        $item = getInventoryItem($entityId, $productId);
        
        if ($item === null) {
            // Create new inventory record
            $sql = "INSERT INTO inventory (entity_id, product_id, quantity, purchase_price, selling_price) 
                    VALUES ($entityId, $productId, $quantity, $purchasePrice, $purchasePrice * 1.2)";
        } else {
            // Update existing record - add quantity and recalculate average purchase price
            $newQuantity = $item['quantity'] + $quantity;
            
            // Only recalculate purchase price if adding stock
            if ($quantity > 0) {
                $totalValue = ($item['quantity'] * $item['purchase_price']) + ($quantity * $purchasePrice);
                $newPurchasePrice = $newQuantity > 0 ? $totalValue / $newQuantity : $purchasePrice;
                
                $sql = "UPDATE inventory SET quantity = $newQuantity, purchase_price = $newPurchasePrice 
                        WHERE entity_id = $entityId AND product_id = $productId";
            } else {
                $sql = "UPDATE inventory SET quantity = $newQuantity 
                        WHERE entity_id = $entityId AND product_id = $productId";
            }
        }
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to update inventory');
        }
        
        // Record stock movement
        $sql = "INSERT INTO stock_movements (entity_id, product_id, quantity, movement_type, unit_price, notes) 
                VALUES ($entityId, $productId, $quantity, 'adjustment', $purchasePrice, '$notes')";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to record stock movement');
        }
        
        // Commit transaction
        $conn->commit();
        
        logActivity('Inventory', "Adjusted inventory for product #$productId in entity #$entityId by $quantity");
        
        return true;
    } catch (Exception $e) {
        // Rollback transaction
        $conn->rollback();
        return $e->getMessage();
    }
}

// Function to get stock movements
function getStockMovements($entityId = null, $productId = null) {
    global $conn;
    
    $sql = "SELECT sm.*, p.name as product_name, p.unit, e.name as entity_name 
            FROM stock_movements sm
            JOIN products p ON sm.product_id = p.id
            JOIN entities e ON sm.entity_id = e.id
            WHERE 1=1 ";
    
    if ($entityId !== null) {
        $sql .= "AND sm.entity_id = " . (int)$entityId . " ";
    }
    
    if ($productId !== null) {
        $sql .= "AND sm.product_id = " . (int)$productId . " ";
    }
    
    $sql .= "ORDER BY sm.created_at DESC";
    
    $result = $conn->query($sql);
    
    $movements = array();
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $movements[] = $row;
        }
    }
    
    return $movements;
}

// Function to transfer money between entities (not a loan)
function transferMoney($fromEntityId, $toEntityId, $amount, $description) {
    global $conn;
    
    $fromEntityId = (int)$fromEntityId;
    $toEntityId = (int)$toEntityId;
    $amount = (float)$amount;
    $description = sanitize($description);
    
    // Validate entities are different
    if ($fromEntityId === $toEntityId) {
        return 'Cannot transfer money to the same entity';
    }
    
    // Validate amount is positive
    if ($amount <= 0) {
        return 'Transfer amount must be positive';
    }
    
    // Get from entity
    $fromEntity = getEntity($fromEntityId);
    
    // Check if from entity has enough balance
    if ($fromEntity['balance'] < $amount) {
        return 'Entity does not have enough balance for this transfer';
    }
    
    // Begin transaction
    $conn->begin_transaction();
    
    try {
        // Insert transfer record
        $sql = "INSERT INTO transfers (from_entity_id, to_entity_id, amount, description) 
                VALUES ($fromEntityId, $toEntityId, $amount, '$description')";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to create transfer record');
        }
        
        $transferId = $conn->insert_id;
        
        // Update from entity balance
        $sql = "UPDATE entities SET balance = balance - $amount WHERE id = $fromEntityId";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to update from entity balance');
        }
        
        // Update to entity balance
        $sql = "UPDATE entities SET balance = balance + $amount WHERE id = $toEntityId";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to update to entity balance');
        }
        
        // Commit transaction
        $conn->commit();
        
        logActivity('Transfer', "Transferred $amount from entity #$fromEntityId to entity #$toEntityId");
        
        return $transferId;
    } catch (Exception $e) {
        // Rollback transaction
        $conn->rollback();
        return $e->getMessage();
    }
}

// Function to get transfers
function getTransfers($entityId = null) {
    global $conn;
    
    $sql = "SELECT t.*, f.name as from_entity_name, to_e.name as to_entity_name 
            FROM transfers t
            JOIN entities f ON t.from_entity_id = f.id
            JOIN entities to_e ON t.to_entity_id = to_e.id
            WHERE 1=1 ";
    
    if ($entityId !== null) {
        $sql .= "AND (t.from_entity_id = " . (int)$entityId . " OR t.to_entity_id = " . (int)$entityId . ") ";
    }
    
    $sql .= "ORDER BY t.transfer_date DESC";
    
    $result = $conn->query($sql);
    
    $transfers = array();
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $transfers[] = $row;
        }
    }
    
    return $transfers;
}

// Function to start a new sale
function createSale($sellerEntityId, $buyerEntityId, $notes = '') {
    global $conn;
    
    $sellerEntityId = (int)$sellerEntityId;
    $buyerEntityId = (int)$buyerEntityId;
    $notes = sanitize($notes);
    
    // Validate entities are different
    if ($sellerEntityId === $buyerEntityId) {
        return 'Cannot create sale to the same entity';
    }
    
    $sql = "INSERT INTO sales (seller_entity_id, buyer_entity_id, notes) 
            VALUES ($sellerEntityId, $buyerEntityId, '$notes')";
    
    if ($conn->query($sql) === TRUE) {
        $saleId = $conn->insert_id;
        logActivity('Sale', "Created new sale #$saleId from entity #$sellerEntityId to entity #$buyerEntityId");
        return $saleId;
    }
    
    return false;
}

// Function to add item to sale
function addSaleItem($saleId, $productId, $quantity, $unitPrice) {
    global $conn;
    
    $saleId = (int)$saleId;
    $productId = (int)$productId;
    $quantity = (float)$quantity;
    $unitPrice = (float)$unitPrice;
    
    // Validate quantity and price
    if ($quantity <= 0) {
        return 'Quantity must be positive';
    }
    
    if ($unitPrice <= 0) {
        return 'Unit price must be positive';
    }
    
    // Get sale
    $sale = getSale($saleId);
    
    if ($sale === null) {
        return 'Sale not found';
    }
    
    // Check if sale is still draft
    if ($sale['status'] !== 'draft') {
        return 'Cannot add items to a ' . $sale['status'] . ' sale';
    }
    
    // Check if seller has enough inventory
    $inventoryItem = getInventoryItem($sale['seller_entity_id'], $productId);
    
    if ($inventoryItem === null || $inventoryItem['quantity'] < $quantity) {
        return 'Seller does not have enough inventory';
    }
    
    // Calculate total price
    $totalPrice = $quantity * $unitPrice;
    
    // Begin transaction
    $conn->begin_transaction();
    
    try {
        // Insert sale item
        $sql = "INSERT INTO sale_items (sale_id, product_id, quantity, unit_price, total_price) 
                VALUES ($saleId, $productId, $quantity, $unitPrice, $totalPrice)";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to add sale item');
        }
        
        // Update sale total
        $sql = "UPDATE sales SET total_amount = total_amount + $totalPrice WHERE id = $saleId";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to update sale total');
        }
        
        // Commit transaction
        $conn->commit();
        
        logActivity('Sale', "Added item #$productId to sale #$saleId, quantity: $quantity, price: $unitPrice");
        
        return true;
    } catch (Exception $e) {
        // Rollback transaction
        $conn->rollback();
        return $e->getMessage();
    }
}

// Function to remove item from sale
function removeSaleItem($saleItemId) {
    global $conn;
    
    $saleItemId = (int)$saleItemId;
    
    // Get sale item
    $sql = "SELECT si.*, s.status, s.id as sale_id 
            FROM sale_items si
            JOIN sales s ON si.sale_id = s.id
            WHERE si.id = $saleItemId";
    
    $result = $conn->query($sql);
    
    if ($result->num_rows === 0) {
        return 'Sale item not found';
    }
    
    $saleItem = $result->fetch_assoc();
    
    // Check if sale is still draft
    if ($saleItem['status'] !== 'draft') {
        return 'Cannot remove items from a ' . $saleItem['status'] . ' sale';
    }
    
    // Begin transaction
    $conn->begin_transaction();
    
    try {
        // Update sale total
        $sql = "UPDATE sales SET total_amount = total_amount - " . $saleItem['total_price'] . " WHERE id = " . $saleItem['sale_id'];
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to update sale total');
        }
        
        // Delete sale item
        $sql = "DELETE FROM sale_items WHERE id = $saleItemId";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to remove sale item');
        }
        
        // Commit transaction
        $conn->commit();
        
        logActivity('Sale', "Removed item #$saleItemId from sale #" . $saleItem['sale_id']);
        
        return true;
    } catch (Exception $e) {
        // Rollback transaction
        $conn->rollback();
        return $e->getMessage();
    }
}

// Function to complete a sale
function completeSale($saleId) {
    global $conn;
    
    $saleId = (int)$saleId;
    
    // Get sale with items
    $sale = getSale($saleId);
    
    if ($sale === null) {
        return 'Sale not found';
    }
    
    // Check if sale is still draft
    if ($sale['status'] !== 'draft') {
        return 'Cannot complete a ' . $sale['status'] . ' sale';
    }
    
    // Check if sale has items
    if (count($sale['items']) === 0) {
        return 'Cannot complete sale with no items';
    }
    
    // Check if buyer has enough balance
    $buyer = getEntity($sale['buyer_entity_id']);
    
    if ($buyer['balance'] < $sale['total_amount']) {
        return 'Buyer does not have enough balance';
    }
    
    // Begin transaction
    $conn->begin_transaction();
    
    try {
        // Update sale status
        $sql = "UPDATE sales SET status = 'completed' WHERE id = $saleId";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to update sale status');
        }
        
        // Transfer money from buyer to seller
        $sql = "UPDATE entities SET balance = balance - " . $sale['total_amount'] . " WHERE id = " . $sale['buyer_entity_id'];
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to update buyer balance');
        }
        
        $sql = "UPDATE entities SET balance = balance + " . $sale['total_amount'] . " WHERE id = " . $sale['seller_entity_id'];
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to update seller balance');
        }
        
        // Update inventory and create stock movements for each item
        foreach ($sale['items'] as $item) {
            // Reduce seller inventory
            $sql = "UPDATE inventory SET quantity = quantity - " . $item['quantity'] . " 
                    WHERE entity_id = " . $sale['seller_entity_id'] . " AND product_id = " . $item['product_id'];
            
            if ($conn->query($sql) !== TRUE) {
                throw new Exception('Failed to update seller inventory');
            }
            
            // Record stock movement for seller (outgoing)
            $sql = "INSERT INTO stock_movements (entity_id, product_id, quantity, movement_type, reference_id, unit_price, notes) 
                    VALUES (" . $sale['seller_entity_id'] . ", " . $item['product_id'] . ", -" . $item['quantity'] . ", 
                            'sale', $saleId, " . $item['unit_price'] . ", 'Sale #$saleId')";
            
            if ($conn->query($sql) !== TRUE) {
                throw new Exception('Failed to record seller stock movement');
            }
            
            // Check if buyer already has this product in inventory
            $buyerInventory = getInventoryItem($sale['buyer_entity_id'], $item['product_id']);
            
            if ($buyerInventory === null) {
                // Create new inventory record for buyer
                $sql = "INSERT INTO inventory (entity_id, product_id, quantity, purchase_price, selling_price) 
                        VALUES (" . $sale['buyer_entity_id'] . ", " . $item['product_id'] . ", " . $item['quantity'] . ", 
                                " . $item['unit_price'] . ", " . ($item['unit_price'] * 1.2) . ")";
            } else {
                // Update buyer inventory - add quantity and recalculate average purchase price
                $newQuantity = $buyerInventory['quantity'] + $item['quantity'];
                $totalValue = ($buyerInventory['quantity'] * $buyerInventory['purchase_price']) + 
                              ($item['quantity'] * $item['unit_price']);
                $newPurchasePrice = $totalValue / $newQuantity;
                
                $sql = "UPDATE inventory SET quantity = $newQuantity, purchase_price = $newPurchasePrice 
                        WHERE entity_id = " . $sale['buyer_entity_id'] . " AND product_id = " . $item['product_id'];
            }
            
            if ($conn->query($sql) !== TRUE) {
                throw new Exception('Failed to update buyer inventory');
            }
            
            // Record stock movement for buyer (incoming)
            $sql = "INSERT INTO stock_movements (entity_id, product_id, quantity, movement_type, reference_id, unit_price, notes) 
                    VALUES (" . $sale['buyer_entity_id'] . ", " . $item['product_id'] . ", " . $item['quantity'] . ", 
                            'purchase', $saleId, " . $item['unit_price'] . ", 'Purchase from sale #$saleId')";
            
            if ($conn->query($sql) !== TRUE) {
                throw new Exception('Failed to record buyer stock movement');
            }
        }
        
        // Commit transaction
        $conn->commit();
        
        logActivity('Sale', "Completed sale #$saleId for " . $sale['total_amount']);
        
        return true;
    } catch (Exception $e) {
        // Rollback transaction
        $conn->rollback();
        return $e->getMessage();
    }
}

// Function to cancel a sale
function cancelSale($saleId) {
    global $conn;
    
    $saleId = (int)$saleId;
    
    // Get sale
    $sale = getSale($saleId);
    
    if ($sale === null) {
        return 'Sale not found';
    }
    
    // Check if sale is still draft
    if ($sale['status'] !== 'draft') {
        return 'Cannot cancel a ' . $sale['status'] . ' sale';
    }
    
    $sql = "UPDATE sales SET status = 'cancelled' WHERE id = $saleId";
    
    if ($conn->query($sql) === TRUE) {
        logActivity('Sale', "Cancelled sale #$saleId");
        return true;
    }
    
    return false;
}

// Function to get all sales
function getSales($entityId = null, $status = null) {
    global $conn;
    
    $sql = "SELECT s.*, 
            seller.name as seller_name, 
            buyer.name as buyer_name
            FROM sales s
            JOIN entities seller ON s.seller_entity_id = seller.id
            JOIN entities buyer ON s.buyer_entity_id = buyer.id
            WHERE 1=1 ";
    
    if ($entityId !== null) {
        $sql .= "AND (s.seller_entity_id = " . (int)$entityId . " OR s.buyer_entity_id = " . (int)$entityId . ") ";
    }
    
    if ($status !== null) {
        $sql .= "AND s.status = '" . sanitize($status) . "' ";
    }
    
    $sql .= "ORDER BY s.sale_date DESC";
    
    $result = $conn->query($sql);
    
    $sales = array();
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $sales[] = $row;
        }
    }
    
    return $sales;
}

// Function to get a single sale with items
function getSale($id) {
    global $conn;
    
    $sql = "SELECT s.*, 
            seller.name as seller_name, 
            buyer.name as buyer_name
            FROM sales s
            JOIN entities seller ON s.seller_entity_id = seller.id
            JOIN entities buyer ON s.buyer_entity_id = buyer.id
            WHERE s.id = " . (int)$id;
    
    $result = $conn->query($sql);
    
    if ($result->num_rows === 0) {
        return null;
    }
    
    $sale = $result->fetch_assoc();
    
    // Get sale items
    $sql = "SELECT si.*, p.name as product_name, p.unit
            FROM sale_items si
            JOIN products p ON si.product_id = p.id
            WHERE si.sale_id = " . (int)$id;
    
    $result = $conn->query($sql);
    
    $sale['items'] = array();
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $sale['items'][] = $row;
        }
    }
    
    return $sale;
}