<?php
// Include configuration file
require_once 'config.php';

// Check if tables exist, if not create them
checkTables();

// Handle authentication
if (!isAuthenticated() && (!isset($_GET['action']) || $_GET['action'] != 'login')) {
    $_GET['action'] = 'login';
}

// Get the requested action
$action = isset($_GET['action']) ? $_GET['action'] : 'dashboard';

// Handle logout
if ($action == 'logout') {
    session_unset();
    session_destroy();
    header('Location: index.php');
    exit;
}

// Process form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if ($action == 'login') {
        $pin = isset($_POST['pin']) ? $_POST['pin'] : '';
        
        if (authenticate($pin)) {
            header('Location: index.php');
            exit;
        } else {
            $error = 'Invalid PIN';
        }
    } elseif ($action == 'change_pin') {
        $oldPin = isset($_POST['old_pin']) ? $_POST['old_pin'] : '';
        $newPin = isset($_POST['new_pin']) ? $_POST['new_pin'] : '';
        $confirmPin = isset($_POST['confirm_pin']) ? $_POST['confirm_pin'] : '';
        
        if ($newPin != $confirmPin) {
            $error = 'New PINs do not match';
        } elseif (strlen($newPin) < 4) {
            $error = 'PIN must be at least 4 characters';
        } elseif (changePin($oldPin, $newPin)) {
            $success = 'PIN changed successfully';
        } else {
            $error = 'Current PIN is incorrect';
        }
    } elseif ($action == 'add_entity') {
        $name = isset($_POST['name']) ? $_POST['name'] : '';
        $description = isset($_POST['description']) ? $_POST['description'] : '';
        
        if (empty($name)) {
            $error = 'Entity name is required';
        } else {
            $result = addEntity($name, $description);
            
            if (is_numeric($result)) {
                header('Location: index.php?action=entities');
                exit;
            } else {
                $error = $result;
            }
        }
    } elseif ($action == 'edit_entity') {
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        $name = isset($_POST['name']) ? $_POST['name'] : '';
        $description = isset($_POST['description']) ? $_POST['description'] : '';
        
        if (empty($name)) {
            $error = 'Entity name is required';
        } else {
            $result = updateEntity($id, $name, $description);
            
            if ($result === true) {
                header('Location: index.php?action=entities');
                exit;
            } else {
                $error = $result;
            }
        }
    } elseif ($action == 'delete_entity') {
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        
        $result = deleteEntity($id);
        
        if ($result === true) {
            header('Location: index.php?action=entities');
            exit;
        } else {
            $error = $result;
        }
    } elseif ($action == 'add_transaction') {
        $fromEntityId = isset($_POST['from_entity_id']) ? (int)$_POST['from_entity_id'] : 0;
        $toEntityId = isset($_POST['to_entity_id']) ? (int)$_POST['to_entity_id'] : 0;
        $amount = isset($_POST['amount']) ? (float)$_POST['amount'] : 0;
        $description = isset($_POST['description']) ? $_POST['description'] : '';
        $dueDate = isset($_POST['due_date']) ? $_POST['due_date'] : null;
        
        if (empty($dueDate)) {
            $dueDate = null;
        }
        
        $result = addTransaction($fromEntityId, $toEntityId, $amount, $description, $dueDate);
        
        if (is_numeric($result)) {
            header('Location: index.php?action=transactions');
            exit;
        } else {
            $error = $result;
        }
    } elseif ($action == 'edit_transaction') {
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        $fromEntityId = isset($_POST['from_entity_id']) ? (int)$_POST['from_entity_id'] : 0;
        $toEntityId = isset($_POST['to_entity_id']) ? (int)$_POST['to_entity_id'] : 0;
        $amount = isset($_POST['amount']) ? (float)$_POST['amount'] : 0;
        $description = isset($_POST['description']) ? $_POST['description'] : '';
        $dueDate = isset($_POST['due_date']) ? $_POST['due_date'] : null;
        
        if (empty($dueDate)) {
            $dueDate = null;
        }
        
        $result = updateTransaction($id, $fromEntityId, $toEntityId, $amount, $description, $dueDate);
        
        if ($result === true) {
            header('Location: index.php?action=transactions');
            exit;
        } else {
            $error = $result;
        }
    } elseif ($action == 'cancel_transaction') {
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        
        $result = cancelTransaction($id);
        
        if ($result === true) {
            header('Location: index.php?action=transactions');
            exit;
        } else {
            $error = $result;
        }
    } elseif ($action == 'make_payment') {
        $transactionId = isset($_POST['transaction_id']) ? (int)$_POST['transaction_id'] : 0;
        $amount = isset($_POST['amount']) ? (float)$_POST['amount'] : 0;
        $notes = isset($_POST['notes']) ? $_POST['notes'] : '';
        
        $result = makePayment($transactionId, $amount, $notes);
        
        if (is_numeric($result)) {
            header('Location: index.php?action=view_transaction&id=' . $transactionId);
            exit;
        } else {
            $error = $result;
            $viewTransaction = getTransaction($transactionId);
        }
    } elseif ($action == 'add_deposit') {
        $entityId = isset($_POST['entity_id']) ? (int)$_POST['entity_id'] : 0;
        $amount = isset($_POST['amount']) ? (float)$_POST['amount'] : 0;
        $description = isset($_POST['description']) ? $_POST['description'] : '';
        
        $result = addDeposit($entityId, $amount, $description);
        
        if (is_numeric($result)) {
            header('Location: index.php?action=cash_flow');
            exit;
        } else {
            $error = $result;
        }
    } elseif ($action == 'add_withdrawal') {
        $entityId = isset($_POST['entity_id']) ? (int)$_POST['entity_id'] : 0;
        $amount = isset($_POST['amount']) ? (float)$_POST['amount'] : 0;
        $description = isset($_POST['description']) ? $_POST['description'] : '';
        
        $result = addWithdrawal($entityId, $amount, $description);
        
        if (is_numeric($result)) {
            header('Location: index.php?action=cash_flow');
            exit;
        } else {
            $error = $result;
        }
    }
}

// Helper functions

// Function to get all transactions with entity details
function getTransactions($status = null, $entityId = null) {
    global $conn;
    
    $sql = "SELECT t.*, 
            f.name as from_entity_name, 
            to_e.name as to_entity_name,
            COALESCE(SUM(p.amount), 0) as paid_amount
            FROM transactions t
            JOIN entities f ON t.from_entity_id = f.id
            JOIN entities to_e ON t.to_entity_id = to_e.id
            LEFT JOIN payments p ON t.id = p.transaction_id
            WHERE 1=1 ";
    
    if ($status !== null) {
        $sql .= "AND t.status = '" . sanitize($status) . "' ";
    }
    
    if ($entityId !== null) {
        $sql .= "AND (t.from_entity_id = " . (int)$entityId . " OR t.to_entity_id = " . (int)$entityId . ") ";
    }
    
    $sql .= "GROUP BY t.id ORDER BY t.created_at DESC";
    
    $result = $conn->query($sql);
    
    $transactions = array();
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            // Calculate remaining amount
            $row['remaining_amount'] = $row['amount'] - $row['paid_amount'];
            $transactions[] = $row;
        }
    }
    
    return $transactions;
}

// Function to get a single transaction
function getTransaction($id) {
    global $conn;
    
    $sql = "SELECT t.*, 
            f.name as from_entity_name, 
            to_e.name as to_entity_name,
            COALESCE(SUM(p.amount), 0) as paid_amount
            FROM transactions t
            JOIN entities f ON t.from_entity_id = f.id
            JOIN entities to_e ON t.to_entity_id = to_e.id
            LEFT JOIN payments p ON t.id = p.transaction_id
            WHERE t.id = " . (int)$id . "
            GROUP BY t.id";
    
    $result = $conn->query($sql);
    
    if ($result->num_rows > 0) {
        $transaction = $result->fetch_assoc();
        // Calculate remaining amount
        $transaction['remaining_amount'] = $transaction['amount'] - $transaction['paid_amount'];
        return $transaction;
    }
    
    return null;
}

// Function to add a new transaction
function addTransaction($fromEntityId, $toEntityId, $amount, $description, $dueDate) {
    global $conn;
    
    $fromEntityId = (int)$fromEntityId;
    $toEntityId = (int)$toEntityId;
    $amount = (float)$amount;
    $description = sanitize($description);
    $dueDate = sanitize($dueDate);
    
    // Validate entities are different
    if ($fromEntityId === $toEntityId) {
        return 'Cannot create transaction between the same entity';
    }
    
    // Validate amount is positive
    if ($amount <= 0) {
        return 'Transaction amount must be positive';
    }
    
    // Get the lending entity to check if they have enough balance
    $lendingEntity = getEntity($toEntityId);
    
    // Check if lending entity has enough balance
    if ($lendingEntity['balance'] < $amount) {
        return 'Lending entity does not have enough balance to create this loan';
    }
    
    // Begin transaction
    $conn->begin_transaction();
    
    try {
        // Insert transaction
        $sql = "INSERT INTO transactions (from_entity_id, to_entity_id, amount, description, due_date) 
                VALUES ($fromEntityId, $toEntityId, $amount, '$description', '$dueDate')";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to create transaction');
        }
        
        $transactionId = $conn->insert_id;
        
        // Update lending entity balance (reduce)
        $sql = "UPDATE entities SET balance = balance - $amount WHERE id = $toEntityId";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to update lending entity balance');
        }
        
        // Update borrowing entity balance (increase)
        $sql = "UPDATE entities SET balance = balance + $amount WHERE id = $fromEntityId";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to update borrowing entity balance');
        }
        
        // Commit transaction
        $conn->commit();
        
        logActivity('Transaction', "Created transaction #$transactionId: $amount from entity #$fromEntityId to entity #$toEntityId");
        
        return $transactionId;
    } catch (Exception $e) {
        // Rollback transaction
        $conn->rollback();
        return $e->getMessage();
    }
}

// Function to update a transaction
function updateTransaction($id, $fromEntityId, $toEntityId, $amount, $description, $dueDate) {
    global $conn;
    
    $id = (int)$id;
    $fromEntityId = (int)$fromEntityId;
    $toEntityId = (int)$toEntityId;
    $amount = (float)$amount;
    $description = sanitize($description);
    $dueDate = sanitize($dueDate);
    
    // Get existing transaction
    $existing = getTransaction($id);
    
    if ($existing === null) {
        return 'Transaction not found';
    }
    
    // Check if transaction is already paid or cancelled
    if ($existing['status'] !== 'active') {
        return 'Cannot update a ' . $existing['status'] . ' transaction';
    }
    
    // Validate entities are different
    if ($fromEntityId === $toEntityId) {
        return 'Cannot create transaction between the same entity';
    }
    
    // Validate amount is positive
    if ($amount <= 0) {
        return 'Transaction amount must be positive';
    }
    
    // Validate amount is not less than paid amount
    if ($amount < $existing['paid_amount']) {
        return 'Transaction amount cannot be less than already paid amount';
    }
    
    // Calculate amount difference
    $amountDifference = $amount - $existing['amount'];
    
    // If entities have changed or amount changed, we need to adjust balances
    $entitiesChanged = ($fromEntityId != $existing['from_entity_id']) || ($toEntityId != $existing['to_entity_id']);
    
    // If entities didn't change but amount did, check if lending entity has enough balance
    if (!$entitiesChanged && $amountDifference > 0) {
        $lendingEntity = getEntity($toEntityId);
        
        if ($lendingEntity['balance'] < $amountDifference) {
            return 'Lending entity does not have enough balance for the amount increase';
        }
    }
    
    // Begin transaction
    $conn->begin_transaction();
    
    try {
        // If entities didn't change but amount did
        if (!$entitiesChanged && $amountDifference != 0) {
            // Update lending entity balance
            $sql = "UPDATE entities SET balance = balance - $amountDifference WHERE id = $toEntityId";
            
            if ($conn->query($sql) !== TRUE) {
                throw new Exception('Failed to update lending entity balance');
            }
            
            // Update borrowing entity balance
            $sql = "UPDATE entities SET balance = balance + $amountDifference WHERE id = $fromEntityId";
            
            if ($conn->query($sql) !== TRUE) {
                throw new Exception('Failed to update borrowing entity balance');
            }
        }
        // If entities changed, need to reverse old transaction and create new balances
        else if ($entitiesChanged) {
            // Reverse old transaction - return money to old lender
            $sql = "UPDATE entities SET balance = balance + " . $existing['amount'] . " WHERE id = " . $existing['to_entity_id'];
            
            if ($conn->query($sql) !== TRUE) {
                throw new Exception('Failed to update old lending entity balance');
            }
            
            // Remove money from old borrower
            $sql = "UPDATE entities SET balance = balance - " . $existing['amount'] . " WHERE id = " . $existing['from_entity_id'];
            
            if ($conn->query($sql) !== TRUE) {
                throw new Exception('Failed to update old borrowing entity balance');
            }
            
            // Check if new lending entity has enough balance
            $newLendingEntity = getEntity($toEntityId);
            
            if ($newLendingEntity['balance'] < $amount) {
                // Roll back the previous adjustments and return error
                $conn->rollback();
                return 'New lending entity does not have enough balance';
            }
            
            // Update new lending entity balance
            $sql = "UPDATE entities SET balance = balance - $amount WHERE id = $toEntityId";
            
            if ($conn->query($sql) !== TRUE) {
                throw new Exception('Failed to update new lending entity balance');
            }
            
            // Update new borrowing entity balance
            $sql = "UPDATE entities SET balance = balance + $amount WHERE id = $fromEntityId";
            
            if ($conn->query($sql) !== TRUE) {
                throw new Exception('Failed to update new borrowing entity balance');
            }
        }
        
        // Update transaction record
        $sql = "UPDATE transactions SET 
                from_entity_id = $fromEntityId,
                to_entity_id = $toEntityId,
                amount = $amount,
                description = '$description',
                due_date = '$dueDate'
                WHERE id = $id";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to update transaction');
        }
        
        // Commit transaction
        $conn->commit();
        
        logActivity('Transaction', "Updated transaction #$id");
        
        return true;
    } catch (Exception $e) {
        // Rollback transaction
        $conn->rollback();
        return $e->getMessage();
    }
}

// Function to cancel a transaction
function cancelTransaction($id) {
    global $conn;
    
    $id = (int)$id;
    
    // Get existing transaction
    $existing = getTransaction($id);
    
    if ($existing === null) {
        return 'Transaction not found';
    }
    
    // Check if transaction is already paid or cancelled
    if ($existing['status'] !== 'active') {
        return 'Cannot cancel a ' . $existing['status'] . ' transaction';
    }
    
    // Begin transaction
    $conn->begin_transaction();
    
    try {
        // Update transaction status
        $sql = "UPDATE transactions SET status = 'cancelled' WHERE id = $id";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to cancel transaction');
        }
        
        // Restore lending entity balance (add back the remaining amount)
        $remainingAmount = $existing['amount'] - $existing['paid_amount'];
        
        $sql = "UPDATE entities SET balance = balance + $remainingAmount WHERE id = " . $existing['to_entity_id'];
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to update lending entity balance');
        }
        
        // Reduce borrowing entity balance
        $sql = "UPDATE entities SET balance = balance - $remainingAmount WHERE id = " . $existing['from_entity_id'];
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to update borrowing entity balance');
        }
        
        // Commit transaction
        $conn->commit();
        
        logActivity('Transaction', "Cancelled transaction #$id");
        
        return true;
    } catch (Exception $e) {
        // Rollback transaction
        $conn->rollback();
        return $e->getMessage();
    }
}

// Function to make payment on a transaction
function makePayment($transactionId, $amount, $notes) {
    global $conn;
    
    $transactionId = (int)$transactionId;
    $amount = (float)$amount;
    $notes = sanitize($notes);
    
    // Validate amount is positive
    if ($amount <= 0) {
        return 'Payment amount must be positive';
    }
    
    // Get transaction
    $transaction = getTransaction($transactionId);
    
    if ($transaction === null) {
        return 'Transaction not found';
    }
    
    // Check if transaction is active
    if ($transaction['status'] !== 'active') {
        return 'Cannot make payment on a ' . $transaction['status'] . ' transaction';
    }
    
    // Check if payment amount is not greater than remaining amount
    if ($amount > $transaction['remaining_amount']) {
        return 'Payment amount cannot be greater than remaining amount';
    }
    
    // Get entities
    $fromEntity = getEntity($transaction['from_entity_id']);
    $toEntity = getEntity($transaction['to_entity_id']);
    
    // Check if from entity has enough balance
    if ($fromEntity['balance'] < $amount) {
        return 'Entity does not have enough balance to make this payment';
    }
    
    // Begin transaction
    $conn->begin_transaction();
    
    try {
        // Insert payment
        $sql = "INSERT INTO payments (transaction_id, amount, notes) 
                VALUES ($transactionId, $amount, '$notes')";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to create payment');
        }
        
        $paymentId = $conn->insert_id;
        
        // Update entity balances
        $sql = "UPDATE entities SET balance = balance - $amount WHERE id = " . $fromEntity['id'];
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to update from entity balance');
        }
        
        $sql = "UPDATE entities SET balance = balance + $amount WHERE id = " . $toEntity['id'];
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to update to entity balance');
        }
        
        // Check if transaction is fully paid
        $newPaidAmount = $transaction['paid_amount'] + $amount;
        
        if ($newPaidAmount >= $transaction['amount']) {
            // Update transaction status to paid
            $sql = "UPDATE transactions SET status = 'paid' WHERE id = $transactionId";
            
            if ($conn->query($sql) !== TRUE) {
                throw new Exception('Failed to update transaction status');
            }
        }
        
        // Commit transaction
        $conn->commit();
        
        logActivity('Payment', "Made payment #$paymentId of $amount on transaction #$transactionId");
        
        return $paymentId;
    } catch (Exception $e) {
        // Rollback transaction
        $conn->rollback();
        return $e->getMessage();
    }
}

// Function to get payments for a transaction
function getPayments($transactionId) {
    global $conn;
    
    $sql = "SELECT * FROM payments WHERE transaction_id = " . (int)$transactionId . " ORDER BY payment_date DESC";
    $result = $conn->query($sql);
    
    $payments = array();
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $payments[] = $row;
        }
    }
    
    return $payments;
}

// Function to add cash deposit
function addDeposit($entityId, $amount, $description) {
    global $conn;
    
    $entityId = (int)$entityId;
    $amount = (float)$amount;
    $description = sanitize($description);
    
    // Validate amount is positive
    if ($amount <= 0) {
        return 'Deposit amount must be positive';
    }
    
    // Begin transaction
    $conn->begin_transaction();
    
    try {
        // Insert cash flow record
        $sql = "INSERT INTO cash_flow (entity_id, amount, type, description) 
                VALUES ($entityId, $amount, 'deposit', '$description')";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to create deposit record');
        }
        
        $flowId = $conn->insert_id;
        
        // Update entity balance
        $sql = "UPDATE entities SET balance = balance + $amount WHERE id = $entityId";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to update entity balance');
        }
        
        // Commit transaction
        $conn->commit();
        
        logActivity('Cash Flow', "Added deposit #$flowId of $amount to entity #$entityId");
        
        return $flowId;
    } catch (Exception $e) {
        // Rollback transaction
        $conn->rollback();
        return $e->getMessage();
    }
}

// Function to add cash withdrawal
function addWithdrawal($entityId, $amount, $description) {
    global $conn;
    
    $entityId = (int)$entityId;
    $amount = (float)$amount;
    $description = sanitize($description);
    
    // Validate amount is positive
    if ($amount <= 0) {
        return 'Withdrawal amount must be positive';
    }
    
    // Get entity
    $entity = getEntity($entityId);
    
    // Check if entity has enough balance
    if ($entity['balance'] < $amount) {
        return 'Entity does not have enough balance for this withdrawal';
    }
    
    // Begin transaction
    $conn->begin_transaction();
    
    try {
        // Insert cash flow record
        $sql = "INSERT INTO cash_flow (entity_id, amount, type, description) 
                VALUES ($entityId, $amount, 'withdrawal', '$description')";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to create withdrawal record');
        }
        
        $flowId = $conn->insert_id;
        
        // Update entity balance
        $sql = "UPDATE entities SET balance = balance - $amount WHERE id = $entityId";
        
        if ($conn->query($sql) !== TRUE) {
            throw new Exception('Failed to update entity balance');
        }
        
        // Commit transaction
        $conn->commit();
        
        logActivity('Cash Flow', "Added withdrawal #$flowId of $amount from entity #$entityId");
        
        return $flowId;
    } catch (Exception $e) {
        // Rollback transaction
        $conn->rollback();
        return $e->getMessage();
    }
}

// Function to get cash flow for an entity
function getCashFlow($entityId = null) {
    global $conn;
    
    $sql = "SELECT cf.*, e.name as entity_name 
            FROM cash_flow cf
            JOIN entities e ON cf.entity_id = e.id
            WHERE 1=1 ";
    
    if ($entityId !== null) {
        $sql .= "AND cf.entity_id = " . (int)$entityId . " ";
    }
    
    $sql .= "ORDER BY cf.flow_date DESC";
    
    $result = $conn->query($sql);
    
    $cashFlows = array();
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $cashFlows[] = $row;
        }
    }
    
    return $cashFlows;
}

// Function to get dashboard summary
function getDashboardSummary() {
    global $conn;
    
    $summary = array();
    
    // Get total number of entities
    $sql = "SELECT COUNT(*) as count FROM entities WHERE is_active = 1";
    $result = $conn->query($sql);
    $row = $result->fetch_assoc();
    $summary['total_entities'] = $row['count'];
    
    // Get total active transactions
    $sql = "SELECT COUNT(*) as count FROM transactions WHERE status = 'active'";
    $result = $conn->query($sql);
    $row = $result->fetch_assoc();
    $summary['active_transactions'] = $row['count'];
    
    // Get total amount of active debt
    $sql = "SELECT COALESCE(SUM(t.amount - COALESCE(SUM(p.amount), 0)), 0) as total_debt
            FROM transactions t
            LEFT JOIN payments p ON t.id = p.transaction_id
            WHERE t.status = 'active'
            GROUP BY t.id";
    
    $result = $conn->query($sql);
    
    $totalDebt = 0;
    
    if ($result && $result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $totalDebt += $row['total_debt'];
        }
    }
    
    $summary['total_debt'] = $totalDebt;
    
    // Get total cash across all entities
    $sql = "SELECT COALESCE(SUM(balance), 0) as total_cash FROM entities WHERE is_active = 1";
    $result = $conn->query($sql);
    $row = $result->fetch_assoc();
    $summary['total_cash'] = $row['total_cash'];
    
    return $summary;
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png"  href="https://app.dezki.biz.id/linkpic/assets/uploads/67dd601f43f04.png">
    <title><?php echo $app_name; ?></title>
    <!-- Add Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <!-- Material Icons -->
    <link href="https://fonts.googleapis.com/icon?family=Material+Icons" rel="stylesheet">
    <style>
        :root {
            --theme-color: <?php echo $theme_color; ?>;
            --secondary-color: <?php echo $secondary_color; ?>;
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: 'Roboto', sans-serif;
            line-height: 1.6;
            background-color: #f5f5f5;
            color: #333;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 15px;
        }
        
        .login-container {
            max-width: 400px;
            margin: 100px auto;
            padding: 20px;
            background: white;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        
        header {
            background-color: var(--theme-color);
            color: white;
            padding: 10px 0;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        header .container {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .logo h1 {
            font-size: 24px;
            font-weight: 500;
        }
        
        nav ul {
            list-style: none;
            display: flex;
        }
        
        nav ul li {
            margin-left: 20px;
        }
        
        nav ul li a {
            color: white;
            text-decoration: none;
            font-weight: 500;
            display: flex;
            align-items: center;
        }
        
        nav ul li a i {
            margin-right: 5px;
        }
        
        .content {
            padding: 20px 0;
        }
        
        .card {
            background: white;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
            margin-bottom: 20px;
            overflow: hidden;
        }
        
        .card-header {
            background-color: var(--theme-color);
            color: white;
            padding: 15px 20px;
            font-weight: 500;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .card-body {
            padding: 20px;
        }
        
        .alert {
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 5px;
        }
        
        .alert-danger {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .alert-success {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        table th, table td {
            padding: 10px;
            text-align: left;
            border-bottom: 1px solid #ddd;
        }
        
        table th {
            background-color: #f2f2f2;
        }
        
        form .form-group {
            margin-bottom: 15px;
        }
        
        form label {
            display: block;
            margin-bottom: 5px;
            font-weight: 500;
        }
        
        form input[type="text"], 
        form input[type="number"], 
        form input[type="password"], 
        form input[type="date"], 
        form textarea, 
        form select {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 16px;
        }
        
        form textarea {
            resize: vertical;
        }
        
        .btn {
            display: inline-block;
            background-color: var(--theme-color);
            color: white;
            padding: 10px 15px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            text-decoration: none;
            font-size: 16px;
        }
        
        .btn-sm {
            padding: 5px 10px;
            font-size: 14px;
        }
        
        .btn-secondary {
            background-color: #6c757d;
        }
        
        .btn-success {
            background-color: #28a745;
        }
        
        .btn-danger {
            background-color: #dc3545;
        }
        
        .dashboard-cards {
            display: flex;
            flex-wrap: wrap;
            margin: 0 -10px;
        }
        
        .dashboard-card {
            flex: 1 0 200px;
            margin: 10px;
            background: white;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
            padding: 20px;
            text-align: center;
        }
        
        .dashboard-card h3 {
            font-size: 36px;
            margin-bottom: 10px;
            color: var(--theme-color);
        }
        
        .dashboard-card p {
            color: #666;
            font-size: 14px;
        }
        
        @media (max-width: 768px) {
            header .container {
                flex-direction: column;
                align-items: center;
            }
            
            nav ul {
                margin-top: 10px;
            }
            
            nav ul li {
                margin: 0 10px;
            }
            
            .dashboard-cards {
                flex-direction: column;
            }
        }
        
        .status-badge {
            display: inline-block;
            padding: 3px 8px;
            border-radius: 3px;
            font-size: 12px;
            font-weight: 500;
            text-transform: uppercase;
        }
        
        .status-active {
            background-color: #e7f7ee;
            color: #28a745;
        }
        
        .status-paid {
            background-color: #e7f5fc;
            color: #007bff;
        }
        
        .status-cancelled {
            background-color: #f8f5f5;
            color: #dc3545;
        }
        
        .filter-bar {
            display: flex;
            margin-bottom: 20px;
            align-items: center;
        }
        
        .filter-bar label {
            margin-right: 10px;
            font-weight: 500;
        }
        
        .filter-bar select {
            margin-right: 20px;
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        
        .material-icons {
            vertical-align: middle;
            line-height: 1;
        }
        
        /* Mobile navigation */
        .mobile-menu-toggle {
            display: none;
            background: none;
            border: none;
            color: white;
            font-size: 24px;
            cursor: pointer;
        }
        
        @media (max-width: 768px) {
            .mobile-menu-toggle {
                display: block;
            }
            
            nav ul {
                display: none;
                flex-direction: column;
                width: 100%;
            }
            
            nav ul.active {
                display: flex;
            }
            
            nav ul li {
                margin: 10px 0;
            }
        }
    </style>
</head>
<body>

<?php if ($action == 'login'): ?>

<!-- Login Page -->
<div class="login-container">
    <h2 style="text-align: center; margin-bottom: 20px;"><?php echo $app_name; ?></h2>
    
    <?php if (isset($error)): ?>
        <div class="alert alert-danger"><?php echo $error; ?></div>
    <?php endif; ?>
    
    <form method="post" action="index.php?action=login">
        <div class="form-group">
            <label for="pin">PIN:</label>
            <input type="password" id="pin" name="pin" required>
        </div>
        
        <button type="submit" class="btn" style="width: 100%;">Login</button>
    </form>
</div>

<?php else: ?>

<!-- Main Application Layout -->
<header>
    <div class="container">
        <div class="logo">
            <h1><?php echo $app_name; ?></h1>
        </div>
        
        <button class="mobile-menu-toggle" id="menuToggle">
            <span class="material-icons">menu</span>
        </button>
        
        <nav>
            <ul id="mainNav">
                <li><a href="index.php"><i class="material-icons">dashboard</i> Dashboard</a></li>
                <li><a href="index.php?action=entities"><i class="material-icons">business</i> Entities</a></li>
                <li><a href="index.php?action=transactions"><i class="material-icons">swap_horiz</i> Transactions</a></li>
                <li><a href="index.php?action=cash_flow"><i class="material-icons">account_balance_wallet</i> Cash Flow</a></li>
                <li><a href="inventory-ui.php"><i class="material-icons">inventory_2</i> Inventory & Sales</a></li>
                <li><a href="index.php?action=settings"><i class="material-icons">settings</i> Settings</a></li>
                <li><a href="index.php?action=logout"><i class="material-icons">exit_to_app</i> Logout</a></li>
            </ul>
        </nav>
    </div>
</header>

<div class="container">
    <div class="content">
        <?php if (isset($error)): ?>
            <div class="alert alert-danger"><?php echo $error; ?></div>
        <?php endif; ?>
        
        <?php if (isset($success)): ?>
            <div class="alert alert-success"><?php echo $success; ?></div>
        <?php endif; ?>
        
        <?php if ($action == 'dashboard' || $action == ''): ?>
            <!-- Dashboard -->
            <div class="card">
                <div class="card-header">Dashboard</div>
                <div class="card-body">
                    <?php $summary = getDashboardSummary(); ?>
                    
                    <div class="dashboard-cards">
                        <div class="dashboard-card">
                            <h3><?php echo $summary['total_entities']; ?></h3>
                            <p>Total Entities</p>
                        </div>
                        
                        <div class="dashboard-card">
                            <h3><?php echo $summary['active_transactions']; ?></h3>
                            <p>Active Transactions</p>
                        </div>
                        
                        <div class="dashboard-card">
                            <h3><?php echo formatCurrency($summary['total_debt']); ?></h3>
                            <p>Total Active Debt</p>
                        </div>
                        
                        <div class="dashboard-card">
                            <h3><?php echo formatCurrency($summary['total_cash']); ?></h3>
                            <p>Total Cash</p>
                        </div>
                    </div>
                    
                    <div style="margin-top: 20px; text-align: center;">
                        <a href="inventory-ui.php" class="btn">Manage Inventory & Sales</a>
                    </div>
                </div>
            </div>
            
            <!-- Recent Transactions -->
            <div class="card">
                <div class="card-header">
                    Recent Active Transactions
                    <a href="index.php?action=transactions" class="btn btn-sm">View All</a>
                </div>
                <div class="card-body">
                    <?php $transactions = getTransactions('active', null); ?>
                    
                    <?php if (count($transactions) > 0): ?>
                        <table>
                            <thead>
                                <tr>
                                    <th>From</th>
                                    <th>To</th>
                                    <th>Amount</th>
                                    <th>Remaining</th>
                                    <th>Due Date</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach (array_slice($transactions, 0, 5) as $transaction): ?>
                                    <tr>
                                        <td><?php echo $transaction['from_entity_name']; ?></td>
                                        <td><?php echo $transaction['to_entity_name']; ?></td>
                                        <td><?php echo formatCurrency($transaction['amount']); ?></td>
                                        <td><?php echo formatCurrency($transaction['remaining_amount']); ?></td>
                                        <td><?php echo $transaction['due_date'] ? date('d M Y', strtotime($transaction['due_date'])) : '-'; ?></td>
                                        <td>
                                            <a href="index.php?action=view_transaction&id=<?php echo $transaction['id']; ?>" class="btn btn-sm">View</a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <p>No active transactions found.</p>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Recent Cash Flow -->
            <div class="card">
                <div class="card-header">
                    Recent Cash Flow
                    <a href="index.php?action=cash_flow" class="btn btn-sm">View All</a>
                </div>
                <div class="card-body">
                    <?php $cashFlows = getCashFlow(); ?>
                    
                    <?php if (count($cashFlows) > 0): ?>
                        <table>
                            <thead>
                                <tr>
                                    <th>Entity</th>
                                    <th>Type</th>
                                    <th>Amount</th>
                                    <th>Description</th>
                                    <th>Date</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach (array_slice($cashFlows, 0, 5) as $flow): ?>
                                    <tr>
                                        <td><?php echo $flow['entity_name']; ?></td>
                                        <td>
                                            <?php if ($flow['type'] == 'deposit'): ?>
                                                <span style="color: green;">Deposit</span>
                                            <?php else: ?>
                                                <span style="color: red;">Withdrawal</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo formatCurrency($flow['amount']); ?></td>
                                        <td><?php echo $flow['description']; ?></td>
                                        <td><?php echo date('d M Y H:i', strtotime($flow['flow_date'])); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <p>No cash flow records found.</p>
                    <?php endif; ?>
                </div>
            </div>
            
        <?php elseif ($action == 'entities'): ?>
            <!-- Entities List -->
            <div class="card">
                <div class="card-header">
                    Entities
                    <a href="index.php?action=add_entity" class="btn btn-sm">Add New Entity</a>
                </div>
                <div class="card-body">
                    <?php $entities = getEntities(); ?>
                    
                    <?php if (count($entities) > 0): ?>
                        <table>
                            <thead>
                                <tr>
                                    <th>Name</th>
                                    <th>Description</th>
                                    <th>Balance</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($entities as $entity): ?>
                                    <tr>
                                        <td><?php echo $entity['name']; ?></td>
                                        <td><?php echo $entity['description']; ?></td>
                                        <td><?php echo formatCurrency($entity['balance']); ?></td>
                                        <td>
                                            <a href="index.php?action=edit_entity&id=<?php echo $entity['id']; ?>" class="btn btn-sm">Edit</a>
                                            <?php if ($entity['name'] != 'Personal Account'): ?>
                                                <a href="index.php?action=delete_entity&id=<?php echo $entity['id']; ?>" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this entity?')">Delete</a>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <p>No entities found.</p>
                    <?php endif; ?>
                </div>
            </div>
            
        <?php elseif ($action == 'add_entity'): ?>
            <!-- Add Entity Form -->
            <div class="card">
                <div class="card-header">Add New Entity</div>
                <div class="card-body">
                    <form method="post" action="index.php?action=add_entity">
                        <div class="form-group">
                            <label for="name">Name:</label>
                            <input type="text" id="name" name="name" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="description">Description:</label>
                            <textarea id="description" name="description" rows="3"></textarea>
                        </div>
                        
                        <button type="submit" class="btn">Add Entity</button>
                        <a href="index.php?action=entities" class="btn btn-secondary">Cancel</a>
                    </form>
                </div>
            </div>
            
        <?php elseif ($action == 'edit_entity'): ?>
            <!-- Edit Entity Form -->
            <?php 
            $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            $entity = getEntity($id);
            
            if ($entity): 
            ?>
                <div class="card">
                    <div class="card-header">Edit Entity</div>
                    <div class="card-body">
                        <form method="post" action="index.php?action=edit_entity">
                            <input type="hidden" name="id" value="<?php echo $entity['id']; ?>">
                            
                            <div class="form-group">
                                <label for="name">Name:</label>
                                <input type="text" id="name" name="name" value="<?php echo $entity['name']; ?>" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="description">Description:</label>
                                <textarea id="description" name="description" rows="3"><?php echo $entity['description']; ?></textarea>
                            </div>
                            
                            <button type="submit" class="btn">Update Entity</button>
                            <a href="index.php?action=entities" class="btn btn-secondary">Cancel</a>
                        </form>
                    </div>
                </div>
            <?php else: ?>
                <div class="alert alert-danger">Entity not found.</div>
            <?php endif; ?>
            
        <?php elseif ($action == 'delete_entity'): ?>
            <!-- Delete Entity Form -->
            <?php 
            $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            $entity = getEntity($id);
            
            if ($entity): 
            ?>
                <div class="card">
                    <div class="card-header">Delete Entity</div>
                    <div class="card-body">
                        <p>Are you sure you want to delete the entity "<?php echo $entity['name']; ?>"?</p>
                        
                        <form method="post" action="index.php?action=delete_entity">
                            <input type="hidden" name="id" value="<?php echo $entity['id']; ?>">
                            
                            <button type="submit" class="btn btn-danger">Delete Entity</button>
                            <a href="index.php?action=entities" class="btn btn-secondary">Cancel</a>
                        </form>
                    </div>
                </div>
            <?php else: ?>
                <div class="alert alert-danger">Entity not found.</div>
            <?php endif; ?>
            
        <?php elseif ($action == 'transactions'): ?>
            <!-- Transactions List -->
            <div class="card">
                <div class="card-header">
                    Transactions
                    <a href="index.php?action=add_transaction" class="btn btn-sm">Add New Transaction</a>
                </div>
                <div class="card-body">
                    <?php 
                    $status = isset($_GET['status']) ? $_GET['status'] : null;
                    $entityId = isset($_GET['entity_id']) ? (int)$_GET['entity_id'] : null;
                    
                    $entities = getEntities();
                    $transactions = getTransactions($status, $entityId);
                    ?>
                    
                    <div class="filter-bar">
                        <label for="status">Status:</label>
                        <select id="status" onchange="window.location.href='index.php?action=transactions&status='+this.value+'&entity_id=<?php echo $entityId; ?>'">
                            <option value="">All</option>
                            <option value="active" <?php echo $status == 'active' ? 'selected' : ''; ?>>Active</option>
                            <option value="paid" <?php echo $status == 'paid' ? 'selected' : ''; ?>>Paid</option>
                            <option value="cancelled" <?php echo $status == 'cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                        </select>
                        
                        <label for="entity">Entity:</label>
                        <select id="entity" onchange="window.location.href='index.php?action=transactions&status=<?php echo $status; ?>&entity_id='+this.value">
                            <option value="">All</option>
                            <?php foreach ($entities as $entity): ?>
                                <option value="<?php echo $entity['id']; ?>" <?php echo $entityId == $entity['id'] ? 'selected' : ''; ?>><?php echo $entity['name']; ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <?php if (count($transactions) > 0): ?>
                        <table>
                            <thead>
                                <tr>
                                    <th>From</th>
                                    <th>To</th>
                                    <th>Amount</th>
                                    <th>Paid</th>
                                    <th>Remaining</th>
                                    <th>Due Date</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($transactions as $transaction): ?>
                                    <tr>
                                        <td><?php echo $transaction['from_entity_name']; ?></td>
                                        <td><?php echo $transaction['to_entity_name']; ?></td>
                                        <td><?php echo formatCurrency($transaction['amount']); ?></td>
                                        <td><?php echo formatCurrency($transaction['paid_amount']); ?></td>
                                        <td><?php echo formatCurrency($transaction['remaining_amount']); ?></td>
                                        <td><?php echo $transaction['due_date'] ? date('d M Y', strtotime($transaction['due_date'])) : '-'; ?></td>
                                        <td>
                                            <span class="status-badge status-<?php echo $transaction['status']; ?>">
                                                <?php echo $transaction['status']; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <a href="index.php?action=view_transaction&id=<?php echo $transaction['id']; ?>" class="btn btn-sm">View</a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <p>No transactions found.</p>
                    <?php endif; ?>
                </div>
            </div>
            
        <?php elseif ($action == 'add_transaction'): ?>
            <!-- Add Transaction Form -->
            <div class="card">
                <div class="card-header">Add New Transaction</div>
                <div class="card-body">
                    <?php $entities = getEntities(); ?>
                    
                    <form method="post" action="index.php?action=add_transaction">
                        <div class="form-group">
                            <label for="from_entity_id">From (Borrower):</label>
                            <select id="from_entity_id" name="from_entity_id" required>
                                <option value="">Select Entity</option>
                                <?php foreach ($entities as $entity): ?>
                                    <option value="<?php echo $entity['id']; ?>"><?php echo $entity['name']; ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="to_entity_id">To (Lender):</label>
                            <select id="to_entity_id" name="to_entity_id" required>
                                <option value="">Select Entity</option>
                                <?php foreach ($entities as $entity): ?>
                                    <option value="<?php echo $entity['id']; ?>"><?php echo $entity['name']; ?> (Balance: <?php echo formatCurrency($entity['balance']); ?>)</option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="amount">Amount:</label>
                            <input type="number" id="amount" name="amount" min="1" step="0.01" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="description">Description:</label>
                            <textarea id="description" name="description" rows="3"></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label for="due_date">Due Date (optional):</label>
                            <input type="date" id="due_date" name="due_date">
                        </div>
                        
                        <button type="submit" class="btn">Add Transaction</button>
                        <a href="index.php?action=transactions" class="btn btn-secondary">Cancel</a>
                    </form>
                </div>
            </div>
            
        <?php elseif ($action == 'view_transaction'): ?>
            <!-- View Transaction Details -->
            <?php 
            $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            $transaction = getTransaction($id);
            
            if ($transaction): 
            ?>
                <div class="card">
                    <div class="card-header">
                        Transaction Details
                        <a href="index.php?action=transactions" class="btn btn-sm">Back to Transactions</a>
                    </div>
                    <div class="card-body">
                        <table>
                            <tr>
                                <th style="width: 150px;">Transaction ID:</th>
                                <td><?php echo $transaction['id']; ?></td>
                            </tr>
                            <tr>
                                <th>From (Borrower):</th>
                                <td><?php echo $transaction['from_entity_name']; ?></td>
                            </tr>
                            <tr>
                                <th>To (Lender):</th>
                                <td><?php echo $transaction['to_entity_name']; ?></td>
                            </tr>
                            <tr>
                                <th>Amount:</th>
                                <td><?php echo formatCurrency($transaction['amount']); ?></td>
                            </tr>
                            <tr>
                                <th>Paid Amount:</th>
                                <td><?php echo formatCurrency($transaction['paid_amount']); ?></td>
                            </tr>
                            <tr>
                                <th>Remaining:</th>
                                <td><?php echo formatCurrency($transaction['remaining_amount']); ?></td>
                            </tr>
                            <tr>
                                <th>Due Date:</th>
                                <td><?php echo $transaction['due_date'] ? date('d M Y', strtotime($transaction['due_date'])) : '-'; ?></td>
                            </tr>
                            <tr>
                                <th>Description:</th>
                                <td><?php echo $transaction['description']; ?></td>
                            </tr>
                            <tr>
                                <th>Status:</th>
                                <td>
                                    <span class="status-badge status-<?php echo $transaction['status']; ?>">
                                        <?php echo $transaction['status']; ?>
                                    </span>
                                </td>
                            </tr>
                            <tr>
                                <th>Created:</th>
                                <td><?php echo date('d M Y H:i', strtotime($transaction['created_at'])); ?></td>
                            </tr>
                        </table>
                        
                        <?php if ($transaction['status'] == 'active'): ?>
                            <div style="margin-top: 20px;">
                                <a href="index.php?action=edit_transaction&id=<?php echo $transaction['id']; ?>" class="btn">Edit Transaction</a>
                                <a href="index.php?action=cancel_transaction&id=<?php echo $transaction['id']; ?>" class="btn btn-danger" onclick="return confirm('Are you sure you want to cancel this transaction?')">Cancel Transaction</a>
                            </div>
                            
                            <div style="margin-top: 30px;">
                                <h3 style="margin-bottom: 15px;">Make Payment</h3>
                                
                                <form method="post" action="index.php?action=make_payment">
                                    <input type="hidden" name="transaction_id" value="<?php echo $transaction['id']; ?>">
                                    
                                    <div class="form-group">
                                        <label for="amount">Payment Amount:</label>
                                        <input type="number" id="amount" name="amount" min="1" max="<?php echo $transaction['remaining_amount']; ?>" step="0.01" required>
                                        <small>Maximum: <?php echo formatCurrency($transaction['remaining_amount']); ?></small>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label for="notes">Notes:</label>
                                        <textarea id="notes" name="notes" rows="2"></textarea>
                                    </div>
                                    
                                    <button type="submit" class="btn btn-success">Make Payment</button>
                                </form>
                            </div>
                        <?php endif; ?>
                        
                        <!-- Payments History -->
                        <div style="margin-top: 30px;">
                            <h3 style="margin-bottom: 15px;">Payment History</h3>
                            
                            <?php 
                            $payments = getPayments($transaction['id']);
                            
                            if (count($payments) > 0): 
                            ?>
                                <table>
                                    <thead>
                                        <tr>
                                            <th>Payment ID</th>
                                            <th>Amount</th>
                                            <th>Date</th>
                                            <th>Notes</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($payments as $payment): ?>
                                            <tr>
                                                <td><?php echo $payment['id']; ?></td>
                                                <td><?php echo formatCurrency($payment['amount']); ?></td>
                                                <td><?php echo date('d M Y H:i', strtotime($payment['payment_date'])); ?></td>
                                                <td><?php echo $payment['notes']; ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            <?php else: ?>
                                <p>No payment records found.</p>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            <?php else: ?>
                <div class="alert alert-danger">Transaction not found.</div>
            <?php endif; ?>
            
        <?php elseif ($action == 'edit_transaction'): ?>
            <!-- Edit Transaction Form -->
            <?php 
            $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            $transaction = getTransaction($id);
            
            if ($transaction && $transaction['status'] == 'active'): 
            ?>
                <div class="card">
                    <div class="card-header">Edit Transaction</div>
                    <div class="card-body">
                        <?php $entities = getEntities(); ?>
                        
                        <form method="post" action="index.php?action=edit_transaction">
                            <input type="hidden" name="id" value="<?php echo $transaction['id']; ?>">
                            
                            <div class="form-group">
                                <label for="from_entity_id">From (Borrower):</label>
                                <select id="from_entity_id" name="from_entity_id" required>
                                    <?php foreach ($entities as $entity): ?>
                                        <option value="<?php echo $entity['id']; ?>" <?php echo $transaction['from_entity_id'] == $entity['id'] ? 'selected' : ''; ?>><?php echo $entity['name']; ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="to_entity_id">To (Lender):</label>
                                <select id="to_entity_id" name="to_entity_id" required>
                                    <?php foreach ($entities as $entity): ?>
                                        <option value="<?php echo $entity['id']; ?>" <?php echo $transaction['to_entity_id'] == $entity['id'] ? 'selected' : ''; ?>><?php echo $entity['name']; ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="amount">Amount:</label>
                                <input type="number" id="amount" name="amount" min="<?php echo $transaction['paid_amount']; ?>" step="0.01" value="<?php echo $transaction['amount']; ?>" required>
                                <small>Minimum: <?php echo formatCurrency($transaction['paid_amount']); ?> (already paid)</small>
                            </div>
                            
                            <div class="form-group">
                                <label for="description">Description:</label>
                                <textarea id="description" name="description" rows="3"><?php echo $transaction['description']; ?></textarea>
                            </div>
                            
                            <div class="form-group">
                                <label for="due_date">Due Date (optional):</label>
                                <input type="date" id="due_date" name="due_date" value="<?php echo $transaction['due_date']; ?>">
                            </div>
                            
                            <button type="submit" class="btn">Update Transaction</button>
                            <a href="index.php?action=view_transaction&id=<?php echo $transaction['id']; ?>" class="btn btn-secondary">Cancel</a>
                        </form>
                    </div>
                </div>
            <?php else: ?>
                <div class="alert alert-danger">Transaction not found or cannot be edited.</div>
            <?php endif; ?>
            
        <?php elseif ($action == 'cancel_transaction'): ?>
            <!-- Cancel Transaction Form -->
            <?php 
            $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            $transaction = getTransaction($id);
            
            if ($transaction && $transaction['status'] == 'active'): 
            ?>
                <div class="card">
                    <div class="card-header">Cancel Transaction</div>
                    <div class="card-body">
                        <p>Are you sure you want to cancel this transaction?</p>
                        
                        <table style="margin-bottom: 20px;">
                            <tr>
                                <th style="width: 150px;">From:</th>
                                <td><?php echo $transaction['from_entity_name']; ?></td>
                            </tr>
                            <tr>
                                <th>To:</th>
                                <td><?php echo $transaction['to_entity_name']; ?></td>
                            </tr>
                            <tr>
                                <th>Amount:</th>
                                <td><?php echo formatCurrency($transaction['amount']); ?></td>
                            </tr>
                            <tr>
                                <th>Remaining:</th>
                                <td><?php echo formatCurrency($transaction['remaining_amount']); ?></td>
                            </tr>
                        </table>
                        
                        <form method="post" action="index.php?action=cancel_transaction">
                            <input type="hidden" name="id" value="<?php echo $transaction['id']; ?>">
                            
                            <button type="submit" class="btn btn-danger">Cancel Transaction</button>
                            <a href="index.php?action=view_transaction&id=<?php echo $transaction['id']; ?>" class="btn btn-secondary">Back</a>
                        </form>
                    </div>
                </div>
            <?php else: ?>
                <div class="alert alert-danger">Transaction not found or cannot be cancelled.</div>
            <?php endif; ?>
            
        <?php elseif ($action == 'cash_flow'): ?>
            <!-- Cash Flow -->
            <div class="card">
                <div class="card-header">
                    Cash Flow
                    <div>
                        <a href="index.php?action=add_deposit" class="btn btn-sm btn-success">Add Deposit</a>
                        <a href="index.php?action=add_withdrawal" class="btn btn-sm btn-danger">Add Withdrawal</a>
                    </div>
                </div>
                <div class="card-body">
                    <?php 
                    $entityId = isset($_GET['entity_id']) ? (int)$_GET['entity_id'] : null;
                    $entities = getEntities();
                    $cashFlows = getCashFlow($entityId);
                    ?>
                    
                    <div class="filter-bar">
                        <label for="entity">Entity:</label>
                        <select id="entity" onchange="window.location.href='index.php?action=cash_flow&entity_id='+this.value">
                            <option value="">All Entities</option>
                            <?php foreach ($entities as $entity): ?>
                                <option value="<?php echo $entity['id']; ?>" <?php echo $entityId == $entity['id'] ? 'selected' : ''; ?>><?php echo $entity['name']; ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <?php if (count($cashFlows) > 0): ?>
                        <table>
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Entity</th>
                                    <th>Type</th>
                                    <th>Amount</th>
                                    <th>Description</th>
                                    <th>Date</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($cashFlows as $flow): ?>
                                    <tr>
                                        <td><?php echo $flow['id']; ?></td>
                                        <td><?php echo $flow['entity_name']; ?></td>
                                        <td>
                                            <?php if ($flow['type'] == 'deposit'): ?>
                                                <span style="color: green;">Deposit</span>
                                            <?php else: ?>
                                                <span style="color: red;">Withdrawal</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo formatCurrency($flow['amount']); ?></td>
                                        <td><?php echo $flow['description']; ?></td>
                                        <td><?php echo date('d M Y H:i', strtotime($flow['flow_date'])); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <p>No cash flow records found.</p>
                    <?php endif; ?>
                </div>
            </div>
            
        <?php elseif ($action == 'add_deposit'): ?>
            <!-- Add Deposit Form -->
            <div class="card">
                <div class="card-header">Add Deposit</div>
                <div class="card-body">
                    <?php $entities = getEntities(); ?>
                    
                    <form method="post" action="index.php?action=add_deposit">
                        <div class="form-group">
                            <label for="entity_id">Entity:</label>
                            <select id="entity_id" name="entity_id" required>
                                <option value="">Select Entity</option>
                                <?php foreach ($entities as $entity): ?>
                                    <option value="<?php echo $entity['id']; ?>"><?php echo $entity['name']; ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="amount">Amount:</label>
                            <input type="number" id="amount" name="amount" min="1" step="0.01" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="description">Description:</label>
                            <textarea id="description" name="description" rows="3"></textarea>
                        </div>
                        
                        <button type="submit" class="btn btn-success">Add Deposit</button>
                        <a href="index.php?action=cash_flow" class="btn btn-secondary">Cancel</a>
                    </form>
                </div>
            </div>
            
        <?php elseif ($action == 'add_withdrawal'): ?>
            <!-- Add Withdrawal Form -->
            <div class="card">
                <div class="card-header">Add Withdrawal</div>
                <div class="card-body">
                    <?php $entities = getEntities(); ?>
                    
                    <form method="post" action="index.php?action=add_withdrawal">
                        <div class="form-group">
                            <label for="entity_id">Entity:</label>
                            <select id="entity_id" name="entity_id" required>
                                <option value="">Select Entity</option>
                                <?php foreach ($entities as $entity): ?>
                                    <option value="<?php echo $entity['id']; ?>"><?php echo $entity['name']; ?> (Balance: <?php echo formatCurrency($entity['balance']); ?>)</option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="amount">Amount:</label>
                            <input type="number" id="amount" name="amount" min="1" step="0.01" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="description">Description:</label>
                            <textarea id="description" name="description" rows="3"></textarea>
                        </div>
                        
                        <button type="submit" class="btn btn-danger">Add Withdrawal</button>
                        <a href="index.php?action=cash_flow" class="btn btn-secondary">Cancel</a>
                    </form>
                </div>
            </div>
            
        <?php elseif ($action == 'settings'): ?>
            <!-- Settings -->
            <div class="card">
                <div class="card-header">Settings</div>
                <div class="card-body">
                    <h3 style="margin-bottom: 15px;">Change PIN</h3>
                    
                    <form method="post" action="index.php?action=change_pin">
                        <div class="form-group">
                            <label for="old_pin">Current PIN:</label>
                            <input type="password" id="old_pin" name="old_pin" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="new_pin">New PIN:</label>
                            <input type="password" id="new_pin" name="new_pin" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="confirm_pin">Confirm New PIN:</label>
                            <input type="password" id="confirm_pin" name="confirm_pin" required>
                        </div>
                        
                        <button type="submit" class="btn">Change PIN</button>
                    </form>
                </div>
            </div>
            
            <!-- Audit Log -->
            <div class="card">
                <div class="card-header">Recent Activity Log</div>
                <div class="card-body">
                    <?php $logs = getAuditLogs(20); ?>
                    
                    <?php if (count($logs) > 0): ?>
                        <table>
                            <thead>
                                <tr>
                                    <th>Time</th>
                                    <th>Action</th>
                                    <th>Details</th>
                                    <th>IP Address</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($logs as $log): ?>
                                    <tr>
                                        <td><?php echo date('d M Y H:i:s', strtotime($log['timestamp'])); ?></td>
                                        <td><?php echo $log['action']; ?></td>
                                        <td><?php echo $log['details']; ?></td>
                                        <td><?php echo $log['ip_address']; ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <p>No activity log records found.</p>
                    <?php endif; ?>
                </div>
            </div>
        
        <?php endif; ?>
    </div>
</div>

<script>
    // Mobile menu toggle
    document.getElementById('menuToggle').addEventListener('click', function() {
        document.getElementById('mainNav').classList.toggle('active');
    });
</script>

<?php endif; ?>

</body>
</html>